% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prepare_tidy_data.R
\name{prepare_tidy_data}
\alias{prepare_tidy_data}
\title{Prepare data in tidy format}
\usage{
prepare_tidy_data(data, sample_names, samples_in = "cols", ...)
}
\arguments{
\item{data}{a data.frame in "wide" format, with samples in either columns or rows. This data.frame should not include any data besides abundance values per sample, per taxonomic unit. Additional data (e.g. taxonomy details) should be added afterwards.}

\item{sample_names}{a vector with the name of all samples.}

\item{samples_in}{a vector specifying the location of the samples. It can either be "cols" (default) if samples are in columns, or "rows" if samples are in rows.}

\item{...}{additional arguments}
}
\value{
An abundance table in long format, compatible with dplyr pipes and \strong{ulrb} package functions.
}
\description{
Function to transforms common abundance table formats into a "long" format.
}
\details{
This function guarantees that the abundance table includes one column with sample ID's and
one column with abundance.

To use this function, the user should have a vector with the samples names as they appear in the abundance table.
Usually simple data wrangling with base R is enough to obtain this information from the abundance table itself.

\strong{Common species table formats}

There are two common formats for abundance tables:
\itemize{
\item samples as rows and taxa as columns;
\item taxa as rows and samples as columns.
}

However, both formats are not tidy/long, because they include several columns with the same variable. They
are in a "wide format" instead of a "long format".

This function re-organizes samples and taxa so that there is a single column with the samples ID's and
another with the abundance scores. Extra columns are allowed.
}
\examples{
library(dplyr)
#
sample_names <- c("ERR2044662", "ERR2044663", "ERR2044664",
                   "ERR2044665", "ERR2044666", "ERR2044667",
                   "ERR2044668", "ERR2044669", "ERR2044670")

# Example for samples in cols and with additional data available
prepare_tidy_data(nice, sample_names = sample_names, samples_in = "cols")

# Example for samples in rows
# Select columns with samples from nice
nice_rows <- nice \%>\% select(all_of(sample_names))

# Change columns to rows
nice_rows <- nice_rows \%>\% t() \%>\% as.data.frame()

# Turn colnames into phylogenetic units ID
colnames(nice_rows) <- paste0("OTU_", seq_along(colnames(nice_rows)))

prepare_tidy_data(nice_rows, sample_names = sample_names, samples_in = "rows")


# Extra examples with mock values
# Mock example 1 - wide table, samples in rows
mock_1 <- data.frame(Sample = paste0("S", 1:10),
                        Taxa1 = sample(10),
                        Taxa2 = sample(10),
                        Taxa3 = sample(10),
                        Taxa4 = sample(10),
                        Taxa5 = sample(10),
                        Taxa6 = sample(10))

prepare_tidy_data(mock_1[, -1], # remove Sample column
                  sample_names = mock_1$Sample,
                  samples_in = "rows")

# Mock example 2 - wide table, sample in columns
mock_2 <- data.frame(Sample = paste0("Taxa_", 1:6),
                        S1 = sample(6),
                        S2 = sample(6),
                        S3 = sample(6),
                        S4 = sample(6),
                        S5 = sample(6),
                        S6 = sample(6))

mock_2 \%>\%
 rename(TaxaID = Sample) \%>\% # Correct column name
 prepare_tidy_data(samples_in = "cols",
                   sample_names = colnames(mock_2)[-1])


}
\seealso{
\code{\link[=define_rb]{define_rb()}}
}
