% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/twomodeClusteringGA.R
\name{twomodeClusteringGA}
\alias{twomodeClusteringGA}
\title{Two-mode clustering using genetic algorithm (with optional validation)}
\usage{
twomodeClusteringGA(
  myMatrix,
  nColClusters,
  nRowClusters,
  seeds = 1:5,
  verbose = FALSE,
  maxiter = 2000,
  popSize = 300,
  pmutation = 0.05,
  pcrossover = 0.5,
  elitism = 100,
  interval = 100,
  parallel = FALSE,
  run = NULL,
  validate = FALSE,
  validateCenter = TRUE,
  validatePerBlock = TRUE,
  validateMonteCarlo = 0L,
  validateFixBlockSizes = TRUE,
  validateStoreNull = FALSE,
  validateSeed = NULL
)
}
\arguments{
\item{myMatrix}{Numeric matrix or data.frame to be clustered. Must be coercible to numeric.}

\item{nColClusters}{Integer. Number of column clusters to form.}

\item{nRowClusters}{Integer. Number of row clusters to form.}

\item{seeds}{Integer vector. Random seeds for multiple GA runs. Default is 1:5.}

\item{verbose}{Logical. If TRUE, prints progress information. Default is FALSE.}

\item{maxiter}{Integer. Maximum number of GA iterations. Default is 2000.}

\item{popSize}{Integer. Population size for the GA. Default is 300.}

\item{pmutation}{Numeric. Probability of mutation (0-1). Default is 0.05.}

\item{pcrossover}{Numeric. Probability of crossover (0-1). Default is 0.5.}

\item{elitism}{Integer. Number of best individuals to preserve. Default is 100. If NULL, uses 5\% of popSize.}

\item{interval}{Integer. Interval for progress monitoring when verbose=TRUE. Default is 100.}

\item{parallel}{Logical. Whether to use parallel processing. Default is FALSE.}

\item{run}{Integer. Number of consecutive generations without improvement before stopping.
If NULL, runs for full maxiter iterations.}

\item{validate}{Logical. If TRUE, run validation on the best partition and attach results
under \code{$validation}. Default FALSE.}

\item{validateCenter}{Logical. Passed to \code{validateTwomodePartition(center=...)}. Default TRUE.}

\item{validatePerBlock}{Logical. Passed to \code{validateTwomodePartition(perBlock=...)}. Default TRUE.}

\item{validateMonteCarlo}{Integer. Number of random partitions for MC p-value.
Passed to \code{validateTwomodePartition(monteCarlo=...)}. Default 0 (disabled).}

\item{validateFixBlockSizes}{Logical. Keep observed cluster sizes in MC. Default TRUE.}

\item{validateStoreNull}{Logical. Store full null vector from MC. Default FALSE.}

\item{validateSeed}{Optional integer seed for the validation step. Default NULL.}
}
\value{
A list of class \code{"twomodeClustering"} containing:
\describe{
\item{bestGa}{The best GA object from all runs}
\item{bestFitness}{Best fitness value achieved (negative SSE)}
\item{bestSeed}{Seed that produced the best result}
\item{rowClusters}{Integer vector of row cluster assignments}
\item{colClusters}{Integer vector of column cluster assignments}
\item{control}{List of control parameters used}
\item{validation}{List returned by \code{validateTwomodePartition()} if \code{validate=TRUE}; otherwise NULL}
}
}
\description{
Performs two-mode clustering on a numeric matrix using a genetic algorithm.
The algorithm simultaneously clusters rows and columns to minimize within-cluster
sum of squared errors (SSE). Optionally, a validation step is executed that tests
the statistical significance of the found partition using \code{validateTwomodePartition()}.
}
\details{
The function runs multiple GA instances with different random seeds and
returns the best solution. The fitness function minimizes the sum of squared errors
within clusters. Row and column clusters are optimized simultaneously.
}
\examples{
data("twomodeToy")
myMatrix_s <- scale(twomodeToy)

#Run the GA-based two-mode clustering
result <- twomodeClusteringGA(
  myMatrix = myMatrix_s,
  nRowClusters = 2,
  nColClusters = 3,
  seeds = 1,
  maxiter = 200,
  popSize = 30,
  elitism = 1,
  validate = TRUE,
  verbose = TRUE
)

#Inspect the result
print(result)
summary(result)
myTwomodeResult <- as.data.frame(result)
head(myTwomodeResult)

#Plot the clustered heatmap
plotTwomodeClustering(
  myMatrix = myMatrix_s,
  result   = result,
  title    = "Two-mode clustering Toy example",
  fixAspect = FALSE
)
}
\references{
Hageman, J. A., van den Berg, R. A., Westerhuis, J. A., van der Werf, M. J., & Smilde, A. K. (2008).
Genetic algorithm based two-mode clustering of metabolomics data. \emph{Metabolomics}, 4, 141–149. \doi{10.1007/s11306-008-0105-7}
}
\seealso{
\code{\link[GA]{ga}} for the underlying genetic algorithm implementation
}
