% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transform-helper-functions.R,
%   R/transform.scdf.R
\name{moving_median}
\alias{moving_median}
\alias{moving_mean}
\alias{local_regression}
\alias{set_na_at}
\alias{center_at}
\alias{first_of}
\alias{across_cases}
\alias{all_cases}
\alias{rowwise}
\alias{transform.scdf}
\title{Transform every single case of a single case data frame}
\usage{
moving_median(x, lag = 1)

moving_mean(x, lag = 1)

local_regression(x, mt = 1:length(x), f = 0.2)

set_na_at(x, first_of, positions = 0)

center_at(x, at = TRUE, shift = 0, part = 0)

first_of(x, positions = 0)

across_cases(...)

all_cases(...)

rowwise(...)

\method{transform}{scdf}(`_data`, ...)
}
\arguments{
\item{x}{A logical vector.}

\item{lag}{Number of values surrounding a value to calculate the average}

\item{mt}{A vector with measurement times.}

\item{f}{the proportion of surrounding data influencing each data point.}

\item{first_of}{A logical vector}

\item{positions}{A numeric vector with relative positions to the first
appearance of a TRUE value in x.}

\item{at}{A logical vector. E.g. \code{phase == "A"}. The first TRUE value of that
vector is the target position for centring. By default, this is the first
position.}

\item{shift}{A value indicating a shift in measurement times for centring.
E.g. \code{shift = 4} will centre four measurement-times after the position
defined by the \code{at} and \code{part} arguments.}

\item{part}{A numeric value between 0 and 1. \code{0} refers to the first \code{TRUE}
in the \code{at} vector, \code{1} to the last, and \code{0.5} to the midpoint of the
sequence of \code{TRUE} values. E.g. if you want to centre at the middle of
phase A, set \verb{at = phase == A, part = 0.5}. Note: decimals are rounded to
integers.}

\item{...}{Expressions.}

\item{_data}{An scdf.}
}
\value{
An scdf.
}
\description{
Takes an scdf and applies transformations to each individual case. This is
useful to calculate or modify new variables.
}
\details{
This function is a method of the generic \code{transform()} function.
Unlike the generic version, expressions are evaluated \strong{serially}:
the result of one expression is used as the basis for subsequent computations.

Several helper functions can be used inside expressions:
\itemize{
\item \strong{\code{n()}}: returns the number of measurements in a case.
\item \strong{\code{all_cases()}}: extracts the values of a variable across all cases.
Takes an expression as argument.
For example:
\itemize{
\item \code{mean(all_cases(values))} calculates the mean of \code{values} across all cases.
\item \code{mean(all_cases(values[phase == "A"]))} calculates the mean of all values where \code{phase == "A"}.
}
\item \strong{\code{rowwise()}}: applies a calculation separately to each row.
Example: \code{rowwise(sum(values, mt, na.rm = TRUE))}.
\item \strong{\code{across_cases()}}: creates new variables or replaces existing ones across all cases.
Example: \code{across_cases(values_ranked = rank(values, na.last = "keep"))}
}
}
\examples{
## Creates a single-case with frequency distributions. The proportion and
## percentage of the frequencies are calculated with transform:
design <- design(
 n = 3,
 level = 5,
 distribution = "binomial",
 n_trials = 20,
 start_value = 0.5
)
study <- random_scdf(design)
transform(study, proportion = values/trials, percentage = proportion * 100)

## Z standardize the dependent variable and add two new variables:
exampleAB |>
  transform(
    values = scale(values),
    mean_values = mean(values),
    sd_values = sd(values)
  )

## Use `all` to calculate global variables.
exampleAB |>
  transform(
    values_center_case = values - mean(values[phase == "A"]),
    values_center_global = values - mean(all(values[phase == "A"])),
    value_dif = values_center_case - values_center_global
  )

## Use `across_cases` to calculate or replace a variable with values from
## all cases. E.g., standardize the dependent variable:
exampleABC |>
  transform(
    across_cases(values = scale(values))
  )

## Rank transform the values based on all cases vs. within each case:
exampleABC |>
  transform(
    across_cases(values_across = rank(values, na.last="keep")),
    value_within = rank(values, na.last="keep")
  )

## Three helper functions to smooth the data
Huber2014$Berta |>
transform(
  "compliance (moving median)" = moving_median(compliance),
  "compliance (moving mean)" = moving_mean(compliance),
  "compliance (local regression)" = local_regression(compliance, mt)
)

## Function first_of() helps to set NAs for specific phases.
## E.g., you want to replace the first two values of phase A and the first
## value of phase B and its preceding value.

byHeart2011 |>
  transform(
    values = set_na_at(values, phase == "A", 0:1),
    values = set_na_at(values, phase == "B", -1:0)
  )
}
\seealso{
Other data manipulation functions: 
\code{\link{add_l2}()},
\code{\link{as.data.frame.scdf}()},
\code{\link{as_scdf}()},
\code{\link{fill_missing}()},
\code{\link{print.sc_outlier}()},
\code{\link{ranks}()},
\code{\link{rescale}()},
\code{\link{scdf}()},
\code{\link{select_cases}()},
\code{\link{set_vars}()},
\code{\link{shift}()},
\code{\link{smooth_cases}()},
\code{\link{standardize}()},
\code{\link{truncate_phase}()}
}
\concept{data manipulation functions}
