% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convolve.R
\name{convolve}
\alias{convolve}
\alias{convolve_signal}
\alias{convolve_image}
\alias{convolve_volume}
\title{Convolution of \code{1D}, \code{2D}, \code{3D} data via \code{FFT}}
\usage{
convolve_signal(x, filter)

convolve_image(x, filter)

convolve_volume(x, filter)
}
\arguments{
\item{x}{one-dimensional signal vector, two-dimensional image, or
three-dimensional volume; numeric or complex}

\item{filter}{kernel with the same number of dimensions as \code{x}}
}
\value{
Convolution results with the same length and dimensions as \code{x}.
If \code{x} is complex, results will be complex, otherwise results will
be real numbers.
}
\description{
Use the 'Fast-Fourier' transform to compute the convolutions of two data
with zero padding. This function is mainly designed for image convolution.
For forward and backward convolution/filter, see \code{\link{filtfilt}}.
}
\details{
This implementation uses 'Fast-Fourier' transform to perform
\code{1D}, \code{2D}, or \code{3D} convolution. Compared to implementations
using original mathematical definition of convolution, this approach is
much faster, especially for image and volume convolutions.

The input \code{x} is zero-padded beyond edges. This is most common in image
or volume convolution, but less optimal for periodic one-dimensional signals.
Please use other implementations if non-zero padding is needed.

The convolution results might be different to the ground truth by a precision
error, usually at \code{1e-13} level, depending on the \code{'FFTW3'}
library precision and implementation.
}
\examples{


# ---- 1D convolution ------------------------------------
x <- cumsum(rnorm(100))
filter <- dnorm(-2:2)
# normalize
filter <- filter / sum(filter)
smoothed <- convolve_signal(x, filter)

plot(x, pch = 20)
lines(smoothed, col = 'red')

# ---- 2D convolution ------------------------------------
x <- array(0, c(100, 100))
x[
  floor(runif(10, min = 1, max = 100)),
  floor(runif(10, min = 1, max = 100))
] <- 1

# smooth
kernel <- outer(dnorm(-2:2), dnorm(-2:2), FUN = "*")
kernel <- kernel / sum(kernel)

y <- convolve_image(x, kernel)

oldpar <- par(mfrow = c(1,2))
image(x, asp = 1, axes = FALSE, main = "Origin")
image(y, asp = 1, axes = FALSE, main = "Smoothed")
par(oldpar)


}
