% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base_surv.R
\name{qgcomp.cox.noboot}
\alias{qgcomp.cox.noboot}
\title{Quantile g-computation for survival outcomes under linearity/additivity}
\usage{
qgcomp.cox.noboot(
  f,
  data,
  expnms = NULL,
  q = 4,
  breaks = NULL,
  id = NULL,
  weights,
  cluster = NULL,
  alpha = 0.05,
  ...
)
}
\arguments{
\item{f}{R style survival formula, which includes \code{\link[survival]{Surv}}
in the outcome definition. E.g. \code{Surv(time,event) ~ exposure}. Offset
terms can be included via \code{Surv(time,event) ~ exposure + offset(z)}}

\item{data}{data frame}

\item{expnms}{character vector of exposures of interest}

\item{q}{NULL or number of quantiles used to create quantile indicator variables
representing the exposure variables. If NULL, then gcomp proceeds with un-transformed
version of exposures in the input datasets (useful if data are already transformed,
or for performing standard g-computation)}

\item{breaks}{(optional) NULL, or a list of (equal length) numeric vectors that
characterize the minimum value of each category for which to
break up the variables named in expnms. This is an alternative to using 'q'
to define cutpoints.}

\item{id}{(optional) NULL, or variable name indexing individual units of
observation (only needed if analyzing data with multiple observations per
id/cluster)}

\item{weights}{"case weights" - passed to the "weight" argument of
\code{\link[survival]{coxph}}}

\item{cluster}{(from coxph function help) optional variable which clusters the observations, for the purposes of a robust variance. If present, it implies robust. This variable will normally be found in data.}

\item{alpha}{alpha level for confidence limit calculation}

\item{...}{arguments to coxph}
}
\value{
a qgcompfit object, which contains information about the effect
measure of interest (psi) and associated variance (var.psi), as well
as information on the model fit (fit) and information on the
weights/standardized coefficients in the positive (pos.weights) and
negative (neg.weights) directions.
}
\description{
This function performs quantile g-computation in a survival
setting. The approach estimates the covariate-conditional hazard ratio for
a joint change of 1 quantile in each exposure variable specified in expnms
parameter
}
\details{
For survival outcomes (as specified using methods from the
survival package), this yields a conditional log hazard ratio representing
a change in the expected conditional hazard (conditional on covariates)
from increasing every exposure by 1 quantile. In general, this quantity
quantity is not equivalent to g-computation estimates. Hypothesis test
statistics and 95\% confidence intervals are based on using the delta
estimate variance of a linear combination of random variables.
}
\examples{
set.seed(50)
N=200
dat <- data.frame(time=(tmg <- pmin(.1,rweibull(N, 10, 0.1))), 
                d=1.0*(tmg<0.1), x1=runif(N), x2=runif(N), z=runif(N))
expnms=paste0("x", 1:2)
f = survival::Surv(time, d)~x1 + x2
(fit1 <- survival::coxph(f, data = dat))
(obj <- qgcomp.cox.noboot(f, expnms = expnms, data = dat))
\dontrun{

# weighted analysis
dat$w = runif(N)
qdata = quantize(dat, expnms=expnms)
(obj2 <- qgcomp.cox.noboot(f, expnms = expnms, data = dat, weight=w))
obj2$fit
survival::coxph(f, data = qdata$data, weight=w)

# not run: bootstrapped version is much slower
(obj2 <- qgcomp.cox.boot(f, expnms = expnms, data = dat, B=200, MCsize=20000))
}
}
\seealso{
\code{\link[qgcomp]{qgcomp.cox.boot}}, \code{\link[qgcomp]{qgcomp.glm.boot}},
and \code{\link[qgcomp]{qgcomp}}

Other qgcomp_methods: 
\code{\link{qgcomp.cch.noboot}()},
\code{\link{qgcomp.cox.boot}()},
\code{\link{qgcomp.glm.boot}()},
\code{\link{qgcomp.glm.ee}()},
\code{\link{qgcomp.glm.noboot}()},
\code{\link{qgcomp.hurdle.boot}()},
\code{\link{qgcomp.hurdle.noboot}()},
\code{\link{qgcomp.multinomial.boot}()},
\code{\link{qgcomp.multinomial.noboot}()},
\code{\link{qgcomp.partials}()},
\code{\link{qgcomp.zi.boot}()},
\code{\link{qgcomp.zi.noboot}()}
}
\concept{qgcomp_methods}
\concept{variance mixtures}
