#' Count a scale's degrees of freedom
#'
#' Some scalar structures can vary their specific pitches much more flexibly
#' than others while retaining the same overall "color." For instance, the
#' meantone family of diatonic scales is generated by a line of fifths and
#' can only vary along one dimension: the size of the generating fifth.
#' This literally defines a line in the MCT geometry, and if the scale moves
#' off that line it ceases to have the same structure as the diatonic scale.
#' (Notably, it stops being non-degenerate well-formed.) By contrast, the
#' 5-limit just diatonic scale is defined by two distinct parameters: the 
#' size of its major third and the size of its perfect fifth. See "Modal 
#' Color Theory," pp. 26-27, for more discussion.
#' 
#' @inheritParams signvector
#' 
#' @returns Single non-negative integer
#' 
#' @examples
#' c_natural_minor <- c(0, 2, 3, 5, 7, 8, 10)
#' c_melodic_minor <- c(0, 2, 3, 5, 7, 9, 11)
#' just_diatonic <- j(dia)
#' howfree(c_natural_minor)
#' howfree(c_melodic_minor)
#' howfree(just_diatonic)
#'
#' howfree(c(0, 4, 7))
#' howfree(c(0, 4, 7), ineqmat="white")
#'
#' howfree(c(0, 2, 6), ineqmat="mct")
#' howfree(c(0, 2, 6), ineqmat="roth")
#'
#' @export
howfree <- function(set, ineqmat=NULL, edo=12, rounder=10) {
  card <- length(set)
  if (card < 2) { 
    return(0) 
  }

  ineqmat <- choose_ineqmat(set, ineqmat)

  zeroesflat <- ineqmat[whichsvzeroes(set, ineqmat, edo, rounder), ]
  rank <- qr(zeroesflat)$rank
  card - (1+rank)
}