#' @name PlotPredictSurvFit
#' 
#' @title Plotting method for \code{SurvPredict} objects
#'
#' @description
#' This is the generic \code{plot} S3 method for the \code{SurvPredict} class. It
#' plots concentration-response fit under target time survival analysis.
#' 
#' @param x an object of class \code{SurvPredict}
#' @param xlab argument for the label of the x-axis
#' @param ylab argument for the label of the y-axis
#' @param main argument for the title of the graphic
#' @param background_concentration Binary. If TRUE (default is FALSE), it print the
#' background exposure profile.
#' @param add_legend add legend to the plot, default is \code{FALSE}
#' @param \dots Further arguments to be passed to generic methods
#' 
#' @return an object of class \code{ggplot}, see function \code{\link[ggplot2]{ggplot}}
#'
#' @keywords plot
#'
#' @export
plot.SurvPredict <- function(x,
                             xlab = "Time",
                             ylab = "Number of Survival",
                             main = "Survival Probability with 95% Credible Interval",
                             background_concentration = FALSE,
                             add_legend = FALSE,
                             ...) {

    df_plt <- x$df_quantile
    plt <- ggplot(data = df_plt) + 
        theme_minimal() + 
        labs(x = xlab, y = ylab, title = main)
    
    if (background_concentration) {
        plt <- plt +
            scale_colour_gradient(
                name = "Concentration",
                low = cexpmin, high = cexpmax ) +
            geom_segment(aes(x = time, xend = time,
                             y = -Inf, yend = Inf, color = conc),
                         show.legend = add_legend, linewidth = 2)
    }
    
    plt <- plt +
        geom_ribbon(aes(x = time, ymin = qinf95, ymax = qsup95), fill = fillci) +
        geom_line(aes(x = time, y = q50), color = colmed) +
        facet_grid(~ replicate)
    
    return(plt)
    
}
