#' Numerical method for maximum likelihood estimation
#'
#' @description
#'  In maximum likelihood estimation, if the Hessian matrix is difficult to obtain,
#'  Numerical method (means an iteration optimization method) will be used to obtain the model parameter.
#'
#'
#'
#' @param x vector of data.
#' @param param inintial four parameters (\code{alpha}, \code{beta}, \code{lambda}, and \code{beta})
#' @param method a numerical method for maximum likelihood estimation, a default method is "L-BFGS-B",
#'               the other specify "BFGS"
#'
#' @importFrom stats optim
#'
#' @references Zhu, C., Byrd, R. H., Lu, P., & Nocedal, J. (1997). Algorithm 778: L-BFGS-B.
#' ACM Transactions on Mathematical Software, 23(4), 550–560. https://doi.org/10.1145/279232.279236.
#'
#'
#' @returns
#' \code{outout} is a list of variables as follows:
#' \code{est_param} gives the estimated four parameters,
#' \code{neg_likelihood} gives the negative-log-likelihood value,
#' \code{code_conversgence} is code of convergence, if \code{0}, convergence , otherwise, divergence.
#' \code{num_method} is numerical method, a default method is \code{"L-BFGS-B"}
#'
#' @export
#'
#' @examples
#' y <- rMOGIW(100,1,3,2,3)
#' pars <- c(alpha=1, beta=3, lambda=2, theta=3)
#' mleMOGIW(x=y,param=pars,method = "L-BFGS-B")
#'
mleMOGIW <- function(x,param,method="L-BFGS-B"){
  #--- Negative Log-likelihood: neg_loglik()
  neg_loglik <- function(par=param, x) {
    alpha <- par[1];  beta <- par[2];  lambda <- par[3];  theta <- par[4]
    n <- length(x)
    t <- (alpha / x)^beta
    t1 <- n*log(lambda) + n*log(beta) + n*log(theta) + n*beta*log(alpha)
    t2 <- -(beta+1)*sum(log(x))
    t3 <- -lambda*sum(t)
    t4 <-  -2 * sum(log(theta - (theta - 1)*exp(-lambda*t)))
    ll <- (t1+t2+t3+t4)
    return(-ll)
  }
  # method
  if (method == "L-BFGS-B"){
    lower_bounds <- c(1e-6, 1e-6, 1e-6, 1e-6)
    upper_bounds <- c(Inf,   Inf,   Inf, Inf)
    fit <- optim(par=param, fn=neg_loglik, x=x,method="L-BFGS-B",
                 lower  = lower_bounds, upper  = upper_bounds,
                 control = list(fnscale = 1, factr = 1e7,maxit=1e4))
  }
  else{
    fit <- optim(par = param, fn = neg_loglik, x = x, method = "BFGS",
                 control = list(maxit=1000, reltol=1e-8),  hessian = TRUE)
  }
  output <- list(est_param=fit$par,neg_likelihood=fit$value,code_conversgence=fit$convergence,num_method=method)
  return(output)
}

