% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{pulse_normalize}
\alias{pulse_normalize}
\title{Normalize PULSE heartbeat rate estimates}
\usage{
pulse_normalize(
  heart_rates,
  FUN = mean,
  t0 = NULL,
  span_mins = 10,
  overwrite = FALSE
)
}
\arguments{
\item{heart_rates}{the output from \code{\link[=PULSE]{PULSE()}}, \code{\link[=pulse_heart]{pulse_heart()}}, \code{\link[=pulse_doublecheck]{pulse_doublecheck()}} or \link{pulse_choose_keep}.}

\item{FUN}{the function to be applied to normalize the data within the baseline period (defaults to \link{mean}; \link{median} may be more suited in some situations; any other function that returns a single numeric value is technically acceptable).}

\item{t0}{either \code{NULL} (default), a \link[lubridate:posix_utils]{lubridate::POSIXct} object or a character string that can be directly converted to a \link[lubridate:posix_utils]{lubridate::POSIXct} object. Represents the beginning of the period to be used to establish the baseline heart beat frequency (same value is used for all channels). If set to \code{NULL}, the baseline period is set to the earliest timestamp available.}

\item{span_mins}{numeric, defaults to 10; number of minutes since \code{t0}, indicating the width of the baseline period (baseline from \code{t0} to \code{t0} + \code{span_mins} mins)}

\item{overwrite}{logical, defaults to \code{FALSE}; should the normalized values be stored in a different column (\code{hz_norm} if \code{overwrite = FALSE}; RECOMMENDED) or replace the data in the column \code{hz} (\code{overwrite = TRUE}; WARNING: the original \code{hz} values cannot be recovered).}
}
\value{
The same tibble provided as input, with an additional column \code{hz_norm} containing the normalized heart beat frequencies (\code{overwrite = FALSE}) or with the same number of columns and normalized data saved to the column \code{hz} (\code{overwrite = TRUE}).
}
\description{
Take the output from \code{\link[=PULSE]{PULSE()}} and compute the normalized heartbeat rates. The normalization of heartbeat rates is achieved by calculating, for each individual (i.e., PULSE channel), the average heartbeat rate during a reference baseline period (ideally measured during acclimation, before the stress-inducing treatment is initiated).
}
\section{Details}{

Normalizing heartbeat rates is important because even individuals from the same species, the same age cohort and subjected to the same treatment will have different basal heart beat frequencies. After normalizing, these differences are minimized, and the analysis can focus on the change of hear beat frequency relative to a reference period (the baseline period chosen) rather than on the absolute values of heart beat frequency - which can be misleading.

The period chosen for the baseline doesn't need to be long - it's much more important that conditions (and hopefully heart beat frequencies) are as stable and least stressful as possible during that period.

After normalization, heart beat frequencies during the baseline period will, by definition, average to \code{1}. Elsewhere, normalized heart beat frequencies represent ratios relative to the baseline: \code{2} represents a heart beat frequency double the basal frequency, while \code{0.5} indicates half of the basal frequency. This means that two individuals may experience a doubling of heart beat frequency throughout an experiment even if their absolute heart beat frequencies are markedly different from each other (e.g., individual 1 with hz = 0.6 at t0 and hz = 1.2 at t1, and individual 2 with hz = 0.8 at t0 and hz = 1.6 at t1, will both show hz_norm = 1 at t0 and hz_norm = 2 at t1).
}

\section{Different baseline periods for each channel}{

\code{pulse_normalize} only allows setting a single baseline period. If different periods are needed for different channels or groups of channels, generate two or more subsets of \code{heart_rates} containing \code{heart_rates$id} that share the same baseline periods, normalize each independently and bind all data together at the end (see the examples section below).
}

\examples{
## Begin prepare data ----
pulse_data_sub <- pulse_data
pulse_data_sub$data <- pulse_data_sub$data[,1:5]
pulse_data_split <- pulse_split(
   pulse_data_sub,
   window_width_secs = 30,
   window_shift_secs = 60,
   min_data_points = 0.8)
pulse_data_split <- pulse_optimize(pulse_data_split, multi = pulse_data$multi)
heart_rates <- pulse_heart(pulse_data_split)
heart_rates <- pulse_doublecheck(heart_rates)
## End prepare data ----

# Normalize data using the same period as baseline for all channels
pulse_normalize(heart_rates)

# Using a different (complex) function
pulse_normalize(heart_rates, FUN = function(x) quantile(x, 0.4))

# Apply different baseline periods to two groups of IDs
group_1 <- c("limpet_1", "limpet_2")
rbind(
  # group_1
  pulse_normalize(heart_rates[ (heart_rates$id \%in\% group_1), ], span_mins = 10),
  # all other IDs
  pulse_normalize(heart_rates[!(heart_rates$id \%in\% group_1), ], span_mins = 30)
)
}
\seealso{
\itemize{
\item \code{\link[=pulse_heart]{pulse_heart()}}, \code{\link[=pulse_doublecheck]{pulse_doublecheck()}} and \code{\link[=pulse_choose_keep]{pulse_choose_keep()}} are the functions that generate the input for \code{pulse_normalize}
\item \code{\link[=pulse_plot]{pulse_plot()}} can be called to visualize the output from \code{pulse_normalize}
\item \code{\link[=PULSE]{PULSE()}} is a wrapper function that executes all the steps needed to process PULSE data at once, and its output can also be passed on to \code{pulse_normalize}
}
}
