% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel_ram.R
\name{kernel_ram}
\alias{kernel_ram}
\title{Robust Adaptive Metropolis (RAM) Transition Kernel}
\usage{
kernel_ram(
  mu = 0,
  eta = function(i, k) min(c(1, i^(-2/3) * k)),
  qfun = function(k) stats::rt(k, k),
  arate = 0.234,
  freq = 1L,
  warmup = 0L,
  Sigma = NULL,
  eps = 1e-04,
  lb = -.Machine$double.xmax,
  ub = .Machine$double.xmax,
  fixed = FALSE,
  until = Inf,
  constr = NULL
)
}
\arguments{
\item{mu}{Either a numeric vector or a scalar. Proposal mean.
If scalar, values are recycled to match the number of parameters in the
objective function.}

\item{eta}{A function that receives the MCMC environment. This is to calculate
the scaling factor for the adaptation.}

\item{qfun}{Function. As described in Vihola (2012)'s, the \code{qfun} function is
a symmetric function used to generate random numbers.}

\item{arate}{Numeric scalar. Objective acceptance rate.}

\item{freq}{Integer scalar. Frequency of updates. How often the
variance-covariance matrix is updated.}

\item{warmup}{Integer scalar. The number of iterations that the algorithm has
to wait before starting to do the updates.}

\item{Sigma}{The variance-covariance matrix. By default this will be an
identity matrix during the warmup period.}

\item{eps}{Double scalar. Default size of the initial step (see details).}

\item{lb, ub}{Either a numeric vector or a scalar. Lower and upper bounds for
bounded kernels. When of length 1, the values are recycled to match the number
of parameters in the objective function.}

\item{fixed}{Logical scalar or vector of length \code{k}. Indicates which parameters
will be treated as fixed or not. Single values are recycled.}

\item{until}{Integer scalar. Last step at which adaptation takes place (see
details).}

\item{constr}{Logical lower-diagonal square matrix of size \code{k}. \strong{Not} in the
original paper, but rather a tweak that imposes a constraint on the \code{S_n}
matrix. If different from \code{NULL}, the kernel multiplates \code{S_n} by this
constraint so that zero elements are pre-imposed.}
}
\value{
An object of class \link{fmcmc_kernel}.
}
\description{
Implementation of Vihola (2012)'s Robust Adaptive Metropolis.
}
\details{
While it has been shown that under regular conditions this transition kernel
generates ergodic chains even when the adaptation does not stop, some
practitioners may want to stop adaptation at some point.

The idea is similar to that of the Adaptive Metropolis algorithm (AM implemented
as \code{\link[=kernel_adapt]{kernel_adapt()}} here) with the difference that it takes into account a
target acceptance rate.

The \code{eta} function regulates the rate of adaptation. The default implementation
will decrease the rate of adaptation exponentially as a function of the iteration
number.

\deqn{%latex
Y_n\equiv X_{n-1} + S_{n-1}U_n,\quad\mbox{where }U_n\sim q\mbox{ (the \texttt{qfun})}%
}{%
Y_n := X_{n-1} + S_{n-1} U_n , where U_n ∼ q (the qfun)%
}

And the \eqn{S_n} matrix is updated according to the following equation:

\deqn{% latex
S_nS_n^T = S_{n-1}\left(I + \eta_n(\alpha_n - \alpha_*)\frac{U_nU_n^T}{\|U_n\|^2}\right)S_{n-1}^T%
}{%
S_n S_n^T = S_{n-1} {I + eta_n[alpha_n - alpha_*] U_n U_n^T/norm(U_n)^2} S_{n-1}^T%
}
}
\examples{
# Setting the acceptance rate to 30 \% and deferring the updates until
# after 1000 steps
kern <- kernel_ram(arate = .3, warmup = 1000)
}
\references{
Vihola, M. (2012). Robust adaptive Metropolis algorithm with coerced acceptance
rate. Statistics and Computing, 22(5), 997–1008.
\doi{10.1007/s11222-011-9269-5}
}
\seealso{
Other kernels: 
\code{\link{kernel_adapt}()},
\code{\link{kernel_mirror}},
\code{\link{kernel_new}()},
\code{\link{kernel_normal}()},
\code{\link{kernel_unif}()}
}
\concept{kernels}
