% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DiscSurvDataTransform.R
\name{dataLongCompRisksTimeDep}
\alias{dataLongCompRisksTimeDep}
\title{Data Long Competing Risks Time Dependent Covariates Transformation}
\usage{
dataLongCompRisksTimeDep(
  dataSemiLong,
  timeColumn,
  eventColumns,
  eventColumnsAsFactor = FALSE,
  idColumn,
  timeAsFactor = FALSE,
  responseAsFactor = FALSE
)
}
\arguments{
\item{dataSemiLong}{Original data in semi-long format ("class data.frame").}

\item{timeColumn}{Character giving the column name of the observed times("logical vector"). It
is required that the observed times are discrete ("integer vector").}

\item{eventColumns}{Character vector giving the column names of the event
indicators (excluding censoring column)("character vector"). It is required that all events are
binary encoded. If the sum of all event indicators is zero, then this is
interpreted as a censored observation. Alternatively a column name of a
factor representing competing events can be given. In this case the argument
\emph{eventColumnsAsFactor} has to be set TRUE and the first level is assumed to
represent censoring.}

\item{eventColumnsAsFactor}{Should the argument eventColumns be intepreted
as column name of a factor variable ("logical vector")? Default is FALSE.}

\item{idColumn}{Name of column of identification number of persons as
character("character vector").}

\item{timeAsFactor}{Should the time intervals be coded as factor ("logical vector")? Default is
FALSE. In the default settings the discrete time intervals 
are treated as quantitative ("numeric vector").}

\item{responseAsFactor}{Should the response columns be given as factor ("logical vector")? 
Default is FALSE.}
}
\value{
Original data set in long format with additional columns \itemize{
\item {obj} Gives identification number of objects (row index in short
format) (integer) \item {timeInt} Gives number of discrete time intervals
(factor) \item {responses} Columns with dimension count of events + 1
(censoring) \itemize{ \item {e0} No event (observation censored in specific
interval) \item {e1} Indicator of first event, 1 if event takes place and 0
otherwise \item ... ...  \item {ek} Indicator of last k-th event, 1 if event
takes place and 0 otherwise}
If argument responseAsFactor=TRUE, then responses will be coded as factor in one column.
}
}
\description{
Transforms short data format to long format for discrete survival modelling
in the case of competing risks with right censoring. Covariates may vary
over time.
}
\details{
There may be some intervals, where no additional information on the
covariates is observed (e. g. observed values in interval one and three but
two is missing). In this case it is assumed, that the values from the last
observation stay constant over time until a new measurement was done.

In contrast to continuous survival (see e. g. \code{\link[survival]{Surv}}) 
the start and stop time notation is not used here. In discrete time survival analysis the only relevant
information is to use the stop time. Start time does not matter, because all discrete intervals need to be  
included in the long data set format to ensure consistent estimation. It is assumed that the supplied 
data set \emph{dataSemiLong} contains all repeated measurements of each cluster in semi-long format (e. g. persons). 
For further information see example \emph{Start-stop notation}.
}
\examples{

# Example Primary Biliary Cirrhosis data
library(survival)
pbcseq_example <- pbcseq

# Convert to months
pbcseq_example$day <- ceiling(pbcseq_example$day/30) + 1
names(pbcseq_example)[7] <- "month"
pbcseq_example$status <- factor(pbcseq_example$status)

# Convert to long format for time varying effects
pbcseq_exampleLong <- dataLongCompRisksTimeDep(dataSemiLong = pbcseq_example, timeColumn = "month", 
eventColumns = "status", eventColumnsAsFactor = TRUE, idColumn = "id", 
timeAsFactor = TRUE)
head(pbcseq_exampleLong)

#####################
# Start-stop notation

library(survival)
?pbcseq

# Choose subset of patients
subsetID <- unique(pbcseq$id)[1:100]
pbcseq_mod <- pbcseq[pbcseq$id \%in\% subsetID, ]

# Convert to start stop notation
pbcseq_mod_split <- split(pbcseq_mod, pbcseq_mod$id)
pbcseq_mod_split <- lapply(1:length(pbcseq_mod_split), function(x) {

 cbind(pbcseq_mod_split[[x]], 
 start_time=c(0, pbcseq_mod_split[[x]][ - dim(pbcseq_mod_split[[x]])[1], "day"]),
 stop_time=pbcseq_mod_split[[x]][, "day"])
 
})
pbcseq_mod <- do.call(rbind, pbcseq_mod_split)

# Convert stop time to months
intervalDef <- c(quantile(pbcseq_mod$stop_time, probs = seq(0.1, 0.9, by=0.1)), Inf)
names(pbcseq_mod)
pbcseq_mod <- contToDisc(dataShort = pbcseq_mod, timeColumn = "stop_time", 
                         intervalLimits = intervalDef, equi = FALSE)
pbcseq_mod$status <- factor(pbcseq_mod$status)

# Conversion to data long format
pbcseq_mod_long <- dataLongCompRisksTimeDep(dataSemiLong = pbcseq_mod, timeColumn = "timeDisc", 
                                           eventColumns = "status",
                                          idColumn = "id", 
                                           eventColumnsAsFactor = TRUE, 
                                          responseAsFactor = TRUE,
                                          timeAsFactor = TRUE)
head(pbcseq_mod_long)

}
\references{
\insertRef{fahrmeirDiscSurv}{discSurv} \cr\cr
\insertRef{thompsonTreatment}{discSurv}
}
\seealso{
\code{\link{contToDisc}}, \code{\link{dataLong}},
\code{\link{dataLongCompRisks}}
}
\author{
Thomas Welchowski \email{welchow@imbie.meb.uni-bonn.de}
}
\keyword{datagen}
