# Data in Package ####
#' @details
#' Type \code{RShowDoc("dendrometry", package = "dendrometry")} to read a HTML
#' user guide vignette.
#'
# Type \code{RShowDoc("dendrometry_pdf", package = "dendrometry")} to read a
# PDF user guide vignette. NOT WORKING NOW!
#'
#' Type \code{demo(dendro, package = "dendrometry")} for a demo of dendrometric
#' computations. Click on \code{Index} bellow to see the index of the package.
#'
#' Type \code{demo(volume, package = "dendrometry")} for a demo of dendrometric
#'  computations. Click on \code{Index} bellow to see the index of the package.
#' @keywords internal
"_PACKAGE"


#' Dendrometric measures on tree
#' @docType data
#' @description Data frame of 10 rows and 5 columns containing tree measures.
#' @format Data frame with ten observations and five variables:
#' \describe{
#'   \item{circum}{Tree circumference in centimeter (cm).}
#'   \item{dist}{Horizontal distance between the person measuring angles and
#'   the tree (m).}
#'   \item{up}{Angle measured for the top part of the tree in degree (°).
#'   It is used to calculate the total tree height.}
#'   \item{down}{Angle measured for the bottom part of the tree in degree (°).}
#'   \item{fut}{Bole angle measure in degree (°); Bole is where the first branch
#'   occurs on the trunk. It is used to calculate the merchantable tree height.}
#' }
#' @usage data(Tree)
#'
#' @examples # demo(dendro)
#' @source Fake data simulated for tutorial purposes.
#' @author Narcisse Yehouenou \email{narcisstar211@gmail.com}
"Tree"

#' Tree metrics for logging
#' @docType data
#' @description Data frame of 24 rows and 8 columns containing tree measures.
#' @format Data frame with twenty five observations and eight variables:
#' \describe{
#'   \item{tree}{Tree name (scientific gender).}
#'   \item{hauteur}{Stem length in meter (m).}
#'   \item{diametreMedian}{Tree median diameter in centimeter (cm).}
#'   \item{perimetreMedian}{Tree median circumference in centimeter (cm).}
#'
#'   \item{diametreSection}{Tree diameter at the end in centimeter (cm).}
#'   \item{perimetreSection}{Tree circumference at the end in centimeter (cm).}
#'
#'   \item{diametreBase}{Tree diameter at the base in centimeter (cm).}
#'   \item{perimetreBase}{Tree circumference at the base in centimeter (cm).}
#' }
#' @usage data(Logging)
#' @examples # demo(volume)
#' @source Fake data simulated for tutorial purposes.
#' @author Narcisse Yehouenou \email{narcisstar211@gmail.com}
"Logging"

# Methods ####

#' Print Angle
#'
#' Method to print angle and returns it invisibly.
#'
#' @param x an angle object.
#' @param ... further arguments passed to or from other methods.
#' @export
#'
print.angle <- function(x, ...) {
  unit <- switch(attr(x, "unit"),
    deg = "degrees (\\u00b0)",
    rad = "radian"
  )
  invisible(cat(unit, "angle values. \n", round(x, 2), "\n"))
}

#' Print Slope
#'
#' Method to print slope and returns it invisibly.
#'
#' @param x a slope object.
#' @param ... further arguments passed to or from other methods.
#' @export
#'
print.slope <- function(x, ...) {
  invisible(cat("slope values in percentage(%). \n", round(x, 2), "\n"))
}

# Core Functions ####

#' Recursive function for creating nested data structures
#' @description
#' Internal recursive function that creates nested list structures by
#' sequentially grouping data by factors and applying a function to each subset.
#' This is the core engine behind nestedFunBuilder and related functions.
#'
#' @param currentData data frame containing the current subset of data to process.
#' @param remainingFactors character vector of factor names still to be processed.
#' @param fun function to apply to each final data subset. Default is \code{identity}.
#' @param funArgs list of additional arguments to pass to \code{fun}.
#'
#' @return A nested list structure where each level corresponds to a factor,
#' or the result of applying \code{fun} to the data subset if no factors remain.
#'
#' @details
#' This function works recursively:
#' \itemize{
#'   \item Base case: If no factors remain, applies \code{fun} to the current data
#'   \item Recursive case: Groups data by the first remaining factor, then calls
#'     itself on each subset with the remaining factors
#' }
#'
#' @keywords internal
baseFunction <- function(currentData, remainingFactors, fun = identity, funArgs = list()) {
  # Base case: no more factors to process
  if (length(remainingFactors) == 0) {
    # Apply the function to the current data subset
    if (identical(fun, identity)) {
      return(currentData)
    } else {
      # Add 'data' as first argument if the function expects it
      return(do.call(fun, c(list(data = currentData), funArgs)))
    }
  }

  # Get current factor and remaining factors
  currentFactor <- remainingFactors[1]
  nextFactors <- remainingFactors[-1]

  # Get unique values for current factor
  uniqueValues <- unique(currentData[[currentFactor]])

  # Create subsets for each unique value
  result <- lapply(uniqueValues, function(value) {
    subsetData <- subset(currentData, currentData[[currentFactor]] == value)
    baseFunction(
      currentData = subsetData, remainingFactors = nextFactors,
      fun = fun, funArgs = funArgs
    )
  })

  # Name and return the list elements
  setNames(result, nm = uniqueValues)
}

#' Create nested data structures with optional function application
#' @description
#' Creates nested list structures by grouping data according to specified factors
#' and optionally applying a function to each final subset. This is the main
#' function for creating hierarchical data structures for analysis.
#'
#' @param data a data frame, list, tibble or object coercible by
#' \code{\link{as.data.frame}} to a data frame containing the variables whose
#' names are given in the factor arguments.
#'
#' @param ... character strings specifying the names of grouping variables
#' (factors) in \code{data}. The nesting order follows the argument order.
#'
#' @param .fun function to apply to each final data subset. Default is
#' \code{identity} which returns the data subset unchanged.
#'
#' @param .funArgs list of additional arguments to pass to \code{.fun}.
#'
#' @return A nested list structure where each level corresponds to a factor
#' level. If \code{.fun = identity}, returns data subsets. Otherwise, returns
#' the result of applying \code{.fun} to each subset.
#'
#' @details
#' The function performs the following steps:
#' \itemize{
#'   \item Validates that all specified factors exist in the data
#'   \item Warns about missing factors and removes them from processing
#'   \item Creates nested subsets using the specified factors
#'   \item Applies the specified function to each final subset
#' }
#'
#' @examples
#' \dontrun{
#' # Create nested data structure
#' nested_data <- nestedFunBuilder(iris, "Species", "Sepal.Length > 5")
#'
#' # Apply function to each subset
#' means <- nestedFunBuilder(iris, "Species",
#'   .fun = function(data) mean(data$Sepal.Length)
#' )
#'
#' # Multiple factors with function
#' results <- nestedFunBuilder(mydata, "site", "species", "treatment",
#'   .fun = myAnalysisFunction,
#'   .funArgs = list(method = "robust")
#' )
#' }
#'
#' @keywords internal
nestedFunBuilder <- function(data, ..., .fun = identity, .funArgs = list()) {
  # Get factor arguments
  factorArgs <- list(...)

  # Convert to character vector, handling different input types
  if (length(factorArgs) == 0) {
    factors <- character(0)
  } else {
    factors <- as.character(unlist(factorArgs))
  }

  # Remove empty strings and NULL values
  factors <- factors[factors != "" & !is.na(factors)]

  # Validate inputs
  if (length(factors) > 0 && !all(sapply(factors, is.character))) {
    stop("All factor arguments should be character strings.\n")
  }

  # Check if factors exist in data
  existingFactorsIndex <- factors %in% names(data)
  missingFactors <- factors[!existingFactorsIndex]
  if (length(missingFactors) > 0) {
    warning(
      "The following are not considered because they are not variable names in the data: ",
      paste(missingFactors, collapse = ", "), ".\n"
    )
    # Remove missing factors
    factors <- factors[existingFactorsIndex]
  }

  # If no valid factors, return original data with warning
  if (length(factors) == 0 && identical(.fun, identity)) {
    warning("No valid factors were defined. The same data is returned.")
    return(data)
  }

  # Create the nested structure
  result <- baseFunction(
    currentData = data,
    remainingFactors = factors,
    fun = .fun,
    funArgs = .funArgs
  )
  return(result)
}

## New functions ####
#' Format Confidence Interval for Display
#'
#' @keywords internal
format_ci <- function(ci_vals, digits = 2) {
  paste0("[", round(ci_vals[1], digits), ", ", round(ci_vals[2], digits), "]")
}


#' Format Confidence Interval for Inline Display
#'
#' @keywords internal
format_ci_inline <- function(ci_vals, digits = 2) {
  paste0("[", round(ci_vals[1], digits), ", ", round(ci_vals[2], digits), "]")
}


#' Compute Confidence Bands Using Delta Method
#'
#' @keywords internal
ci_bands_delta <- function(params, vcov, x_range, amplitude, level = 0.95) {
  # Generate sequence of x values for evaluation
  x_seq <- seq(x_range[1], x_range[2], length.out = 200)

  # Compute density at each x
  densities <- dweibull3(x_seq, params["shape"], params["scale"], params["loc"])
  densities_scaled <- 100 * amplitude * densities

  # Compute gradient of log-density with respect to parameters
  # This is a simplified approximation - for exact CI, use numerical derivatives
  z_alpha <- qnorm(1 - (1 - level) / 2)

  # Approximate SE using delta method (simplified)
  # For more accurate bands, would need full gradient computation
  se_approx <- sqrt(diag(vcov)[1]) * densities_scaled / params["shape"]

  lower <- pmax(densities_scaled - z_alpha * se_approx, 0)
  upper <- densities_scaled + z_alpha * se_approx

  list(x = x_seq, lower = lower, upper = upper, fitted = densities_scaled)
}

#' Compute Confidence Bands Using Parametric Bootstrap
#'
#' @keywords internal
ci_bands_bootstrap <- function(x, params, method, ties_method,
                               x_range, amplitude, n_boot = 1000,
                               level = 0.95) {
  n <- length(x)
  x_seq <- seq(x_range[1], x_range[2], length.out = 200)

  # Store bootstrap density curves
  boot_densities <- matrix(NA, nrow = n_boot, ncol = length(x_seq))

  for (i in 1:n_boot) {
    # Generate bootstrap sample
    x_boot <- rweibull3(n, params["shape"], params["scale"], params["loc"])

    # Fit to bootstrap sample
    fit_boot <- tryCatch(
      {
        fit_dist(x_boot,
          dist = "weibull3", method = method,
          start = params, ties_method = ties_method
        )
      },
      error = function(e) NULL
    )

    if (!is.null(fit_boot) && fit_boot$convergence == 0) {
      # Compute density curve
      boot_densities[i, ] <- 100 * amplitude *
        dweibull3(
          x_seq, fit_boot$estimate["shape"],
          fit_boot$estimate["scale"], fit_boot$estimate["loc"]
        )
    }
  }

  # Compute quantiles across bootstrap samples
  alpha <- 1 - level
  lower <- apply(boot_densities, 2, quantile, probs = alpha / 2, na.rm = TRUE)
  upper <- apply(boot_densities, 2, quantile, probs = 1 - alpha / 2, na.rm = TRUE)

  list(x = x_seq, lower = lower, upper = upper)
}


# Useful functions ####


#' Skewness coefficient
#' @param x numeric vector.
#' @examples data("Logging")
#' skewness(Logging$hauteur)
#' hist(Logging$hauteur, 3)
#' @return The skewness coefficient.
#' @import stats
#' @export
skewness <- function(x) {
  if (!is.numeric(x)) {
    stop("'x' must be numeric. \n")
  }

  x <- na.omit(x)
  n <- length(x)
  n * (sum(((x - mean(x)) / sd(x))**3)) / ((n - 1) * (n - 2))
}

#' Degree and Radian
#' @description \code{deg} converts angle values from radians to degrees. \cr
#' \code{rad} converts angle values from degrees to radians.
#'
#' @aliases rad degree radian
#' @usage deg(radian)
#' @usage rad(degree)
#'
#' @param radian numeric, vector of radian values to be converted to degrees.
#' @param degree numeric, vector of degree values to be converted to radians.
#'
#' @return \code{deg} returns vector of degree values while
#' \code{rad} returns vector of radian values.
#'
#' @examples
#' deg(pi / 2)
#' rad(180)
#'
#' @seealso \code{\link{principal}}.
#'
#' @export
deg <- function(radian) {
  if (!is.numeric(radian)) {
    stop("'radian' must be numeric. \n")
  }

  angle <- radian * (180 / pi)
  attr(angle, "unit") <- "deg"

  class(angle) <- c("angle", "numeric")
  return(angle)
}

#' @export
rad <- function(degree) {
  if (!is.numeric(degree)) {
    stop("'degree' must be numeric. \n")
  }

  angle <- degree * (pi / 180)
  attr(angle, "unit") <- "rad"

  class(angle) <- c("angle", "numeric")
  return(angle)
}


#' Angle - Slope conversion and Principal Measure determination
#' @description Conversion of angle to slope values and reciprocally.\cr
#' \code{angle2slope} converts angle to slope values. \cr
#' \code{slope2angle} converts slope to angle values. \cr
#' \code{principal} determines the principal measure of an angle value.
#' Principal measure ranges from -pi to pi for radian unit while it ranges from
#' -180 to 180 for degree unit.
#'
#' @aliases slope2angle principal
#'
#' @usage angle2slope(angle, angleUnit = c("deg", "rad"))
#' @usage slope2angle(slope, angleUnit = c("deg", "rad"))
#' @usage principal(angle, angleUnit = c("deg", "rad"))
#'
#' @param angle numeric, vector of angle to be converted to slope.
#' @param slope numeric, vector of slope to be converted to angle.
#'
#' @param angleUnit character, unit of \code{angle}.\cr
#' For \code{slope2angle}, the desired unit for the returned angle value.\cr
#' For \code{principal}, both the angle input and output unit.\cr
#' Either \code{deg} or \code{rad}. Default is \code{deg}.
#'
#' @return Object of class \code{angle}.\cr
#' \code{angle2slope} returns vector of slope values while
#' \code{slope2angle} and \code{principal} return vector of angle values in unit
#' specified in \code{angle} argument.
#'
#' @examples
#' angle2slope(10)
#' angle2slope(angle = 45)
#' angle2slope(angle = pi / 4, angleUnit = "rad")
#' angle2slope(1.047198, "rad")
#' angle2slope(seq(0.2, 1.5, .4), angleUnit = "rad") #'
#'
#' slope2angle(100)
#' slope2angle(100, "rad")
#' round(pi / 4, 2)
#'
#' slope2angle(17.6327)
#' slope2angle(angle2slope(30))
#'
#' principal(303)
#' principal(23 * pi / 8, "rad")
#' principal(7 * pi / 4, angleUnit = "rad")
#' deg(principal(7 * pi / 4, angleUnit = "rad"))
#' principal(7 * 45)
#'
#' @seealso \code{\link{deg}} and \code{\link{rad}}.
#'
#' @note Use \code{principal} in position computations, not distance computations.
#' @export
angle2slope <- function(angle, angleUnit = c("deg", "rad")) {
  if (!(angleUnit[[1L]] %in% c("deg", "rad"))) {
    stop("angleUnit should be either 'deg' or 'rad'. \n")
  }

  if (missing(angleUnit) || angleUnit == "deg") {
    angle <- rad(angle)
  }

  slope <- 100 * tan(angle)
  class(slope) <- c("slope", "numeric")

  return(slope)
}

#' @export
slope2angle <- function(slope, angleUnit = c("deg", "rad")) {
  if (!(angleUnit[[1L]] %in% c("deg", "rad"))) {
    stop("angleUnit should be either 'deg' or 'rad'. \n")
  }

  angle <- atan(.01 * slope)

  if (missing(angleUnit) || angleUnit == "deg") {
    angle <- deg(angle)
    attr(angle, "unit") <- "deg"
  } else {
    attr(angle, "unit") <- "rad"
  }

  class(angle) <- c("angle", "numeric")
  return(angle)
}

#' @export
principal <- function(angle, angleUnit = c("deg", "rad")) {
  if (!(angleUnit[[1L]] %in% c("deg", "rad"))) {
    stop("angleUnit should be either 'deg' or 'rad'. \n")
  }

  if (missing(angleUnit) || angleUnit == "deg") {
    angle <- rad(angle)
    angle <- Arg(complex(real = cos(angle), imaginary = sin(angle)))
    angle <- deg(angle)
    attr(angle, "unit") <- "deg"
  } else {
    angle <- Arg(complex(real = cos(angle), imaginary = sin(angle)))
    attr(angle, "unit") <- "rad"
  }

  class(angle) <- c("angle", "numeric")
  return(angle)
}


#' Horizontal distance
#' @description Horizontal distance calculation for sloping area.
#'
#' @param distance numeric, vector of the distance measured on sloping area.
#' @param angle numeric, vector of angle or slope values.
#' @param type character, type of \code{angle} argument.
#' Either \code{"angle"} or \code{"slope"}. Default is \code{"slope"}.
#' @param  angleUnit character, unit of \code{angle} measures if
#' \code{type = "angle"}. Either \code{"deg"} for degree or \code{"rad"} for
#' radian. Default is \code{"deg"}.
#'
#' @return A vector of horizontal distance.
#' @examples
#' distanceH(20, 30)
#' distanceH(20, angle = 30, type = "slope")
#' distanceH(20, angle = 25, type = "angle")
#' @export
#'
distanceH <- function(distance, angle, type = c("slope", "angle"),
                      angleUnit = c("deg", "rad")) {
  if (!(type[[1L]] %in% c("slope", "angle"))) {
    stop("'type' should be either 'angle' or 'slope'. \n")
  }

  if (missing(type) || type == "slope") {
    angle <- slope2angle(slope = angle, angleUnit = "rad")
  } else if (missing(angleUnit) || angleUnit == "deg") {
    angle <- rad(angle)
  }
  as.numeric(distance * cos(angle))
}

#' Relative Frequency
#' @description
#' Relative Frequency in percentage.
#' @param x numeric vector.
#' @export
rfreq <- function(x) {
  100 * x / sum(x)
}


#' Girard Form Class
#' Girard Form Class is a form quotient used to estimate taper.
#'
#' @param dbhIn numeric, diameter inside bark at the top of the first log
#' @param dbh numeric, diameter outside bark at breast height.
#' @references Strimbu, B. (2021). Dendrometry Field Manual.
#' @export
#'
girard <- function(dbh, dbhIn) {
  if (is.numeric(dbh) && is.numeric(dbhIn)) {
    return(dbhIn / dbh)
  } else {
    stop("'dbh' and 'dbhIn' should be numeric. \n")
  }
}


#' Bark factor
#' The bark factor (k) is computed for trees in order to assess the importance
#' of the valuable wood in the overall volume of a tree (Husch et al., 1982):
#'
#' @param thickness numeric, bark thickness measured on individual trees.
#' @param dbh numeric, diameter over bark of the individual trees.
#'
#' @references Husch, B., Miller, C., Beers, T., 1982. Forest mensuration.
#' Ronald Press Company, London, pp. 1 – 410.
#' @export

barkFactor <- function(dbh, thickness) {
  if (is.numeric(dbh) && is.numeric(thickness)) {
    dbhIn <- dbh - 2 * thickness
    res <- sum(dbhIn) / sum(dbh)
  } else {
    stop("'dbh' and 'thickness' should be numeric. \n")
  }

  return(res)
}



#' Create nested data subsets
#' @description
#' A convenient wrapper around \code{nestedFunBuilder} that creates nested
#' data subsets without applying any function. This is useful for exploring
#' data structure or preparing data for further analysis.
#'
#' @param data a data frame, list, tibble or object coercible by
#' \code{\link{as.data.frame}} to a data frame containing the variables whose
#' names are given in the factor arguments.
#'
#' @param ... character strings specifying the names of grouping variables
#' (factors) in \code{data}. The nesting order follows the argument order.
#'
#' @return A nested list structure where each level corresponds to a factor
#' level, with the deepest level containing the actual data subsets. If no valid
#' factors are provided, returns the original data frame with a warning.
#'
#' @details
#' This function is equivalent to calling \code{nestedFunBuilder} with
#' \code{.fun = identity}. It provides a simpler interface when you only
#' need to create nested data structures without applying functions.
#'
#' @examples
#' \dontrun{
#' # require(BiodiversityR)
#' # data(ifri, package = "BiodiversityR")
#' # a1 <- makedata(ifri, "forest", "plotID", "species")
#' # a2 <- makedata(ifri, "species")
#' # a3 <- makedata(ifri, "forest", "plotID", "species", "size_class")
#' # identical(makedata(ifri), ifri)
#' }
#'
#' @seealso \code{\link{nestedFunBuilder}} for applying functions to subsets
#'
#' @export
makedata <- function(data, ...) {
  nestedFunBuilder(data = data, .fun = identity, ...)
}


#' The three-parameter Weibull Distribution
#' @description
#' Density, distribution function, quantile function and random generation for
#' the three-parameter Weibull.
#'
#' @aliases pweibull3 qweibull3 rweibull3
#'
#' @usage dweibull3(x, shape, scale = 1, loc = 0, log = FALSE)
#' pweibull3(q, shape, scale, loc = 0, lower.tail = TRUE, log.p = FALSE)
#' qweibull3(p, shape, scale, loc = 0, lower.tail = TRUE, log.p = FALSE)
#' rweibull3(n, shape, scale = 1, loc = 0)
#'
#' @param x,q vector of quantiles.
#'
#' @param p vector of probabilities.
#'
#' @param n number of observations. If \code{length(n) > 1}, the length is
#' taken to be the number required.
#'
#' @param shape,scale,loc shape, scale and location parameters. The two latter
#' default to  \code{1} and \code{0} respectively.
#'
#' @param log,log.p logical; if \code{TRUE}, probabilities \code{p} are given as
#' \code{log(p)}.
#'
#' @param lower.tail logical; if TRUE (default), probabilities are
#' \eqn{P[X \le x]}, otherwise, \eqn{P[X > x]}.
#'
#' @seealso \code{\link[stats]{dweibull}} for the Weibull distribution.
#'
#' @export
dweibull3 <- function(x, shape, scale = 1, loc = 0, log = FALSE) {
  dweibull(x - loc, shape, scale, log = log)
}

#' @export
pweibull3 <- function(q, shape, scale, loc = 0, lower.tail = TRUE, log.p = FALSE) {
  pweibull(q - loc, shape, scale, lower.tail, log.p)
}

#' @export
qweibull3 <- function(p, shape, scale, loc = 0, lower.tail = TRUE, log.p = FALSE) {
  qweibull(p, shape, scale, lower.tail, log.p) + loc
}

#' @export
rweibull3 <- function(n, shape, scale = 1, loc = 0) {
  rweibull(n, shape, scale) + loc
}

# END ####

#' Making factor vectors
#' @description Changes character vectors of a data set to factor vectors.
#' @aliases factorise
#' @param data data frame or tibble data set.
#' @param binary logical indicating if binary numeric data should be considered
#' as factor.
#' Default is \code{FALSE}.
#' @return Data frame with all character vectors changed to factor vectors.
#' @details When \code{binary = TRUE}, variables stored as numeric and which have
#' exactly two levels are changed to factor.
#' @export
factorize <- function(data, binary = FALSE) {
  # if(!"data.frame" %in% class(data)) warning("data is not a data frame.\n")

  for (i in 1:length(data)) {
    if (is.character(data[[i]]) ||
      (binary && is.numeric(data[[i]]) &&
        length(levels(as.factor(data[[i]]))) == 2)) {
      data[[i]] <- as.factor(data[[i]])
    }

    # else data[[i]] <- data[[i]]
  }
  return(data)
}

#' @export
factorise <- factorize

#' Sample size
#'
#' @param confLev numeric, the confidence level. Default is \code{0.05}.
#' @param popPro numeric, proportion of population which have considered factor.
#' Default is \code{0.5}.
#'
#' @param errorMargin numeric, margin error. Default is \code{0.05}.
#' @param size integer, population size when it is known. If not specified,
#' simple random sampling will be used. Allows infinite.
#'
#' @param method optional character string specifying method to use if not
#' simple adjusted is desired. Only "cauchran" is implemented now.
#'
#' @param cv variation coefficient.
#' @note Population size to be considered as large or infinite heavily depends
#' on error margin. Lower error margin increases population size to be conidered
#'  as large or infinite. For errorMargin = .05, size = 152 231 and cauchran
#'  151 760 when confLev = .05
#' @return The sample size.

#' @examples sampleSize(confLev = .95, popPro = 0.4, errorMargin = .05)
#' sampleSize(confLev = .95, popPro = 0.5, errorMargin = .05, size = 150)
#' sampleSize(
#'   confLev = .95, popPro = 0.5, errorMargin = .05, size = 150,
#'   method = "cauchran"
#' )
#' sampleSize()
#'
#' @export
sampleSize <- function(confLev = .95, popPro = .5, errorMargin = .05,
                       size = NULL, method = "", cv = NULL) {
  alpha <- 1 - confLev
  if (!missing(cv)) {
    n <- qt(p = 1 - (alpha / 2), df = size - 1)**2 * cv**2 / errorMargin**2
    return(n)
  }
  if (is.null(size) || is.infinite(size)) {
    n <- qnorm(p = 1 - (alpha / 2))**2 * popPro * (1 - popPro) / errorMargin**2
    return(n)
  } else if (method == "cauchran") {
    if (size >= 30) {
      n <- qnorm(p = 1 - (alpha / 2))**2 * popPro * (1 - popPro) / errorMargin**2
    } else {
      n <- qt(p = 1 - (alpha / 2), df = size - 1)**2 * popPro * (1 - popPro) / errorMargin**2
    }
    nc <- n / ((n - 1) / size + 1)
    return(nc)
  } else {
    if (size >= 30) {
      n <- qnorm(p = 1 - (alpha / 2))**2 * popPro * (1 - popPro) / errorMargin**2
    } else {
      n <- qt(p = 1 - (alpha / 2), df = size - 1)**2 * popPro * (1 - popPro) / errorMargin**2
    }
    np <- size * n / (size + n)
    return(np)
  }
}


#' Stack all vectors of a data frame or list
#' @description Stacking all columns of a data frame or vectors of a list into
#' a single vector.
#' @param data data frame, tibble or list.
#' @return A vector of all element of the argument \code{data}.
#' @export
stacking <- function(data) {
  if (inherits(data, what = "list")) {
    data <- list2DF(data)
  }
  as.vector(as.matrix(data))
}


#' Fibonacci series
#' @description Generates numbers from Fibonacci series.
#' @param n integer, the size of the series.
#' @param Uo,U1 integer, the first two numbers of the series.
#' @param PrintFib logical, indicating if the series should be printed.
#' @return Either an integer, result of the function or a vector of \code{n}
#' first numbers of the series.
#' @examples fibonacci(n = 10, PrintFib = TRUE)
#' fibonacci(n = 10, Uo = 1, U1 = 3, PrintFib = FALSE)
#' @details The series equation is Un = U_(n-2) /U_(n-1).
#' @seealso \code{\link{fiboRate}}
#' @author Narcisse Yehouenou \email{narcisstar211@gmail.com}
#' @export
fibonacci <- function(n, PrintFib = FALSE, Uo = 0, U1 = 1) {
  Un <- numeric(length = n)
  Un[1:2] <- c(Uo, U1)

  if (n < 2) {
    return(Uo)
  } else if (n > 2) {
    for (i in 3:n) {
      Un[i] <- Un[i - 1] + Un[i - 2]
    }
    Fib <- Un
  } else {
    Fib <- Un
  }

  if (!PrintFib) {
    Fib <- Fib[n]
  }

  return(Fib)
}

#' Fibonacci series ratio
#' @description Computes rates from Fibonacci series.
#' @param n integer, the size of the series.
#' @param Uo,U1 integer, the first number of the series.
#' @param PrintSer logical, indicating if the series should be printed.
#' @return Either a numeric, result of the rate of \code{nth} and \code{(n-1)th}
#'  numbers
#' in Fibonacci series or all \code{(n-1)th} those rates.
#' @examples ## Golden number (Le Nombre d'Or)
#' fiboRate(n = 18, PrintSer = FALSE, Uo = 0, U1 = 1)
#' ## (1+sqrt(5))/2
#' fiboRate(n = 10, PrintSer = TRUE, Uo = 0, U1 = 1)
#' @details The series equation is Un = U_(n-2) /U_(n-1).
#' The function returns golden number when Uo = 0, and U1 = 1. Larger n is, more
#'  precise the number (result) is.
#' @seealso \code{\link{fibonacci}}
#' @author Narcisse Yehouenou \email{narcisstar211@gmail.com}
#' @export

fiboRate <- function(n, PrintSer = FALSE, Uo = 0, U1 = 1) {
  a <- fibonacci(n = n, Uo = Uo, U1 = U1, PrintFib = TRUE)
  # $$U_n = \frac{U_{n-2}}{U_{n-1}}$$
  serie <- a[2:n] / a[1:n - 1]
  if (PrintSer) {
    return(serie)
  } else {
    return(serie[n - 1])
  }
}

# Tree parameter ####

#' Diameter (DBH) and Circumference
#' @description \code{DBH} computes diameter (at breast height) based on
#' circumference (at breast height). \cr
#' \code{circum} computes circumference (at breast height) based on diameter
#' (at breast height). \cr
#' They are based on circle diameter and perimeter formulas.
#'
#' @aliases circum
#' @usage dbh(circum)
#' @usage circum(dbh)
#'
#' @param circum numeric, vector of circumference.
#' @param dbh numeric, vector of diameter.
#'
#' @return \code{dbh}, returns diameter and \code{circum}, returns circumference.
#' @examples
#' perimeter <- seq(30, 60, 1.4)
#' diameter <- dbh(perimeter)
#' circum(diameter)
#'
#' @seealso See also \code{\link{height}} for tree  height.
#' @export
dbh <- function(circum) {
  if (is.numeric(circum)) {
    return(circum / pi)
  } else {
    stop("'circum' must be numeric. \n")
  }
}

#' @export
circum <- function(dbh) {
  if (is.numeric(dbh)) {
    return(dbh * pi)
  } else {
    stop("'dbh' must be numeric. \n")
  }
}


#' Individual Basal Area and DBH (diameter)
#' @description \code{basal_i} computes the basal area of a tree stem
#' (individual), the area of a circle of diameter \code{dbh}. \cr
#' \code{basal2dbh} computes the dbh (diameter) based on the basal area.
#'
#' @aliases basal2dbh
#'
#' @usage basal_i(dbh, circum = NULL)
#' @usage basal2dbh(basal)
#'
#' @param dbh numeric, vector of diameter.
#' @param circum numeric, vector of circumference. Is used only if \code{dbh}
#' is not given.
#' @param basal numeric, individual basal area.
#'
#' @return \code{basal_i} returns individual basal area while \code{basal2dbh}
#' returns DBH.
#'
#' @examples
#' basal_i(dbh = 10)
#' basal_i(circum = 31.41)
#' basal2dbh(78.53982)
#'
#' @details If \code{circum} is given, \code{dbh} is not used.
#' @export
basal_i <- function(dbh = NULL, circum = NULL) {
  if (is.numeric(circum)) {
    dbh <- dbh(circum = circum)
  } else if (!is.numeric(dbh)) {
    stop("'dbh' or 'circum' should be given and numeric. \n")
  }

  return(pi * .25 * dbh**2)
}

#' @export
basal2dbh <- function(basal) {
  2 * sqrt(basal / pi)
}



#' Height of Tree or any vertical Object
#' @description Computes the height of tree, pillar, girder, mast or any
#' vertical object. It allows either slope (in percent) or angle (in degrees or
#' radians). No matter the relative position of the persons who measures the
#' angle or the slope.
#'
#' @param distance numeric, vector of the horizontal distance between object
#' and the person who measures angle.
#' @param top,bottom numeric vector of top angle and bottom angle respectively
#' (readings from a clinometer).
#' @param type the type of \code{top} and \code{bottom} measures. Either
#' \code{"angle"} or \code{"slope"}. Default is \code{"slope"}.
#' @param  angleUnit the unit of \code{top} and \code{bottom} measures when
#' \code{type = "angle"}. Either \code{"deg"} for degree or \code{"rad"} for
#' radian. Default is \code{"deg"}.
#'
#' @return A vector of heights.
#' @examples
#' height(10, 80, 17)
#' height(17, top = -18, bottom = -113)
#' height(distance = 18, top = 42, bottom = -12, type = "angle", angleUnit = "deg")
#' height(
#'   distance = 18:21, top = 42:45, bottom = -12:-15, type = "angle",
#'   angleUnit = "deg"
#' )
#' ## Below shows warning messages
#' height(
#'   distance = 18:21, top = -42:-45, bottom = -12:-15, type = "angle",
#'   angleUnit = "deg"
#' )
#' @export
height <- function(distance, top, bottom, type = c("angle", "slope"),
                   angleUnit = c("deg", "rad")) {
  if (!is.numeric(distance)) {
    stop("'distance' must be numeric. \n")
  }

  if (!is.numeric(top)) {
    stop("'top' must be numeric. \n")
  }

  if (!is.numeric(bottom)) {
    stop("'bottom' must be numeric. \n")
  }

  if (!(type[[1L]] %in% c("slope", "angle"))) {
    stop("'type' should be either 'angle' or 'slope'. \n")
  }

  if (!(angleUnit[[1L]] %in% c("deg", "rad"))) {
    stop("'angleUnit' should be either 'deg' or 'rad'. \n")
  }

  if (sum(top <= bottom)) {
    warning("One or more top angles are less than or equal to their bottom angles. \n")
  }

  if (missing(type) || type == "slope") {
    res <- as.numeric(0.01 * distance * (top - bottom))
  } else {
    if (missing(angleUnit) || angleUnit == "deg") {
      top <- rad(top)
      bottom <- rad(bottom)
    }

    res <- as.numeric(distance * (tan(top) - tan(bottom)))
  }
  return(res)
}

# END ####

#' The decrease coefficient
#' @description This coefficient expresses the ratio between the diameter
#' (or circumference) at mid-height of the bole and the diameter
#' (or circumference) measured at breast height.
#' @param middle numeric, the diameter or circumference at middle height.
#' @param breast numeric, the diameter or circumference at breast height.
#' @details Both \code{middle} and \code{breast} arguments should be of the
#' same type (either diameter or circumference). Not mixture.
#' @return A vector of decrease coefficients.
#' @examples decrease(30, 120)
#' decrease(middle = 40, breast = 90)
#' @export
decrease <- function(middle, breast) {
  if (!is.numeric(middle)) {
    stop("'middle' should be numeric")
  }

  if (!is.numeric(breast)) {
    stop("'breast' should be numeric")
  }

  if (any(middle > breast)) {
    warning("One or more breast value are less than their middle values.
          Please check your data")
  }
  return(middle / breast)
}


#' The reduction coefficient
#' @description The reduction coefficient is the ratio between the difference
#' in size at breast height and mid-height on the one hand, and the size at
#' breast height on the other. It is thus the complement to 1 of the
#' coefficient of decrease.
#' @param middle numeric, the diameter or circumference at middle height.
#' @param breast numeric, the diameter or circumference at breast height.
#' @details Both \code{middle} and \code{breast} arguments should be of the
#' same type (either diameter or circumference). Not mixture.
#' @return The reduction coefficient.
#' @examples reducecoef(30, 120)
#' reducecoef(middle = 40, breast = 90)
#' @seealso \code{decrease}
#' @export
reducecoef <- function(middle, breast) {
  if (!is.numeric(middle)) {
    stop("'middle' should be numeric")
  }

  if (!is.numeric(breast)) {
    stop("'breast' should be numeric")
  }

  if (any(middle > breast)) {
    warning("One or more breast value are less than their middle values.
          Please check your data")
  }
  r <- (breast - middle) / breast
  return(r)
}


#' Metric scrolling or decay
#' @description The average metric decay expresses the difference, in
#' centimeters per meter, between the diameter (or circumference) at breast
#' height and its diameter at mid-height of a stem related to the difference
#' between the height at mid-height and that at breast height.
#' @param dmh numeric, the diameter at middle height in centimeter (cm).
#' @param dbh numeric, the diameter at breast height in centimeter (cm).
#' @param mh numeric, the middle (or cut) height in meter (m).
#' @param bh Either a numeric value standing for the breast height in meter (m)
#' of all trees or a numeric vector standing for the breast height of each tree.
#' Default is \code{1.3}.
#' @return Metric decay
#' @examples decreaseMetric(dmh = 40, dbh = 90, mh = 7)
#' decreaseMetric(45, 85, 9)
#' @seealso \code{reducecoef}
#' @export
decreaseMetric <- function(dmh, dbh, mh, bh = 1.3) {
  if (!is.numeric(dbh)) {
    stop("'dbh' should be numeric")
  }
  if (!is.numeric(dmh)) {
    stop("'dmh' should be numeric")
  }
  if (!is.numeric(mh)) {
    stop("'mh' should be numeric")
  }

  if (any(dmh > dbh)) {
    warning("One or more middle height diameter are greater than their DBH.
            Please check your data")
  }
  if (any(mh <= bh)) {
    warning("One or more middle height are greater or equal to their breast
    height. Please check your data")
  }
  d <- (dbh - dmh) / (mh - bh)
  return(d)
}


.huberMethod <- function(height, dm, circum, successive, log) {
  if (is.null(dm) && is.null(circum)) {
    stop("Specify either 'dm' or 'circum'")
  } else if (!is.null(dm) && !is.null(circum)) {
    warning("Don't specify both 'dm' (diameter) and 'circum' (circumference).
            Only 'dm' is considered.")
  }

  if (is.null(dm)) {
    dm <- dbh(circum)
  }
  v <- .25 * pi * dm**2 * height

  if (successive) {
    v <- sapply(unique(log), FUN = function(i) sum(v[log == i]))
  }

  return(v)
}
.smalianMethod <- function(height, do, ds, circumo, circums, successive,
                           log) {
  # Don't specify mixture of args
  if (all(
    any(is.null(circumo), is.null(circums)),
    any(is.null(do), is.null(ds))
  )) {
    stop("Specify either both 'circumo' and 'circums' or both 'do' and 'ds'
           when using 'smalian' method.")
  } else if (sum(
    !is.null(do), !is.null(ds), !is.null(circumo),
    !is.null(circums)
  ) > 2) {
    warning("Don't specify both diameters and circumferences.")
  }
  # le && est inutile
  if (!is.null(do) && !is.null(ds)) {
    v <- .125 * pi * (do**2 + ds**2) * height
  } else if (!is.null(circumo) && !is.null(circums)) {
    v <- .125 * pi * (dbh(circumo)**2 + dbh(circums)**2) * height
  }

  if (successive) {
    v <- sapply(unique(log), FUN = function(i) sum(v[log == i]))
  }
  return(v)
}
.coneMethod <- function(height, do, ds, circumo, circums, successive, log) {
  # Don't specify mixture of args
  if (all(
    any(is.null(circumo), is.null(circums)),
    any(is.null(do), is.null(ds))
  )) {
    stop("Specify either 'circumo' and 'circums' or 'do' and 'ds'
           when using 'cone' method.")
  } else if (sum(
    !is.null(do), !is.null(ds), !is.null(circumo),
    !is.null(circums)
  ) > 2) {
    warning("Don't specify both diameters and circumferences.")
  }
  # le && est inutile
  if (!is.null(do) && !is.null(ds)) {
    v <- pi * (do**2 + do * ds + ds**2) * height / 12
  } else if (!is.null(circumo) && !is.null(circums)) {
    v <- pi * (dbh(circumo)**2 + dbh(circumo) * dbh(circums) + dbh(circums)**2) *
      height / 12
  }
  if (successive) {
    v <- sapply(unique(log), FUN = function(i) sum(v[log == i]))
  }
  return(v)
}
.newtonMethod <- function(height, do, dm, ds, circumo, circum, circums,
                          successive, log) {
  if (all(
    any(is.null(circum), is.null(circumo), is.null(circums)),
    any(is.null(dm), is.null(do), is.null(ds))
  )) {
    stop("Specify either 'circum', 'circumo' and 'circums' or 'dm', 'do' and 'ds'
           when using 'newton' method.")
  } # "Specify either only diameters or only circumferences."
  else if (sum(
    is.null(dm), !is.null(do), !is.null(ds), is.null(circum),
    !is.null(circumo), !is.null(circums)
  ) > 3) {
    warning("Don't specify both diameters and circumferences.")
  }

  if (!is.null(dm) && !is.null(do) && !is.null(ds)) {
    v <- pi * (do**2 + 4 * dm**2 + ds**2) * height / 24
  } else if (!is.null(circum) && !is.null(circumo) && !is.null(circums)) {
    v <- pi * (dbh(circumo)**2 + 4 * dbh(circum)**2 + dbh(circums)**2) *
      height / 24
  }

  if (successive) {
    v <- sapply(unique(log), FUN = function(i) sum(v[log == i]))
  }
  return(v)
}


#' Tree stem and log Volume
#' @description Determining the volume of the log or of the tree.
#' @usage volume(height, dm, do, ds, circum, circumo, circums,
#'        method = "huber", successive = FALSE, log)
#' @param height numeric, stem (whole bole) length. When \code{successive} is
#' "\code{TRUE}",
#' it stands for log length.
#' @param do,dm,ds numeric, respectively base, median and end diameter.
#' @param circumo,circum,circums numeric, respectively base, median and end
#' circumference.
#' @param method character string, the method of volume computation. Can be one
#' of "\code{huber}", "\code{smalian}", "\code{cone}", or "\code{newton}".
#' Default is "\code{huber}".
#' @param successive logical. If \code{TRUE}, Successive method is applied.
#' is applied. Default is \code{FALSE}.
#' @param log a vector indicating tree to which belongs each log.
#' Is used only when \code{successive} is "\code{TRUE}".
#' @examples ## huber method
#' volume(height = 10, dm = 35)
#' volume(height = 10, circum = 100)
#'
#' ## smalian method
#' volume(height = 10, do = 45, ds = 15, method = "smalian")
#' volume(height = 10, circumo = 200, circums = 110, method = "smalian")
#'
#' ## cone method
#' volume(height = 10, do = 45, ds = 15, method = "cone")
#' volume(height = 10, circumo = 200, circums = 110, method = "cone")
#'
#' ## newton method
#' volume(height = 10, dm = 35, do = 45, ds = 15, method = "newton")
#' volume(
#'   height = 10, circum = 100, circumo = 200, circums = 110,
#'   method = "newton"
#' )
#' @return A numeric vector of logs or trees volume.
#' @details Using \code{method = cone} refers to truncated cone method.
#' @seealso \code{\link{shape}}, for shape coefficient.
#' @export
volume <- function(height, dm = NULL, do = NULL, ds = NULL, circum = NULL,
                   circumo = NULL, circums = NULL, method = "huber",
                   successive = FALSE, log = NULL) {
  if (!(method %in% c("huber", "smalian", "cone", "newton"))) {
    stop("'method' should be one of 'huber', 'smalian', 'cone', or 'newton'.")
  }

  if (all(!successive, !is.null(log))) {
    warning("Don't specify 'log' when 'successive' is not TRUE")
  } # unused arg ...


  if (method == "huber") {
    return(.huberMethod(
      height = height, dm = dm,
      circum = circum, log = log,
      successive = successive
    ))
  } else if (method == "smalian") {
    return(.smalianMethod(
      height = height, do = do,
      circumo = circumo,
      ds = ds,
      circums = circums,
      log = log,
      successive = successive
    ))
  } else if (method == "cone") {
    return(.coneMethod(
      height = height, do = do,
      ds = ds, circumo = circumo,
      circums = circums,
      successive = successive,
      log = log
    ))
  } else if (method == "newton") {
    return(.newtonMethod(
      height = height, do = do,
      dm = dm, ds = ds,
      circumo = circumo,
      circum = circum,
      circums = circums,
      successive = successive,
      log = log
    ))
  }
}


#' The shape coefficient
#' @description The shape coefficient of the tree is the ratio of the actual
#' volume of the tree to the volume of a cylinder having as base the surface of
#' the section at 1.3 m (or a given breast height) and as length, the height
#' (at bole level) of the tree.
#' @usage shape(volume, height, dbh, basal = NULL)
#' @param volume numeric, tree real volume.
#' @param height numeric, tree height.
#' @param dbh numeric, diameter at breast height (DBH).
#' @param basal numeric, basal area. Is used when \code{dbh} is not specified.
#' @examples shape(volume = 10000, 11, dbh = 40)
#' shape(volume = 10000, 11, 40)
#' shape(volume = 10000, 11, basal = 2256.637)
#' ## Bellow gives warning
#' shape(volume = 10000, height = 11, dbh = 40, basal = 2256.637)
#' @return The shape coefficient.
#' @seealso \code{\link{volume}}, for tree real volume.
#' @export
shape <- function(volume, height, dbh = NULL, basal = NULL) {
  if (all(is.null(dbh), is.null(basal))) {
    stop("Specify either 'dbh' or 'basal'")
  } else if ((!any(is.null(dbh), is.null(basal)))) {
    warning("Both of 'dbh' and 'basal' are specified. Only 'dbh' is considered.")
  }

  if (!is.null(dbh)) {
    f <- volume / (basal_i(dbh = dbh) * height)
  } else {
    f <- volume / (basal * height)
  }
  return(f)
}

# Plot parameter ####
#' @title Structural parameters for stands
#'
#' @description
#' Computes various forest stand parameters (basal area, mean diameter, height, etc.)
#' for forest inventory data, with support for grouping by multiple factors and
#' plot-level analysis.
#'
#' @param data a data frame, list, tibble or object coercible by
#' \code{\link{as.data.frame}} to a data frame containing the forest inventory
#' variables.
#'
#' @param plot optional character, name of the variable containing plot identifiers.
#' If empty (""), all data is treated as a single plot.
#'
#' @param DBH optional, character, name of the variable containing diameter at breast
#' height measurements.
#'
#' @param height optional, character, name of the variable containing tree height
#' measurements.
#'
#' @param crown optional, character, name of the variable containing crown diameter
#' measurements.
#'
#' @param area optional, numeric value of plot area, or character name of variable
#' containing plot areas. If NULL, density calculations are omitted.
#'
#' @param k numeric, conversion factor for basal area calculation (default: 100).
#'
#' @param kCrown numeric, conversion factor for crown basal area calculation
#' (default: 1).
#'
#' @param ... additional character strings specifying grouping variables
#' (factors) in \code{data}. Results will be nested by these factors.
#'
#' @return A nested list structure containing calculated parameters for each
#' group. Parameters include:
#' \itemize{
#'   \item \code{MeanDBH}: Mean diameter at breast height
#'   \item \code{Basal}: Basal area per unit area
#'   \item \code{MeanCrown}: Mean crown diameter
#'   \item \code{BasalCrown}: Crown basal area per unit area
#'   \item \code{Height}: Mean height
#'   \item \code{LoreyHeight}: Lorey's height (basal area weighted mean height)
#'   \item \code{Density}: Number of trees per unit area
#' }
#'
#' @details
#' The function supports hierarchical grouping by multiple factors. For example,
#' grouping by species and site will create a nested structure where parameters
#' are calculated for each species within each site.
#'
#' If plot-level analysis is requested (plot != ""), the function will further
#' subdivide each group by plot and calculate parameters for each plot within
#' each group. Else, it treats the entire dataset as a single plot; area should
#' then be specified accordingly.
#'
#' Blackman and Green indices are returned if combinations of specified factors
#' contain more than one plot. Otherwise, the right (correct) ones are returned as attributes.
#'
#' @examples
#' param(
#'   data = Logging, plot = "tree", DBH = "diametreMedian",
#'   height = "hauteur", crown = "perimetreBase", area = 0.03, kCrown = 100
#' )
#'
#' set.seed(123)
#' Logging$surperficie <- abs(rnorm(24, mean = 0.03, sd = 0.01))
#' head(Logging)
#'
#' param(
#'   data = Logging, plot = "tree", DBH = "diametreMedian",
#'   height = "hauteur", crown = "perimetreBase", area = "surperficie", kCrown = 100
#' )
#'
#' \dontrun{
#' # Basic usage - single plot
#' params <- param(forest_data, DBH = "dbh", height = "height", area = 1000)
#'
#' # Multiple plots
#' params <- param(forest_data,
#'   plot = "plot_id", DBH = "dbh",
#'   height = "height", area = "plot_area"
#' )
#'
#' # Grouped analysis
#' params <- param(forest_data,
#'   plot = "plot_id", DBH = "dbh",
#'   height = "height", area = 1000,
#'   "species", "site", "treatment"
#' )
#'
#' # Access specific results
#' oak_site1 <- params$oak$site1
#' }
#'
#' @export
param <- function(data, ..., plot = "", DBH = "", height = "", crown = "",
                  area = NULL, k = 100, kCrown = 1) {
  # Single calculation function
  calc <- function(dat, a) {
    nbTrees <- nrow(dat)

    if (DBH != "") {
      MeanDBH <- diameterMean(dbh = dat[[DBH]])

      Basal <- if (!is.null(a)) {
        basal(dbh = dat[[DBH]], area = a, k = k)
      } else {
        NULL
      }
    } else {
      Basal <- MeanDBH <- NULL
    }

    if (crown != "") {
      MeanCrown <- diameterMean(dbh = dat[[crown]])
      BasalCrown <- if (!is.null(a)) {
        basal(dbh = dat[[crown]], area = a, k = kCrown)
      } else {
        NULL
      }
    } else {
      BasalCrown <- MeanCrown <- NULL
    }

    if (height != "") {
      Height <- mean(dat[[height]])
      LoreyHeight <- ifelse(DBH == "", NULL,
        loreyHeight(
          basal = basal_i(dbh = dat[[DBH]]),
          height = dat[[height]]
        )
      )
    } else {
      Height <- LoreyHeight <- NULL
    }

    Density <- if (!is.null(a)) {
      densityTree(number = nbTrees, area = a)
    } else {
      NULL
    }

    res <- c(
      MeanDBH = MeanDBH, Basal = Basal, MeanCrown = MeanCrown,
      BasalCrown = BasalCrown, Height = Height, LoreyHeight = LoreyHeight,
      Density = Density, Area = a, nbTrees = nrow(dat)
    )
    return(res)

    # Note: The following commented code is an alternative structure for the return value
    # c(
    #   MeanDBH = if (DBH != "") diameterMean(dbh = dat[[DBH]]) else NULL,
    #   Basal = if (DBH != "" && !is.null(a)) basal(dbh = dat[[DBH]], area = a, k = k) else NULL,
    #   MeanCrown = if (crown != "") diameterMean(dbh = dat[[crown]]) else NULL,
    #   BasalCrown = if (crown != "" && !is.null(a)) basal(dbh = dat[[crown]], area = a, k = kCrown) else NULL,
    #   Height = if (height != "") mean(dat[[height]]) else NULL,
    #   LoreyHeight = if (height != "" && DBH != "") loreyHeight(basal = basal_i(dbh = dat[[DBH]]), height = dat[[height]]) else NULL,
    #   Density = if (!is.null(a)) densityTree(number = nrow(dat), area = a) else NULL
    # )
  }

  # Function to handle parameter calculation with plot grouping
  param_calc <- function(data, plot, DBH, height, crown, area, k, kCrown) {
    # Data validation
    namesData <- names(data)
    vars <- c(DBH, height, crown)
    missing <- !(vars %in% c(namesData, ""))
    if (any(missing)) {
      warning(
        "/nThese variables are not found in data: ",
        paste(vars[missing], collapse = ", ")
      )

      # Reset missing variables to empty string
      for (nmi in c("DBH", "height", "crown")[missing]) {
        assign(nmi, "")
      }

      # Alternative
      # if (DBH != "" && !DBH %in% namesData) DBH <- ""
      # if (height != "" && !height %in% namesData) height <- ""
      # if (crown != "" && !crown %in% namesData) crown <- ""
    }

    # Apply calculation based on plot grouping
    if (plot == "") {
      areaVal <- if (is.character(area) && area %in% namesData) {
        data[[area]][1]
      } else {
        area
      }
      calc(data, areaVal)
    } else {
      plotData <- makedata(data, plot)
      if (is.numeric(area) || is.null(area)) {
        sapply(plotData, function(d) calc(d, area), simplify = TRUE)
      } else if (is.character(area) && area %in% namesData) {
        sapply(plotData, function(d) calc(d, d[[area]][1]), simplify = TRUE)
      } else {
        stop("'area' should be a numeric or a character specifying area column name.")
      }
    }
  }

  # Use nestedFunBuilder to apply the parameter calculation function
  nestedFunBuilder(
    data = data, ..., .fun = param_calc,
    .funArgs = list(
      plot = plot, DBH = DBH, height = height,
      crown = crown, area = area, k = k, kCrown = kCrown
    )
  )
}

# Stand parameter ####

#' Density of regeneration (efficient version)
#'
#' @description
#' Computes the density per plot of tree regeneration based on counts in subplots.
#' Can be grouped by additional factors for nested analysis.
#'
#' @param data an optional data frame, list, tibble or object coercible by
#' \code{\link{as.data.frame}} to a data frame containing the variables whose
#' names are given in \code{count} and \code{plot}.
#'
#' @param plot an optional character, name of the variable containing the plot
#' identities. If \code{data} is missing, a vector providing the plot identities.
#'
#' @param count character, name of the variable containing the counts: number
#' of stems (individuals). If \code{data} is missing, a numeric vector
#' providing the the counts: number of stems (individuals).
#'
#' @param nbSubPlot numeric, number of subplots per plot.
#' @param area numeric, area of each subplot.
#' @param ... additional factor variables for grouping (e.g., species, site, treatment)
#'
#' @export
densityRegen <- function(data = NULL, plot = NULL, count, nbSubPlot, area, ...) {
  # Input validation
  if (!is.numeric(area)) {
    stop("'area' must be numeric.\n")
  }
  if (!is.numeric(nbSubPlot)) {
    stop("'nbSubPlot' must be numeric.\n")
  }

  # Handle the case where data is not provided
  if (is.null(data)) {
    if (!is.numeric(count)) {
      stop("'count' must be numeric vector if 'data' is not provided.\n")
    }

    # If no plot grouping, calculate density for entire dataset
    if (is.null(plot)) {
      return(sum(count) / (nbSubPlot * area))
    }

    if (length(count) != length(plot)) {
      stop("'count' and 'plot' must be vectors of same length.\n")
    }
    data <- data.frame(count = count, plot = plot)
    plot <- "plot"
    count <- "count"
  }

  # If data is provided, ensure it is a data frame and validate inputs
  else {
    # Data is provided - validate inputs
    data <- droplevels(as.data.frame(data))
    if (!is.character(count)) {
      stop("'count' must be character if 'data' is provided.\n")
    }
    if (!is.null(plot)) {
      if (!is.character(plot)) {
        stop("'plot' must be character if 'data' is provided.\n")
      }
      if (!(plot %in% names(data))) {
        stop("'plot' column not found in data.\n")
      }
    }

    # Enhanced validation
    if (!(count %in% names(data))) {
      stop("'count' column not found in data.\n")
    }
    if (!is.numeric(data[[count]])) {
      stop("'count' column must be numeric.\n")
    }
  }

  calc <- function(data, plot, count, nbSubPlot, area) {
    # If no plot grouping, calculate density for entire dataset
    if (is.null(plot)) {
      return(sum(data[[count]]) / (nbSubPlot * area))
    }

    # Group by plot and calculate density for each plot
    PlotData <- makedata(data, plot)
    sapply(PlotData, function(plotData) {
      sum(plotData[[count]]) / (nbSubPlot * area)
    })
  }

  # Use nestedFunBuilder with inline calculation function
  res <- nestedFunBuilder(
    data = data, ..., .fun = calc,
    .funArgs = list(
      plot = plot, count = count,
      nbSubPlot = nbSubPlot, area = area
    )
  )

  return(res)
}




#' Tree density
#' @description Density of trees per plot.
#'
#' @param number numeric, vector of tree count in each plot.
#' @param area numeric, area of a plot.
#' @param overall logical, if \code{TRUE}, an overall mean density is computed,
#' otherwise density is computed for each plot. Default is \code{TRUE}.
#'
#' @return Vector of density.
#'
#' @details If every plot have same area, \code{area} is a numeric value,
#' otherwise \code{area} is a vector of each plot area.
#'
#' @examples
#' count <- setNames(
#'   c(87, 104, 83, 132, 107, 84, 110, 115, 112, 94),
#'   LETTERS[1:10]
#' )
#' densityTree(count, 10)
#' densityTree(count, area = 10, overall = FALSE)
#' densityTree(count, area = 10:19, overall = FALSE)
#'
#' @seealso \code{\link{densityRegen}} for regeneration density.
#' @export
#'
densityTree <- function(number, area, overall = TRUE) {
  if (!is.numeric(area)) {
    stop("'area' must be numeric. \n")
  }

  if (is.numeric(number)) {
    if (overall) {
      res <- mean(number / area)
    } else {
      res <- number / area
    }
  } else {
    stop("'number' must be numeric. \n")
  }

  return(res)
}


#' Mean diameter
#' @description Mean diameter of a forestry stand.
#'
#' @param dbh numeric, vector of diameter.
#'
#' @return Mean diameter.
#'
#' @seealso \code{\link{dbh}}, \code{\link{basal_i}}
#'
#' @examples
#' set.seed(1)
#' diameter <- rnorm(10, 100, 20)
#' diameterMean(dbh = diameter)
#' @export
diameterMean <- function(dbh) {
  if (is.numeric(dbh)) {
    return(sqrt(mean(dbh**2)))
  } else {
    stop("'dbh' must be numeric. \n")
  }
}


#' The basal area of plots
#' @description Computes the basal area of tree stems in a plot. The basal area
#' is the cross sectional area of the bole or stem of a tree at breast height.
#'
#' @param dbh numeric, vector of diameter.
#' @param circum numeric, vector of circumference. Is used only if \code{dbh}
#' is not given.
#' @param area numeric, area of the plot (see \code{details} for unit).
#' @param k numeric, used to convert diameter unit. Default is \code{100}
#' (coverts from cm to m. See \code{details}).
#'
#' @return A vector of basal area of stands.
#'
#' @details If \code{area} is expressed in ha and \code{dbh} expressed in cm,
#' the basal area unit is cm\\u00b2/ha when \code{k = 1}.
#' In order to convert centimeter (cm) to meter (m) for \code{dbh}, set
#' \code{k = 100}. Because 1m = 100 cm. Then, basal area unit will be
#' \code{m\\u00b2/ha}.
#'
#' If \code{dbh} is in meter (m), and \code{area} in in hectare (ha), setting
#' \code{k = 1} returns basal area in m\\u00b2/ha.
#'
#' If \code{dbh} is in feet, and \code{area} in acre, setting \code{k = 1}
#'  returns basal area in ft\\u00b2/ac.
#'
#' If \code{dbh} is in inch, and \code{area} in acre, setting
#' \code{k = 12} returns basal area in feet\\u00b2/acres (ft\\u00b2/ac).
#' @export
basal <- function(dbh, area, k = 100, circum = NULL) {
  sum(basal_i(dbh = dbh, circum = circum)) / (area * k**2)
}


#' Basal area contribution
#' @description The basal area contribution (in per cent) is defined as the part
#' of a given species trees in the overall basal area of all trees in an area.
#'
#' @param basal numeric, basal area per species.
#'
#' @export
basalContribution <- function(basal) {
  rfreq(x = basal)
}



#' Lorey's mean height
#' @description The average height of the trees in a plot, weighted by their
#' basal area.
#'
#' @param basal numeric, vector of trees' individual basal area.
#' @param height numeric, vector of trees' individual height.
#'
#' @return Average Lorey height of a stand.
#'
#' @examples
#' set.seed(1)
#' donnee <- data.frame(
#'   hauteur = rnorm(10, 12, 3),
#'   area = basal_i(rnorm(10, 100, 20))
#' )
#' loreyHeight(basal = donnee$area, height = donnee$hauteur)
#'
#' @seealso \code{\link{height}}, \code{\link{basal_i}}
#' @export
loreyHeight <- function(basal, height) {
  sum(basal * height) / sum(basal)
}

#' Index of Blackman
#' @param density numeric, vector of the density.
#' @return Index of Blackman.
#' @export
#'
blackman <- function(density) {
  if (!is.numeric(density)) {
    stop("'density' must be numeric. \n")
  } else {
    var(density) / mean(density)
  }
}


#' Index of Green
#' @param density numeric, vector of the density.
#' @return Index of Green.
#' @export
#'
green <- function(density) {
  if (!is.numeric(density)) {
    stop("'density' must be numeric. \n")
  } else {
    (blackman(density) - 1) / (length(density) - 1)
  }
}



#' Fit and Plot Three-Parameter Weibull Distribution with Confidence Bands
#'
#' @description
#' Fits a three-parameter Weibull distribution to diameter data and optionally
#' visualizes the fit with a histogram, fitted density curve, and confidence bands.
#'
#' @param x numeric vector of diameter observations (typically tree diameters in cm).
#' @param amplitude numeric bin width for histogram (default: 10).
#' @param shape numeric initial value for shape parameter (default: 2).
#' @param plot logical; if TRUE, produces a histogram with fitted curve (default: TRUE).
#' @param show_ci logical; if TRUE, displays confidence bands around fitted curve
#'    (default: TRUE). Only applicable when plot = TRUE and standard errors are available.
#' @param ci_level numeric confidence level for bands (default: 0.95).
#' @param ci_method character string for CI computation: "delta" (delta method using vcov),
#'   "bootstrap" (parametric bootstrap), or "both" (default: "delta").
#' @param n_boot integer number of bootstrap samples (default: 1000). Only used if
#'   ci_method is "bootstrap" or "both".
#' @param main character string for plot title (default: NULL).
#' @param title.col color for legend title (default: "black").
#' @param mid logical; if TRUE, curve is drawn from min to max of bin midpoints;
#'   if FALSE, from min to max of bin breaks (default: TRUE).
#' @param line.col color for fitted curve (default: "blue").
#' @param ci.col color for confidence bands (default: "lightblue").
#' @param ci.alpha numeric transparency for confidence bands (default: 0.3).
#' @param legendPos position of legend (default: "topright").
#' @param lowLim numeric lower limit for histogram breaks (default: NULL, uses min(x)).
#' @param ymax numeric upper limit for y-axis (default: NULL, auto-computed).
#' @param bg background color for legend box (default: "aliceblue").
#' @param method character string specifying estimation method. Options:
#'   "mle" (Maximum Likelihood - default), "mps" (Maximum Product Spacing),
#'   "mom" (Method of Moments).
#'
#' @param ties_method character string for ties correction in MPS: "cheng_amin", "none",
#'   or "cheng_stephens" (default: "cheng_amin"). Only used when method = "mps".
#' @param cex.axis numeric character expansion factor for axis annotation (default: 0.6).
#' @param cex.lab numeric character expansion factor for axis labels (default: 0.8).
#' @param las numeric orientation of axis labels (default: 1).
#' @param xlab character string for x-axis label (default: "Diameter class (cm)").
#' @param ylab character string for y-axis label (default: "Relative frequency (\%)" ).
#' @param cex.legend numeric character expansion factor for legend (default: 1).
#' @param ... additional graphical parameters passed to \code{\link{plot}}.
#'
#'
#' @return
#' A list (invisibly) containing:
#' \item{estimate}{Named vector of parameter estimates (shape, scale, location)}
#' \item{se}{Standard errors (NULL if unavailable or method doesn't support it)}
#' \item{vcov}{Variance-covariance matrix (NULL if unavailable)}
#' \item{ci}{Confidence intervals for parameters at specified level (NULL if unavailable)}
#' \item{measures}{Named vector of goodness-of-fit statistics (KS statistic, p-value, AIC, BIC)}
#' \item{convergence}{Integer convergence code (0 = successful)}
#' \item{method}{Character string of estimation method used}
#' \item{ci_method}{Character string of confidence interval method used}
#' \item{ci_level}{Numeric confidence level used}
#' \item{note}{Character string with interpretation note for KS test}
#'
#'
#' @import graphics grDevices
#'
#' @examples
#' # Simulate tree diameter data
#' set.seed(123)
#' diameters <- rweibull3(100, shape = 2.5, scale = 25, loc = 10)
#'
#' # Fit with confidence bands
#' fit1 <- fit_weibull_plot(diameters, amplitude = 5, show_ci = TRUE)
#'
#' # Fit without confidence bands
#' fit2 <- fit_weibull_plot(diameters, amplitude = 5, show_ci = FALSE)
#'
#' # Use bootstrap confidence bands
#' fit3 <- fit_weibull_plot(diameters,
#'   amplitude = 5,
#'   ci_method = "bootstrap", n_boot = 500
#' )
#' @seealso \code{\link{fit_dist}}
#'
#' @export
fit_weibull_plot <- function(x, amplitude = 10, shape = 2, plot = TRUE,
                             show_ci = TRUE, ci_level = 0.95,
                             ci_method = "delta", n_boot = 1000,
                             main = NULL, title.col = "black",
                             mid = TRUE, line.col = "blue",
                             ci.col = "lightblue", ci.alpha = 0.3,
                             legendPos = "topright", lowLim = NULL, ymax = NULL,
                             bg = "aliceblue", method = "mle",
                             ties_method = "cheng_amin",
                             cex.axis = 0.6, cex.lab = 0.8, las = 1,
                             xlab = "Diameter class (cm)",
                             ylab = "Relative frequency (%)",
                             cex.legend = 1, ...) {
  # Input validation
  if (!is.numeric(x) || length(x) < 3) {
    stop("'x' must be a numeric vector with at least 3 observations")
  }

  if (!ci_method %in% c("delta", "bootstrap", "both")) {
    stop("'ci_method' must be one of: 'delta', 'bootstrap', 'both'")
  }

  if (any(is.na(x))) {
    warning("NA values removed from data")
    x <- x[!is.na(x)]
  }

  # Data range
  MIN <- min(x)
  MAX <- max(x)
  n <- length(x)

  # Compute initial values based on empirical quantiles
  brk <- hist(x, breaks = seq(MIN, MAX + 1, 1), plot = FALSE)
  scale_init <- which(cumsum(brk$density) >= 0.632)[1]
  scale_init <- brk$mids[scale_init] - MIN

  # Set up starting values
  initials <- c(shape = shape, scale = scale_init, loc = MIN)

  # Fit three-parameter Weibull distribution using fit_dist
  fit_result <- fit_dist(
    data = x,
    dist = "weibull3",
    method = method,
    start = initials,
    ties_method = ties_method
  )

  # Extract parameter estimates
  shape_est <- fit_result$estimate["shape"]
  scale_est <- fit_result$estimate["scale"]
  loc_est <- fit_result$estimate["loc"]

  # Build measures vector
  measures <- c(
    loglik = fit_result$loglik,
    AIC = fit_result$aic,
    BIC = fit_result$bic,
    KS = fit_result$ks_statistic,
    KS_p.value = fit_result$ks_pvalue
  )

  # Add standard errors if available
  if (!is.null(fit_result$se)) {
    measures <- c(
      measures,
      SE_shape = fit_result$se["shape"],
      SE_scale = fit_result$se["scale"],
      SE_loc = fit_result$se["loc"]
    )
  }

  # Prepare confidence intervals if available
  ci <- NULL
  if (!is.null(fit_result$se)) {
    ci <- confint(fit_result, level = ci_level)
  }

  # Compute confidence bands for the curve if requested
  ci_bands <- NULL
  if (show_ci && !is.null(fit_result$vcov) && ci_method %in% c("delta", "both")) {
    ci_bands <- ci_bands_delta(
      params = fit_result$estimate,
      vcov = fit_result$vcov,
      x_range = c(MIN, MAX),
      amplitude = amplitude,
      level = ci_level
    )
  }

  # Bootstrap confidence bands if requested
  ci_bands_boot <- NULL
  if (show_ci && ci_method %in% c("bootstrap", "both")) {
    ci_bands_boot <- ci_bands_bootstrap(
      x = x,
      params = fit_result$estimate,
      method = method,
      ties_method = ties_method,
      x_range = c(MIN, MAX),
      amplitude = amplitude,
      n_boot = n_boot,
      level = ci_level
    )
  }

  # Prepare output
  result <- list(
    estimate = fit_result$estimate,
    se = fit_result$se,
    vcov = fit_result$vcov,
    ci = ci,
    ci_bands = ci_bands,
    ci_bands_boot = ci_bands_boot,
    measures = measures,
    convergence = fit_result$convergence,
    method = method,
    ci_method = ci_method,
    ci_level = ci_level,
    note = "KS alternative hypothesis: data do not follow the specified Weibull distribution."
  )

  # Add convergence warning if needed
  if (!is.null(fit_result$convergence) && fit_result$convergence != 0) {
    warning(
      "Optimization did not converge successfully (code: ",
      fit_result$convergence, ")"
    )
  }

  # Plotting
  if (plot) {
    # Define fitted density function (scaled to percentage)
    f <- function(x_val, ampl = amplitude) {
      100 * ampl * dweibull3(x_val,
        shape = shape_est, scale = scale_est,
        loc = loc_est, log = FALSE
      )
    }

    # Set histogram breaks
    lowLim <- if (is.null(lowLim)) MIN else lowLim
    brk <- hist(x, breaks = seq(lowLim, MAX + amplitude, amplitude), plot = FALSE)
    brk$density <- rfreq(brk$density)

    # Set y-axis limit (accounting for CI bands if present)
    if (is.null(ymax)) {
      ymax_hist <- max(brk$density) + diff(range(brk$density)) * 0.2
      ymax_ci <- if (!is.null(ci_bands)) max(ci_bands$upper, na.rm = TRUE) else 0
      ymax_ci_boot <- if (!is.null(ci_bands_boot)) max(ci_bands_boot$upper, na.rm = TRUE) else 0
      ymax <- min(max(ymax_hist, ymax_ci, ymax_ci_boot), 100)
    }

    # Create plot
    minb <- min(brk$breaks)
    maxb <- max(brk$breaks)
    plot(brk,
      freq = FALSE, xlim = c(minb, maxb), ylim = c(0, ymax),
      las = las, xlab = xlab, ylab = ylab,
      cex.axis = cex.axis, cex.lab = cex.lab,
      main = main, ...
    )

    # Add confidence bands if available
    if (show_ci) {
      # Delta method bands
      if (!is.null(ci_bands)) {
        polygon(c(ci_bands$x, rev(ci_bands$x)),
          c(ci_bands$lower, rev(ci_bands$upper)),
          col = adjustcolor(ci.col, alpha.f = ci.alpha),
          border = NA
        )
      }

      # Bootstrap bands (dotted if both methods shown)
      if (!is.null(ci_bands_boot)) {
        if (!is.null(ci_bands)) {
          # Show bootstrap as dotted lines if delta also shown
          lines(ci_bands_boot$x, ci_bands_boot$lower,
            col = line.col, lty = 3, lwd = 1
          )
          lines(ci_bands_boot$x, ci_bands_boot$upper,
            col = line.col, lty = 3, lwd = 1
          )
        } else {
          # Show bootstrap as shaded region if it's the only one
          polygon(c(ci_bands_boot$x, rev(ci_bands_boot$x)),
            c(ci_bands_boot$lower, rev(ci_bands_boot$upper)),
            col = adjustcolor(ci.col, alpha.f = ci.alpha),
            border = NA
          )
        }
      }
    }

    # Add fitted curve
    curve_from <- if (mid) min(brk$mids) else min(brk$breaks)
    curve_to <- if (mid) max(brk$mids) else max(brk$breaks)
    curve(f, col = line.col, add = TRUE, from = curve_from, to = curve_to, lwd = 2)

    # Build legend with parameter estimates
    if (!is.null(ci)) {
      # Show parameters with confidence intervals
      legend_text <- c(
        "Weibull fit",
        bquote(alpha == .(round(shape_est, 2)) ~ .(format_ci_inline(ci["shape", ]))),
        bquote(beta == .(round(scale_est, 2)) ~ .(format_ci_inline(ci["scale", ]))),
        bquote(theta == .(round(loc_est, 2)) ~ .(format_ci_inline(ci["loc", ])))
      )
    } else {
      # Show parameters without confidence intervals
      legend_text <- c(
        "Weibull fit",
        bquote(alpha == .(round(shape_est, 2))),
        bquote(beta == .(round(scale_est, 2))),
        bquote(theta == .(round(loc_est, 2)))
      )
    }

    legend_lty <- c(1, rep(0, 3))
    legend_lwd <- c(2, rep(0, 3))
    legend_col <- c(line.col, rep(NA, 3))

    # Add CI info to legend
    if (show_ci && (!is.null(ci_bands) || !is.null(ci_bands_boot))) {
      ci_pct <- paste0(round(ci_level * 100), "%")

      if (ci_method == "delta" && !is.null(ci_bands)) {
        legend_text <- c(legend_text, "", paste0(ci_pct, " CI"))
        legend_lty <- c(legend_lty, 0, 0)
        legend_lwd <- c(legend_lwd, 0, 8)
        legend_col <- c(legend_col, NA, adjustcolor(ci.col, alpha.f = ci.alpha))
      } else if (ci_method == "bootstrap" && !is.null(ci_bands_boot)) {
        legend_text <- c(legend_text, "", paste0(ci_pct, " CI (bootstrap)"))
        legend_lty <- c(legend_lty, 0, 0)
        legend_lwd <- c(legend_lwd, 0, 8)
        legend_col <- c(legend_col, NA, adjustcolor(ci.col, alpha.f = ci.alpha))
      } else if (ci_method == "both") {
        legend_text <- c(
          legend_text, "",
          paste0(ci_pct, " CI (delta)"),
          paste0(ci_pct, " CI (bootstrap)")
        )
        legend_lty <- c(legend_lty, 0, 0, 3)
        legend_lwd <- c(legend_lwd, 0, 8, 1)
        legend_col <- c(legend_col, NA, adjustcolor(ci.col, alpha.f = ci.alpha), line.col)
      }
    }

    # Add goodness-of-fit (KS test only, no AIC)
    legend_text <- c(
      legend_text, "",
      bquote(KS[p] == .(round(fit_result$ks_pvalue, 3)))
    )
    legend_lty <- c(legend_lty, rep(0, 2))
    legend_lwd <- c(legend_lwd, rep(0, 2))
    legend_col <- c(legend_col, rep(NA, 2))

    legend(legendPos,
      title.col = title.col,
      lty = legend_lty,
      lwd = legend_lwd,
      col = legend_col,
      box.lty = 0,
      bty = "o",
      bg = bg,
      cex = cex.legend,
      legend = legend_text
    )
  }

  return(invisible(result))
}


# Create alias for backward compatibility
#' @rdname fit_weibull_plot
#' @export
adjWeibull <- fit_weibull_plot

# END ####


#' Abbreviates a Botanical or Zoological Latin Name into an Eight-character
#' from 'Gender epithet' to 'G. epithet'
#' @description To abbreviate  species name from 'Gender epithet' to
#' 'G. epithet'.
#' Useful in plots with species names.
#' @param name a factor coercible vector of species name in forms
#' 'Gender epithet'.
#' @param sep character string which separates Gender and epithet.
#' Default is space " ".
#' @details Returned reduced names are made unique.
#' @return A factor vector of species reduced names in forms 'G. epithet'.
#' @seealso \code{\link[vegan]{make.cepnames}} in \code{vegan} package.
#' @export
spNmReduce <- function(name, sep = " ") {
  name <- as.factor(name)
  nm <- strsplit(levels(name), sep)
  nm <- sapply(nm, function(sp) {
    paste(substr(sp[1], 1, 1), ". ", paste(sp[-1], collapse = " "), sep = "")
  })
  levels(name) <- make.unique(nm)
  return(name)
}
