% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fingerprint.R
\name{fingerprint}
\alias{fingerprint}
\title{Optimal Fingerprinting via total least square regression.}
\usage{
fingerprint(
  Xtilde,
  Y,
  mruns,
  ctlruns.sigma,
  ctlruns.bhvar,
  S,
  T,
  B = 0,
  Proj = diag(ncol(Xtilde)),
  method = c("EE", "PBC", "TS", "TS.TempSt"),
  cov.method = c("l2", "mv"),
  conf.level = 0.9,
  missing = FALSE,
  cal.a = TRUE,
  ridge = 0
)
}
\arguments{
\item{Xtilde}{\eqn{n \times p} matrix, signal pattern to be detected.}

\item{Y}{\eqn{n \times 1} matrix, length \eqn{S \times T}, observed climate variable.}

\item{mruns}{number of ensembles to estimate the corresponding pattern. 
It is used as the scale of the covariance matrix for \eqn{X_i}.}

\item{ctlruns.sigma}{\eqn{m \times n} matrix, a group of \eqn{m} independent control 
runs for estimating covariance matrix, which is used in point estimation of 
the signal factors.}

\item{ctlruns.bhvar}{\eqn{m \times n} matrix, another group of \eqn{m} independent control 
runs for estimating the corresponding confidence interval of the signal factors, 
in EE or PBC approach should be same as ctlruns.sigma.}

\item{S}{number of locations for the observed responses.}

\item{T}{number of time steps for the observed responses.}

\item{B}{number of replicates in bootstrap procedure, mainly for the PBC and TS methods, can be 
specified in "EE" method but not necessary. By default B = 0 as the default method is "EE".}

\item{Proj}{The projection matrix for computing for scaling factors of other external forcings 
with the current input when using EE. For example, when ALL and NAT are used for modeling, 
specifying the Proj matrix to return the results for ANT and NAT.}

\item{method}{for estimating the scaling factors and corresponding confidence interval}

\item{cov.method}{method for estimation of covariance matrix in confidence 
interval estimation of PBC method. (only for PBC method).}

\item{conf.level}{confidence level for confidence interval estimation.}

\item{missing}{indicator for whether missing values present in Y.}

\item{cal.a}{indicator for calculating the a value, otherwise use default value a = 1. (only for EE method)}

\item{ridge}{shrinkage value for adjusting the method for missing observations if missing = TRUE. (only for EE method)}
}
\value{
a list of the fitted model including point estimate and
interval estimate of coefficients and corresponding estimate of standard error.
}
\description{
This function estimates the signal factors and corresponding confidence 
interval via the estimating equation or total least squares.
}
\examples{
## load the example dataset
data(simDat)
Cov <- simDat$Cov[[1]]
ANT <- simDat$X[, 1]
NAT <- simDat$X[, 2]

## generate the simulated data set
## generate regression observation
Y <- MASS::mvrnorm(n = 1, mu = ANT + NAT, Sigma = Cov)
## generate the forcing responses
mruns <- c(1, 1)
Xtilde <- cbind(MASS::mvrnorm(n = 1, mu = ANT, Sigma = Cov / mruns[1]),
               MASS::mvrnorm(n = 1, mu = NAT, Sigma = Cov / mruns[2]))
## control runs
ctlruns <- MASS::mvrnorm(100, mu = rep(0, nrow(Cov)), Sigma = Cov)
## ctlruns.sigma for the point estimation and ctlruns.bhvar for the interval estimation
ctlruns.sigma <- ctlruns.bhvar <- ctlruns
## number of locations
S <- 25
## number of year steps
T <- 10

## call the function to estimate the signal factors via EE
fingerprint(Xtilde, Y, mruns,
          ctlruns.sigma, ctlruns.bhvar,
          S, T,
          ## B = 0, by default
          method = "EE",
          conf.level = 0.9,
          cal.a = TRUE,
          missing = FALSE, ridge = 0)
}
\references{
\itemize{ 
\item  Gleser (1981), Estimation in a Multivariate "Errors in Variables" 
Regression Model: Large Sample Results, \emph{Ann. Stat.} 9(1) 24--44.
\item Golub and Laon (1980), An Analysis of the Total Least Squares Problem,
\emph{SIAM J. Numer. Anal}. 17(6) 883--893.
\item Pesta (2012), Total least squares and bootstrapping with 
applications in calibration, \emph{Statistics} 47(5), 966--991.
\item Li et al (2021), Uncertainty in Optimal Fingerprinting is Underestimated, 
      \emph{Environ. Res. Lett.} 16(8) 084043.
\item Sai et al (2023), Optimal Fingerprinting with Estimating Equations,
      \emph{Journal of Climate} 36(20), 7109–-7122.
\item Li et al (2025), Improved Optimal Fingerprinting Based on Estimating 
      Equations Reaffirms Anthropogenic Effect on Global Warming,
      \emph{Journal of Climate}. 38(8), 1779–-1790.
}
}
\author{
Yan Li
}
\keyword{equation}
\keyword{estimating}
\keyword{fingerprinting}
\keyword{regression}
\keyword{tls}
