% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter_ard_hierarchical.R
\name{filter_ard_hierarchical}
\alias{filter_ard_hierarchical}
\title{Filter Stacked Hierarchical ARDs}
\usage{
filter_ard_hierarchical(
  x,
  filter,
  var = NULL,
  keep_empty = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{x}{(\code{card})\cr
a stacked hierarchical ARD of class \code{'card'} created using \code{\link[=ard_stack_hierarchical]{ard_stack_hierarchical()}} or
\code{\link[=ard_stack_hierarchical_count]{ard_stack_hierarchical_count()}}.}

\item{filter}{(\code{expression})\cr
an expression that is used to filter variable groups of the hierarchical ARD. See the
Details section below.}

\item{var}{(\code{\link[dplyr:dplyr_tidy_select]{tidy-select}})\cr
hierarchy variable from \code{x} to perform filtering on. If \code{NULL}, the last hierarchy variable
from \code{x} (\code{dplyr::last(attributes(x)$args$variables)}) will be used.}

\item{keep_empty}{(scalar \code{logical})\cr
Logical argument indicating whether to retain summary rows corresponding to hierarchy
sections that have had all rows filtered out. Default is \code{FALSE}.}

\item{quiet}{(\code{logical})\cr
logical indicating whether to suppress any messaging. Default is \code{FALSE}.}
}
\value{
an ARD data frame of class 'card'
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}\cr

This function is used to filter stacked hierarchical ARDs.

For the purposes of this function, we define a "variable group" as a combination of ARD rows
grouped by the combination of all their variable levels, but excluding any \code{by} variables.
}
\details{
The \code{filter} argument can be used to filter out variable groups of a hierarchical
ARD which do not meet the requirements provided as an expression.
Variable groups can be filtered on the values of any of the possible
statistics (\code{n}, \code{p}, and \code{N}) provided they are included at least once
in the ARD, as well as the values of any \code{by} variables.

Additionally, filters can be applied on individual levels of the \code{by} variable via the
\code{n_XX}, \code{N_XX}, and \code{p_XX} statistics, where each \code{XX} represents the index of the \code{by}
variable level to select the statistic from. For example, \code{filter = n_1 > 5} will check
whether \code{n} values for the first level of \code{by} are greater than 5 in each row group.

Overall statistics for each row group can be used in filters via the \code{n_overall}, \code{N_overall},
and \code{p_overall} statistics. If the ARD is created with parameter \code{overall=TRUE}, then these
overall statistics will be extracted directly from the ARD, otherwise the statistics will be
derived where possible. If \code{overall=FALSE}, then \code{n_overall} can only be derived if the \code{n}
statistic is present in the ARD for the filter variable, \code{N_overall} if the \code{N} statistic is
present for the filter variable, and \code{p_overall} if both the \code{n} and \code{N} statistics are
present for the filter variable.

By default, filters will be applied at the level of the innermost hierarchy variable, i.e.
the last variable supplied to \code{variables}. If filters should instead be applied at the level
of one of the outer hierarchy variables, the \code{var} parameter can be used to select a different
variable to filter on. When \code{var} is set to a different (outer) variable and a level of the
variable does not meet the filtering criteria then the section corresponding to that variable
level and all sub-sections within that section will be removed.

To illustrate how the function works, consider the typical example below
where the AE summaries are provided by treatment group.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{ADAE |>
  dplyr::filter(AESOC == "GASTROINTESTINAL DISORDERS",
                AEDECOD \%in\% c("VOMITING", "DIARRHOEA")) |>
  ard_stack_hierarchical(
    variables = c(AESOC, AEDECOD),
    by = TRTA,
    denominator = ADSL,
    id = USUBJID
  )
}\if{html}{\out{</div>}}\tabular{lrrr}{
   \strong{SOC} / AE \tab Placebo \tab Xanomeline High Dose \tab Xanomeline Low Dose \cr
   \strong{GASTROINTESTINAL DISORDERS} \tab 11 (13\%) \tab 10 (12\%) \tab 8 (9.5\%) \cr
   DIARRHOEA \tab 9 (10\%) \tab 4 (4.8\%) \tab 5 (6.0\%) \cr
   VOMITING \tab 3 (3.5\%) \tab 7 (8.3\%) \tab 3 (3.6\%) \cr
}


Filters are applied to the summary statistics of the innermost variable in the hierarchy by
default---\code{AEDECOD} in this case. If we wanted to filter based on SOC rates instead of AE
rates we could specify \code{var = AESOC} instead.
If any of the summary statistics meet the filter requirement for any of the treatment groups,
the entire row is retained.
For example, if \code{filter = n >= 9} were passed, the criteria would be met for DIARRHOEA
as the Placebo group observed 9 AEs and as a result the summary statistics for the other
treatment groups would be retained as well.
Conversely, no treatment groups' summary statistics satisfy the filter requirement
for VOMITING so all rows associated with this AE would be removed.

In addition to filtering on individual statistic values, filters can be applied
across the treatment groups (i.e. across all \code{by} variable values) by using
aggregate functions such as \code{sum()} and \code{mean()}. For simplicity, it is suggested to use
the \code{XX_overall} statistics in place of \code{sum(XX)} in equivalent scenarios. For example,
\code{n_overall} is equivalent to \code{sum(n)}.
A value of \code{filter = sum(n) >= 18} (or \code{filter = n_overall >= 18}) retains AEs where the sum of
the number of AEs across the treatment groups is greater than or equal to 18.

If \code{filter = n_overall >= 18} and \code{var = AESOC} then all rows corresponding to an SOC with an
overall rate less than 18 - including all AEs within that SOC - will be removed.

If \code{ard_stack_hierarchical(overall=TRUE)} was run, the overall column is \strong{not} considered in
any filtering except for \code{XX_overall} statistics, if specified.

If \code{ard_stack_hierarchical(over_variables=TRUE)} was run, any overall statistics are kept regardless
of filtering.

Some examples of possible filters:
\itemize{
\item \code{filter = n > 5}: keep AEs where one of the treatment groups observed more than 5 AEs
\item \code{filter = n == 2 & p < 0.05}: keep AEs where one of the treatment groups observed exactly 2
AEs \emph{and} one of the treatment groups observed a proportion less than 5\%
\item \code{filter = n_overall >= 4}: keep AEs where there were 4 or more AEs observed across the treatment groups
\item \code{filter = mean(n) > 4 | n > 3}: keep AEs where the mean number of AEs is 4 or more across the
treatment groups \emph{or} one of the treatment groups observed more than 3 AEs
\item \code{filter = n_2 > 2}: keep AEs where the \code{"Xanomeline High Dose"} treatment group (second \code{by} variable
level) observed more than 2 AEs
}
}
\examples{
\dontshow{if ((identical(Sys.getenv("NOT_CRAN"), "true") || identical(Sys.getenv("IN_PKGDOWN"), "true"))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# create a base AE ARD
ard <- ard_stack_hierarchical(
  ADAE,
  variables = c(AESOC, AEDECOD),
  by = TRTA,
  denominator = ADSL,
  id = USUBJID,
  overall = TRUE
)

# Example 1 ----------------------------------
# Keep AEs from TRTA groups where more than 3 AEs are observed across the group
filter_ard_hierarchical(ard, sum(n) > 3)

# Example 2 ----------------------------------
# Keep AEs where at least one level in the TRTA group has more than 3 AEs observed
filter_ard_hierarchical(ard, n > 3)

# Example 3 ----------------------------------
# Keep AEs that have an overall prevalence of greater than 5\%
filter_ard_hierarchical(ard, sum(n) / sum(N) > 0.05)

# Example 4 ----------------------------------
# Keep AEs that have a difference in prevalence of greater than 3\% between reference group with
# `TRTA = "Xanomeline High Dose"` and comparison group with `TRTA = "Xanomeline Low Dose"`
filter_ard_hierarchical(ard, abs(p_2 - p_3) > 0.03)

# Example 5 ----------------------------------
# Keep AEs from SOCs that have an overall prevalence of greater than 20\%
filter_ard_hierarchical(ard, p_overall > 0.20, var = AESOC)
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=sort_ard_hierarchical]{sort_ard_hierarchical()}}
}
