#'
#' Creates the estimation data for a component model
#'
#' @param data a data frame.
#'
#' @param cluster
#' A variable name, identifying the test (e.g., a job offer number).
#'
#' @param candid
#' A list of factor names defining the candidates (e.g., gender, origin).
#'
#' @param callback
#' A Boolean variable, equal to TRUE for non negative callbacks.
#'
#' @param model a list of string lists, defining the components of the model,
#'  in difference from the reference candidate.
#'
#' @return a list with class \code{callback_comp} containing:
#' \itemize{
#' \item\bold{aux_cand: }list of the candidates.
#' \item\bold{aux_model: }summary of the components model.
#' \item\bold{aux_boole: }Boole matrix of the components model.
#' \item\bold{aux_det: }determinant of \code{t(aux_boole)\%*\%aux_boole}.
#' \item\bold{aux_coef: }auxilliary parameters.
#' \item\bold{aux_vcov: }covariance matrix of the auxilliary parameters.
#' \item\bold{aux_cor: }correlation matrix of the auxilliary parameters.
#' }
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' data(mobility1)
#' model <- list(c("license"),c("woman"),c("woman","license","inter"))
#' callback_comp(data = mobility1, cluster = "offer",
#' candid = c("gender","licenses"), callback = "callback",
#' model = model)
#'
#' @importFrom stats cov cor
#'
#' @export
callback_comp <- function(data = NULL,
                          cluster = NULL,
                          candid =  NULL,
                          callback = NULL,
                          model = NULL) {
  # no missing callback or candidate variables
  m <-
    subset(data[, c(cluster, candid, callback)], !is.na(callback))
  for (x in candid) {
    m <- subset(m, !is.na(x))
  }
  # concatenation of candidate variables into one variable
  if (length(candid) > 1) {
    m[, "candid"] <-
      interaction(m[, candid], drop = TRUE, lex.order = TRUE)
  } else {
    m[, "candid"] <- m[, candid, drop = FALSE]
  }
  # formatted data set
  m <- m[, c(cluster, "candid", callback)]
  colnames(m)[c(1, 3)] <- c("cluster", "callback")
  # average answer to each candidate for each offer
  am <- aggregate(data = m, callback ~ candid + cluster, mean)
  #block transposition
  tam <- as.data.frame(aggregate(data = am, callback ~ cluster, t)$callback)
  rownames(tam) <- levels(am$cluster)
  l_rf <- levels(am$candid)
  colnames(tam) <- sort(l_rf)
  #proportions and their covariance matrix
  p <- colMeans(tam, na.rm = TRUE)#proportions
  vp <- cov(tam, use = "complete.obs") / nrow(tam)#covariances of the means
  rp <- cor(tam, use = "complete.obs")#correlation coefficients
  
  # reduced form in levels
  n_rf <- length(l_rf)
  addref <- function(x) {
    c("reference", x)
  }
  model_rf <- lapply(model, addref)
  model_rf <- c("reference", model_rf)
  names(model_rf) <- l_rf
  #Boole matrix of the model
  l_sf <- unique(unlist(c("reference", model)))
  n_sf <- length(l_sf)
  boole_mat_rf <- matrix(0L,
                         nrow = n_rf,
                         ncol = n_sf,
                         dimnames = list(l_rf, l_sf))
  for (rf in l_rf) {
    boole_mat_rf[rf, model_rf[[rf]]] <- 1L
  }
  boole_mat_rf <- as.matrix(boole_mat_rf)
  # column rank check
  boole_det_rf <- det(t(boole_mat_rf) %*% boole_mat_rf)
  #output
  z <- list(
    aux_cand = l_rf,
    aux_model = model_rf,
    aux_boole = boole_mat_rf,
    aux_det = boole_det_rf,
    aux_coef = p,
    aux_vcov = vp,
    aux_cor = rp
  )
  class(z) <- "callback_comp"
  return(z)
}