#' Function to optimize the full pseudo-loglikelihood and perform new forecasts
#'
#' @param nsim Integer, number of simulation replications.
#' @param n_train Integer, number of training observations.
#' @param n_test Integer, number of test observations.
#' @param copula Character, specifying the copula type: "Clayton", "Frank",
#' "Gumbel", "Joe", or "Gaussian".
#' @param init_params Numeric vector, initial parameter values for optimization.
#' @param fn Function, log-likelihood function for parameter estimation.
#' @param u1 Numeric vector (n_train), first pseudo-observation for the copula.
#' @param u2 Numeric vector (n_train), second pseudo-observation for the copula.
#' @param u3 Numeric vector (n_train), third pseudo-observation for the copula.
#' @param z1_train Numeric matrix (n_train x M), observed data for the first margin and sub-feature.
#' @param z2_train Numeric matrix (n_train x M), observed data for the second margin and sub-feature.
#' @param z3_train Numeric matrix (n_train x M), observed data for the third margin and sub-feature.
#' @param z1_test Numeric matrix (n_test x M), true future data for the first margin and sub-feature.
#' @param z2_test Numeric matrix (n_test x M), true future data for the second margin and sub-feature.
#' @param z3_test Numeric matrix (n_test x M), true future data for the third margin and sub-feature.
#' @param X_t Numeric matrix (n_train x M), risk factors for the dynamic copula parameter.
#' @param y1_test Numeric vector (n_test), true future values for the first response variable.
#' @param y2_test Numeric vector (n_test), true future values for the second response variable.
#' @param y3_test Numeric vector (n_test), true future values for the third response variable.
#' @param BSTS_1 Fitted BSTS model for the first response variable.
#' @param BSTS_2 Fitted BSTS model for the second response variable.
#' @param BSTS_3 Fitted BSTS model for the third response variable.
#'
#' @return A list containing:
#' \item{theta_simulated}{Simulated copula parameters across replications.}
#' \item{y1_simulated}{Simulated values for the first response variable.}
#' \item{y2_simulated}{Simulated values for the second response variable.}
#' \item{y3_simulated}{Simulated values for the third response variable.}
#' \item{MSE}{Mean squared error for each simulation run.}
#' \item{optim_results}{Results from the optimization process.}
#'
#' @importFrom stats cor optim rnorm dnorm qnorm predict
#' @importFrom copula claytonCopula frankCopula gumbelCopula joeCopula normalCopula rCopula
#' @import bsts
#' @export
#'
simul_fun_noGEV_3d <- function(nsim = 100,
                               n_train,
                               n_test,
                               copula,
                               init_params,
                               fn,
                               u1,
                               u2,
                               u3,
                               z1_train,
                               z2_train,
                               z3_train,
                               z1_test,
                               z2_test,
                               z3_test,
                               X_t,
                               y1_test,
                               y2_test,
                               y3_test,
                               BSTS_1,
                               BSTS_2,
                               BSTS_3){

  results <- list()

  if(copula != "Clayton" & copula != "Frank" & copula != "Gumbel" & copula != "Joe" & copula != "Gaussian"){stop("Invalid copula provided")}

  optim_results <- optim(par = init_params,
                         fn = fn,
                         u1 = u1,
                         u2 = u2,
                         u3 = u3,
                         X_t = X_t,
                         z1 = z1_train,
                         z2 = z2_train,
                         z3 = z3_train,
                         copula = copula)

  omega <- optim_results$par[1]
  alpha <- optim_results$par[2]
  gamma <- optim_results$par[3:(3+ncol(z1_train)-1)]

  if(copula == "Gaussian"){
    theta_simulated <- lapply(1:3, function(x) {
      matrix(NA, nrow = nsim, ncol = n_test)
    })
  } else{
    theta_simulated <- matrix(NA, nrow = nsim, ncol = n_test)
  }

  y1_simulated <- matrix(NA, nrow = nsim, ncol = n_test)
  y2_simulated <- matrix(NA, nrow = nsim, ncol = n_test)
  y3_simulated <- matrix(NA, nrow = nsim, ncol = n_test)

  MSE <- rep(NA, nsim)

  if(copula == "Clayton"){
    theta_simulated[,1] <- clayton.theta(mean(cor(cbind(u1, u2, u3), method = "kendall")))
  }
  else if(copula == "Frank"){
    theta_simulated[,1] <- frank.theta(mean(cor(cbind(u1, u2, u3), method = "kendall")))
  }
  else if(copula == "Gumbel"){
    theta_simulated[,1] <- GH.theta(mean(cor(cbind(u1, u2, u3), method = "kendall")))
  }
  else if(copula == "Joe"){
    theta_simulated[,1] <- joe.theta(mean(cor(cbind(u1, u2, u3), method = "kendall")))
  }
  else if(copula == "Gaussian"){
    #theta_simulated[,1] <- mean(cor(cbind(u1, u2, u3, u4), method = "pearson"))
    theta_simulated[[1]][,1] <- cor(u1, u2, method = "pearson")
    theta_simulated[[2]][,1] <- cor(u1, u3, method = "pearson")
    theta_simulated[[3]][,1] <- cor(u2, u3, method = "pearson")
  }


  for(i in 1:nsim){

    for(j in 1:ncol(z1_test)){

      for(t in 2:n_test){

        if(copula == "Clayton"){
          theta_simulated[i,t] <- abs(omega + alpha*theta_simulated[i,t-1] + sum(gamma*colMeans(rbind(z1_test[t-1,],z2_test[t-1,],z3_test[t-1,])) ) )
        }
        else if(copula == "Frank"){
          theta_simulated[i,t] <- abs(omega + alpha*theta_simulated[i,t-1] + sum(gamma*colMeans(rbind(z1_test[t-1,],z2_test[t-1,],z3_test[t-1,])) ) )
        }
        else if(copula == "Gumbel"){
          theta_simulated[i,t] <- abs(omega + alpha*theta_simulated[i,t-1] + sum(gamma*colMeans(rbind(z1_test[t-1,],z2_test[t-1,],z3_test[t-1,])) ) ) + 1
        }
        else if(copula == "Joe"){
          theta_simulated[i,t] <- abs(omega + alpha*theta_simulated[i,t-1] + sum(gamma*colMeans(rbind(z1_test[t-1,],z2_test[t-1,],z3_test[t-1,])) ) ) + 1
        }
        else if(copula == "Gaussian"){
          #theta_simulated[i,t] <- tanh(omega + alpha*theta_simulated[i,t-1] + sum(gamma*max(z1_sim[i,t-1],z2_sim[i,t-1],z3_sim[i,t-1],z4_sim[i,t-1]) ) )
          theta_simulated[[1]][i,t] <- tanh(omega + alpha*theta_simulated[[1]][i,t-1] + sum(gamma*colMeans(rbind(z1_test[t-1,],z2_test[t-1,])) ))
          theta_simulated[[2]][i,t] <- tanh(omega + alpha*theta_simulated[[2]][i,t-1] + sum(gamma*colMeans(rbind(z1_test[t-1,],z3_test[t-1,])) ))
          theta_simulated[[3]][i,t] <- tanh(omega + alpha*theta_simulated[[3]][i,t-1] + sum(gamma*colMeans(rbind(z2_test[t-1,],z3_test[t-1,])) ))
        }

      }
    }

    #u_sim <- matrix(NA, nrow = n_test, ncol = 2)
    u_sim <- matrix(NA, nrow = BSTS_1$niter, ncol = 3*n_test)

    col_index_u1 <- seq(1, 3*n_test, by = 3)
    col_index_u2 <- seq(2, 3*n_test, by = 3)
    col_index_u3 <- seq(3, 3*n_test, by = 3)


    #for (j in 1:nrow(u_sim)){
    for (j in 1:n_test){

      if(copula == "Clayton"){
        u_sim[,c(col_index_u1[j],col_index_u2[j],col_index_u3[j])] <- rCopula(BSTS_1$niter, claytonCopula(param = theta_simulated[i,j], dim = 3))
      }
      else if(copula == "Frank"){
        u_sim[,c(col_index_u1[j],col_index_u2[j],col_index_u3[j])] <- rCopula(BSTS_1$niter, frankCopula(param = theta_simulated[i,j], dim = 3))
      }
      else if(copula == "Gumbel"){
        u_sim[,c(col_index_u1[j],col_index_u2[j],col_index_u3[j])] <- rCopula(BSTS_1$niter, gumbelCopula(param = theta_simulated[i,j], dim = 3))
      }
      else if(copula == "Joe"){
        u_sim[,c(col_index_u1[j],col_index_u2[j],col_index_u3[j])] <- rCopula(BSTS_1$niter, joeCopula(param = theta_simulated[i,j], dim = 3))
      }
      else if(copula == "Gaussian"){
        cor_matrix_offdiag <- c(theta_simulated[[1]][i,j],theta_simulated[[2]][i,j],theta_simulated[[3]][i,j])
        u_sim[,c(col_index_u1[j],col_index_u2[j],col_index_u3[j])] <- rCopula(BSTS_1$niter, normalCopula(param = cor_matrix_offdiag, dim = 3, dispstr = "un"))
      }

    }

    epsilon_y1 <- matrix(NA, nrow = BSTS_1$niter, ncol = n_test)
    epsilon_y2 <- matrix(NA, nrow = BSTS_2$niter, ncol = n_test)
    epsilon_y3 <- matrix(NA, nrow = BSTS_3$niter, ncol = n_test)

    selected_columns_1 <- seq(1, 3*n_test, by = 3)
    selected_columns_2 <- seq(2, 3*n_test, by = 3)
    selected_columns_3 <- seq(3, 3*n_test, by = 3)

    for(j in 1:n_test){
      epsilon_y1[,j] <- qnorm(u_sim[,selected_columns_1[j]], mean = 0, sd = BSTS_1$sigma.obs)
      epsilon_y2[,j] <- qnorm(u_sim[,selected_columns_2[j]], mean = 0, sd = BSTS_2$sigma.obs)
      epsilon_y3[,j] <- qnorm(u_sim[,selected_columns_3[j]], mean = 0, sd = BSTS_3$sigma.obs)
    }

    predictions_y1 <- matrix(NA, nrow = BSTS_1$niter, ncol = n_test)
    predictions_y2 <- matrix(NA, nrow = BSTS_2$niter, ncol = n_test)
    predictions_y3 <- matrix(NA, nrow = BSTS_3$niter, ncol = n_test)

    predy1 <- predict(BSTS_1, horizon = 1, newdata = z1_test, burn = BSTS_1$niter*0.1)$distribution
    predy2 <- predict(BSTS_2, horizon = 1, newdata = z2_test, burn = BSTS_2$niter*0.1)$distribution
    predy3 <- predict(BSTS_3, horizon = 1, newdata = z3_test, burn = BSTS_3$niter*0.1)$distribution

    predictions_y1 <- apply(predy1, 2, function(col, sds) {
      col - rnorm(1, mean = 0, sd = sds)
    }, sds = BSTS_1$sigma.obs)

    predictions_y2 <- apply(predy2, 2, function(col, sds) {
      col - rnorm(1, mean = 0, sd = sds)
    }, sds = BSTS_2$sigma.obs)

    predictions_y3 <- apply(predy3, 2, function(col, sds) {
      col - rnorm(1, mean = 0, sd = sds)
    }, sds = BSTS_3$sigma.obs)

    y1_simulated[i,] <- (predictions_y1 + epsilon_y1[c((BSTS_1$niter*0.1+1):BSTS_1$niter),])[sample(1),]
    y2_simulated[i,] <- (predictions_y2 + epsilon_y2[c((BSTS_2$niter*0.1+1):BSTS_2$niter),])[sample(1),]
    y3_simulated[i,] <- (predictions_y3 + epsilon_y3[c((BSTS_3$niter*0.1+1):BSTS_3$niter),])[sample(1),]

    MSE[i] <- sum((y1_simulated[i,] - y1_test)^2 + (y2_simulated[i,] - y2_test)^2 + (y3_simulated[i,] - y3_test)^2)

   # print(i/nsim)

  }


  results[[1]] <- theta_simulated
  results[[2]] <- y1_simulated
  results[[3]] <- y2_simulated
  results[[4]] <- y3_simulated
  results[[5]] <- MSE
  results[[6]] <- optim_results

  return(results)

}
