#' Function responsible for initializing and running the REXoplanets shiny application.
#' @param ... Additional arguments passed to `shiny::runApp()`.
#' @param run If true, runs the application. If false, returns an app object.
#' @return App object.
#' @export
app = function(..., run = TRUE) {
  check_app_dependencies()

  ui = bslib::page_navbar(
    title = "REXoplanets",
    bslib::nav_panel(
      shiny::span(bsicons::bs_icon("house"), "Home"),
      shiny::h1("Welcome to REXoplanets"),
      shiny::p("This is the home page of the REXoplanets application.")
    ),
    bslib::nav_panel(
      shiny::span(bsicons::bs_icon("stars"), "Star Systems"),
      star_systems_ui("star_systems")
    )
  )

  server = function(input, output, session) {
    message("REXoplanets application initialized.")

    data = shiny::reactive(closest_50_exoplanets)

    star_systems_server("star_systems", data)
  }

  app_obj = shiny::shinyApp(
    ui = ui,
    server = server
  )

  if (run) {
    logger::log_info("REXoplanets application startup.")
    shiny::runApp(app_obj, ...)
  } else {
    app_obj
  }
}

#' Checks if packages required for shiny application are installed on user machine. If not,
#' asks if the user wants to install them. If yes, installs the packages using `pak`.
#' @importFrom purrr keep
#' @importFrom utils install.packages
#' @noRd
#' @keywords internal
check_app_dependencies = function() {
  APP_PACKAGES = c(
    "bsicons",
    "bslib",
    "htmltools",
    "shiny",
    "shinyjs"
  )

  missing_packages = purrr::keep(APP_PACKAGES, \(dep) !requireNamespace(dep, quietly = TRUE))

  if (length(missing_packages) > 0) {
    stop(paste0(
      "The following packages are required for shiny application, but are missing:\n",
      paste0("    - ", missing_packages, collapse = "\n"), "\n",
      "Install missing packages to run the application."
    ))
  }
}
