% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/peer.q.R
\name{qpeer.test}
\alias{qpeer.test}
\title{Specification Tests for Peer Effects Models}
\usage{
qpeer.test(
  model1,
  model2 = NULL,
  which,
  full = FALSE,
  boot = 10000,
  maxit = 1e+06,
  eps_f = 1e-09,
  eps_g = 1e-09
)
}
\arguments{
\item{model1, model2}{Objects of class \code{\link{qpeer}}, \code{\link{linpeer}}, or \code{\link{genpeer}}.}

\item{which}{A character string indicating the type of test to be implemented.
The value must be one of \code{"uniform"}, \code{"increasing"}, \code{"decreasing"}, \code{"wald"}, \code{"sargan"}, and \code{"encompassing"} (see Details).}

\item{full}{A Boolean indicating whether the parameters associated with the exogenous variables should be compared. This is used
for tests that compare two competing parameter sets.}

\item{boot}{An integer indicating the number of bootstrap replications to use for computing \code{p-values} in the \code{"increasing"} and \code{"decreasing"} tests.}

\item{maxit, eps_f, eps_g}{Control parameters for the \code{optim_lbfgs} solver used to optimize the objective function in the \code{"increasing"} and \code{"decreasing"} tests (see Kodde and Palm, 1986).
The \code{optim_lbfgs} function is provided by the \pkg{RcppNumerical} package and is based on the \code{L-BFGS} method.}
}
\value{
A list containing:
\item{test}{A vector or matrix containing the test statistics, degrees of freedom, and p-values.}
\item{lambda}{Peer effect estimates from tests based on a single model (monotonicity tests).}
\item{diff.theta}{Differences in peer effect estimates from tests based on two models (endogeneity and encompassing tests).}
\item{delta}{The estimate of \eqn{\delta} for the encompassing test.}
\item{which}{The value of \code{which} returned by the function.}
\item{boot}{The value of \code{boot} returned by the function.}
}
\description{
\code{qpeer.test} performs specification tests on peer effects models. These include monotonicity tests on quantile peer effects, as well as tests for instrument validity when an alternative set of instruments is available.
}
\details{
The monotonicity tests evaluate whether the quantile peer effects \eqn{\lambda_{\tau}} are constant, increasing, or decreasing. In this case, \code{model1} must be an object of class \code{\link{qpeer}}, and the \code{which} argument specifies the null hypothesis: \code{"uniform"}, \code{"increasing"}, or \code{"decreasing"}.
For the \code{"uniform"} test, a standard Wald test is performed. For the \code{"increasing"} and \code{"decreasing"} tests, the procedure follows Kodde and Palm (1986). \cr

The instrument validity tests assess whether a second set of instruments \eqn{Z_2} is valid, assuming that a baseline set \eqn{Z_1} is valid. In this case, both \code{model1} and \code{model2} must be objects of class \code{\link{qpeer}}, \code{\link{linpeer}}, or \code{\link{genpeer}}.
The test compares the estimates obtained using each instrument set. If \eqn{Z_2} nests \eqn{Z_1}, it is recommended to compare the overidentification statistics from both estimations (see Hayashi, 2000, Proposition 3.7).
If \eqn{Z_2} does not nest \eqn{Z_1}, the estimates themselves are compared. To compare the overidentification statistics, set the \code{which} argument to \code{"sargan"}. To compare the estimates directly, set the \code{which} argument to \code{"wald"}.\cr

Given two competing models, it is possible to test whether one is significantly worse using an encompassing test by setting \code{which} to \code{"encompassing"}. The null hypothesis is that \code{model1} is not worse.
}
\examples{
\donttest{
set.seed(123)
ngr  <- 50  # Number of subnets
nvec <- rep(30, ngr)  # Size of subnets
n    <- sum(nvec)

### Simulating Data
## Network matrix
G <- lapply(1:ngr, function(z) {
  Gz <- matrix(rbinom(nvec[z]^2, 1, 0.3), nvec[z], nvec[z])
  diag(Gz) <- 0
  # Adding isolated nodes (important for the structural model)
  niso <- sample(0:nvec[z], 1, prob = (nvec[z] + 1):1 / sum((nvec[z] + 1):1))
  if (niso > 0) {
    Gz[sample(1:nvec[z], niso), ] <- 0
  }
  Gz
})

tau <- seq(0, 1, 1/3)
X   <- cbind(rnorm(n), rpois(n, 2))
l   <- c(0.2, 0.15, 0.1, 0.2)
b   <- c(2, -0.5, 1)
eps <- rnorm(n, 0, 0.4)

## Generating `y`
y <- qpeer.sim(formula = ~ X, Glist = G, tau = tau, lambda = l, 
               beta = b, epsilon = eps)$y

### Estimation
## Computing instruments
Z1 <- qpeer.inst(formula = ~ X, Glist = G, tau = seq(0, 1, 0.1), 
                 max.distance = 2, checkrank = TRUE)$instruments
Z2 <- qpeer.inst(formula = y ~ X, Glist = G, tau = seq(0, 1, 0.1), 
                 max.distance = 2, checkrank = TRUE)$instruments

## Reduced-form model 
rest1 <- qpeer(formula = y ~ X, excluded.instruments = ~ Z1, Glist = G, tau = tau)
summary(rest1, diagnostic = TRUE)  
rest2 <- qpeer(formula = y ~ X, excluded.instruments = ~ Z1 + Z2, Glist = G, tau = tau)
summary(rest2, diagnostic = TRUE)  

qpeer.test(model1 = rest1, which = "increasing")
qpeer.test(model1 = rest1, which = "decreasing")
qpeer.test(model1 = rest1, model2 = rest2, which = "sargan")

#' A model with a mispecified tau
rest3 <- qpeer(formula = y ~ X, excluded.instruments = ~ Z1 + Z2, Glist = G, tau = c(0, 1))
summary(rest3)
#' Test is rest3 is worse than rest1
qpeer.test(model1 = rest3, model2 = rest1, which = "encompassing")
}
}
\references{
Hayashi, F. (2000). \emph{Econometrics}. Princeton University Press.

Kodde, D. A., & Palm, F. C. (1986). Wald criteria for jointly testing equality and inequality restrictions. \emph{Econometrica}, 54(5), 1243–1248.
}
