% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcee_helper_estimation.R
\name{mcee_helper_2stage_estimation}
\alias{mcee_helper_2stage_estimation}
\title{Two-stage helper for mediated causal excursion effects (MCEE)}
\usage{
mcee_helper_2stage_estimation(
  data,
  id_var,
  dp_var,
  outcome_var,
  treatment_var,
  mediator_var,
  avail_var = NULL,
  config_p,
  config_q,
  config_eta,
  config_mu,
  config_nu,
  omega_nrows,
  f_nrows
)
}
\arguments{
\item{data}{A long-format `data.frame` (one row per subject-by-decision point).}

\item{id_var}{Character scalar. Name of the subject ID column.}

\item{dp_var}{Character scalar. Name of the decision point column
(values need not be consecutive; they may vary in count across subjects).}

\item{outcome_var}{Character scalar. Name of the distal outcome column.}

\item{treatment_var}{Character scalar. Name of the binary treatment column (coded 0/1).}

\item{mediator_var}{Character scalar. Name of the mediator column.}

\item{avail_var}{Character scalar or `NULL`. Name of the availability column
(1 = available, 0 = unavailable). If `NULL`, availability is treated as all 1.}

\item{config_p}{Configuration for \eqn{p_t(a\mid H_t)} (propensity). A **list**
using one of the following schemas:
\itemize{
  \item \emph{Known constant(s)} (skips fitting):
    \code{list(known = c(...))} or arm-specific \code{known_a1}/\code{known_a0}.
  \item \emph{Model fit}: \code{list(formula = ~ rhs, method = m, ...)} where
    \code{method} is one of \code{"glm"}, \code{"gam"}, \code{"rf"},
    \code{"ranger"}, \code{"sl"}, \code{"sl.user-specified-library"}.
    Optional fields:
    \itemize{
      \item \code{family}: a GLM/GAM family. If omitted, **auto-detected** as
            \code{binomial()} for \eqn{p} and \eqn{q}, otherwise \code{gaussian()}.
      \item \code{clipping}: numeric length-2 \code{c(lo, hi)} to clip probabilities
            into \code{[lo, hi]} (useful for stability).
      \item For \code{method = "sl"}: \code{SL.library} (character vector of learners);
      if omitted, a simple default library is used: c("SL.mean", "SL.glm", "SL.gam").
    }
}}

\item{config_q}{Configuration for \eqn{q_t(a\mid H_t, M_t)}. Same schema as \code{config_p}.}

\item{config_eta}{Configuration for \eqn{\eta_t(a, H_t)} (outcome given \eqn{A,H}).
Same schema as \code{config_p}; default family auto-detected to \code{gaussian()} if omitted.}

\item{config_mu}{Configuration for \eqn{\mu_t(a, H_t, M_t)} (outcome given \eqn{A,H,M}).
Same schema as \code{config_p}; default family auto-detected to \code{gaussian()} if omitted.}

\item{config_nu}{Configuration for \eqn{\nu_t(a, H_t)} (cross-world ICE based on \eqn{\mu}).
Same schema as \code{config_p}; default family auto-detected to \code{gaussian()} if omitted.}

\item{omega_nrows}{Numeric vector of length \code{nrow(data)}. Per-row weights
\eqn{\omega(i,t) \ge 0}. Rows are aligned with \code{data} (no reordering).}

\item{f_nrows}{Numeric matrix with \code{nrow(data)} rows and \code{p} columns.
Row \code{r} contains \eqn{f(t_r)^\top} (the basis evaluated at the decision point
of row \code{r}). The same basis is used for both \eqn{\alpha} (NDEE) and \eqn{\beta} (NIEE).}
}
\value{
A list with components:
\describe{
  \item{\code{fit}}{A list with entries
    \code{alpha_hat}, \code{alpha_se}, \code{beta_hat}, \code{beta_se},
    and \code{varcov} (the \eqn{2p\times 2p} sandwich variance for \eqn{(\alpha^\top,\beta^\top)^\top}).}
  \item{\code{nuisance_models}}{A list of fitted Stage-1 objects:
    \code{p}, \code{q}, \code{eta1}, \code{eta0}, \code{mu1}, \code{mu0}, \code{nu1}, \code{nu0}.
    (For \code{known}/\code{known_a0}/\code{known_a1}, a small descriptor list is returned.)}
}
}
\description{
Fits all nuisance components (Stage 1) and then computes the MCEE parameters
(Stage 2) and their sandwich variance. This is a low-level driver used by the
high-level wrapper; it assumes `omega_nrows` and `f_nrows` are already aligned
to the rows of `data`.
}
\details{
\strong{Availability handling:}
When \code{avail_var} exists and equals 0, Stage 1 sets the working probabilities
to 1 for that row (e.g., \eqn{\hat{p}_t(1\mid H_t)=1}, \eqn{\hat{p}_t(0\mid H_t)=1}, similarly
for \eqn{\hat q_t}). This prevents division-by-zero in the estimating equations.

\strong{Auto-family rules:}
If \code{family} is omitted in a GLM/GAM config, it defaults to \code{binomial()}
for \code{config_p} and \code{config_q}, and to \code{gaussian()} for
\code{config_eta}, \code{config_mu}, and \code{config_nu}.

\strong{Learners:}
\itemize{
  \item \code{"glm"}: uses \code{stats::glm()}.
  \item \code{"gam"}: uses \code{mgcv::gam()} (supports \code{s()} smooths).
  \item \code{"rf"}: uses \code{randomForest::randomForest()}.
  \item \code{"ranger"}: uses \code{ranger::ranger()}.
  \item \code{"sl"}: uses \code{SuperLearner::SuperLearner()}.
        If \code{SL.library} is not given, a simple default library is used:
        c("SL.mean", "SL.glm", "SL.gam").
}
}
\examples{
\dontrun{
# Minimal sketch (assuming `df` has columns id, t, A, M, Y, I):
fit <- mcee_helper_2stage_estimation(
    data = df,
    id_var = "id", dp_var = "t", outcome_var = "Y",
    treatment_var = "A", mediator_var = "M", avail_var = "I",
    config_p = list(formula = ~ t + M, method = "glm"), # binomial auto
    config_q = list(formula = ~ t + M + A, method = "glm"), # binomial auto
    config_eta = list(formula = ~t, method = "gam"), # gaussian auto
    config_mu = list(formula = ~ t + s(M), method = "gam"), # gaussian auto
    config_nu = list(formula = ~t, method = "glm"), # gaussian auto
    omega_nrows = rep(1, nrow(df)),
    f_nrows = cbind(1) # marginal (p = 1)
)
fit$fit$alpha_hat
fit$fit$beta_hat
}
}
\seealso{
\code{\link{mcee_general}} for a high-level wrapper that constructs
\code{omega_nrows} and \code{f_nrows} from user-friendly arguments.
}
