% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grin.assoc.lsn.outcome.R
\name{grin.assoc.lsn.outcome}
\alias{grin.assoc.lsn.outcome}
\title{Associate Lesions with Clinical Outcomes}
\usage{
grin.assoc.lsn.outcome(
  lsn.mtx,
  clin.data,
  annotation.data,
  clinvars,
  covariate = NULL
)
}
\arguments{
\item{lsn.mtx}{A binary lesion matrix where each row represents a unique gene-lesion pair (e.g., \code{ENSG00000148400_gain}). Each column represents a patient. Values are denoted as \code{1} if the patient harbors the specified lesion, and \code{0} otherwise. This matrix is typically produced using the \code{\link{prep.binary.lsn.mtx}} function.}

\item{clin.data}{A clinical data \code{data.frame}, where the first column \code{"ID"} represent patient identifiers matching those in \code{lsn.mtx}.}

\item{annotation.data}{A gene annotation \code{data.frame}, either provided by the user or retrieved using the \code{\link{get.ensembl.annotation}} function. Must include the columns: \code{"gene"} (Ensembl ID), \code{"chrom"} (chromosome), \code{"loc.start"} (gene start position), and \code{"loc.end"} (gene end position).}

\item{clinvars}{A character vector of clinical outcome variables to analyze. Binary variables (e.g., MRD) should be coded as \code{0} (negative) and \code{1} (positive). Survival outcomes (e.g., EFS, OS) must be precomputed using the \code{\link[survival]{Surv}} function and added as new columns to \code{clin.data}.}

\item{covariate}{Optional. A character vector specifying covariates to include as model adjustments (e.g., risk group, age, gender, etc...).}
}
\value{
A results \code{data.frame} containing gene annotation and association statistics for each gene-lesion pair across the specified clinical outcomes. The output includes:
\itemize{
\item Odds ratio (OR), lower and upper 95CI, p-value, and q-value (FDR) for logistic regression models.
\item Hazard ratio (HR), lower and upper 95CI, p-value, and q-value for Cox proportional hazards models.
\item Patient counts for those with and without lesions, and corresponding outcome event statuses.
}
}
\description{
Performs statistical association analysis between binary gene-lesion events and clinical outcomes of interest, including binary outcomes (e.g., Minimal Residual Disease (MRD)) and time-to-event outcomes (e.g., Event-Free Survival (EFS), and Overall Survival (OS)). The function supports both univariate and covariate-adjusted logistic regression and Cox proportional hazards models.
}
\details{
For each gene-lesion pair in the binary lesion matrix, the function can performs:
\itemize{
\item \strong{Logistic regression} for binary outcomes (e.g., MRD), producing odds ratios (OR), 95_confidence intervals (CI), p-values, and FDR-adjusted q-values.
\item \strong{Cox proportional hazards models} for survival outcomes (e.g., EFS, OS), producing hazard ratios (HR), 95\\% CI, p-values, and FDR-adjusted q-values.
}
Models can optionally be adjusted for covariates such as clinical or demographic factors. Summary counts of patients with and without lesions, stratified by outcome status, are also included in the output.
}
\examples{
data(lesion_data)
data(hg38_gene_annotation)
data(clin_data)

# Step 1: Prepare gene-lesion overlap
gene.lsn <- prep.gene.lsn.data(lesion_data, hg38_gene_annotation)
gene.lsn.overlap <- find.gene.lsn.overlaps(gene.lsn)

# Step 2: Create a binary lesion matrix (minimum 5 patients per lesion)
lsn.binary.mtx <- prep.binary.lsn.mtx(gene.lsn.overlap, min.ngrp = 5)

# Step 3: Create survival objects and add to clinical data
library(survival)
clin_data$EFS <- Surv(clin_data$efs.time, clin_data$efs.censor)
clin_data$OS <- Surv(clin_data$os.time, clin_data$os.censor)

# Step 4: Specify outcomes of interest
clinvars <- c("MRD.binary", "EFS", "OS")

# Step 5: Run association analysis
assc.outcomes <- grin.assoc.lsn.outcome(lsn.binary.mtx,
                                        clin_data,
                                        hg38_gene_annotation,
                                        clinvars)

# Optional: Adjust for covariates using the 'covariate' argument
}
\references{
Andersen, P. K., & Gill, R. D. (1982). Cox's regression model for counting processes: A large sample study.

Therneau, T. M., & Grambsch, P. M. (2000). \emph{Modeling Survival Data: Extending the Cox Model}.

Dobson, A. J. (1990). \emph{An Introduction to Generalized Linear Models}.
}
\seealso{
\code{\link{prep.binary.lsn.mtx}}, \code{\link[survival]{coxph}}, \code{\link[stats]{glm}}
}
\author{
Abdelrahman Elsayed \email{abdelrahman.elsayed@stjude.org} and Stanley Pounds \email{stanley.pounds@stjude.org}
}
