#' @title Calculate Convex Hull
#' @description Internal function to calculate 2D convex hull coordinates and area.
#' This function replaces the dependency on SIBER::siberConvexhull.
#' @param x numeric vector of x-coordinates
#' @param y numeric vector of y-coordinates
#' @return A list containing:
#' \item{TA}{Total area of the convex hull}
#' \item{x}{x-coordinates of hull vertices}
#' \item{y}{y-coordinates of hull vertices}
#' \item{xcoords}{x-coordinates of hull vertices (alternative name for compatibility)}
#' \item{ycoords}{y-coordinates of hull vertices (alternative name for compatibility)}
#' \item{samples}{Row names of points on the hull}
#' @keywords internal
#' @importFrom grDevices chull
convexhull <- function(x, y) {
  # Handle edge cases
  n <- length(x)

  if (n < 3) {
    # Not enough points for a convex hull
    return(list(
      TA = 0,
      x = x,
      y = y,
      xcoords = x,
      ycoords = y,
      samples = if (!is.null(names(x))) names(x) else as.character(seq_along(x))
    ))
  }

  # Remove NA values
  valid <- !is.na(x) & !is.na(y)
  x <- x[valid]
  y <- y[valid]

  if (length(x) < 3) {
    return(list(
      TA = 0,
      x = x,
      y = y,
      xcoords = x,
      ycoords = y,
      samples = if (!is.null(names(x))) names(x) else as.character(seq_along(x))
    ))
  }

  # Get convex hull indices
  hull_indices <- chull(x, y)

  # Extract hull coordinates
  hull_x <- x[hull_indices]
  hull_y <- y[hull_indices]

  # Calculate area using shoelace formula
  # Area = 0.5 * |sum(x[i] * y[i+1] - x[i+1] * y[i])|
  n_hull <- length(hull_x)
  area <- 0
  for (i in seq_len(n_hull)) {
    j <- if (i == n_hull) 1 else i + 1
    area <- area + (hull_x[i] * hull_y[j] - hull_x[j] * hull_y[i])
  }
  area <- abs(area) / 2

  # Get sample names if available
  if (!is.null(names(x))) {
    sample_names <- names(x)[hull_indices]
  } else {
    sample_names <- as.character(hull_indices)
  }

  # Return in same format as siberConvexhull
  return(list(
    TA = area,
    x = hull_x,
    y = hull_y,
    xcoords = hull_x,
    ycoords = hull_y,
    samples = sample_names
  ))
}
