% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/otel-with.R
\name{withOtelCollect}
\alias{withOtelCollect}
\alias{localOtelCollect}
\title{Temporarily set OpenTelemetry (OTel) collection level}
\usage{
withOtelCollect(collect, expr)

localOtelCollect(collect, envir = parent.frame())
}
\arguments{
\item{collect}{Character string specifying the OpenTelemetry collection level.
Must be one of the following:

\if{html}{\out{<div class="sourceCode">}}\preformatted{* `"none"` - No telemetry data collected
* `"reactivity"` - Collect reactive execution spans (includes session and
  reactive update events)
* `"all"` - All available telemetry (currently equivalent to `"reactivity"`)
}\if{html}{\out{</div>}}}

\item{expr}{Expression to evaluate with the specified collection level
(for \code{withOtelCollect()}).}

\item{envir}{Environment where the collection level should be set
(for \code{localOtelCollect()}). Defaults to the parent frame.}
}
\value{
\itemize{
\item \code{withOtelCollect()} returns the value of \code{expr}.
\item \code{localOtelCollect()} is called for its side effect and returns the previous
\code{collect} value invisibly.
}
}
\description{
Control Shiny's OTel collection level for particular reactive expression(s).

\code{withOtelCollect()} sets the OpenTelemetry collection level for
the duration of evaluating \code{expr}. \code{localOtelCollect()} sets the collection
level for the remainder of the current function scope.
}
\details{
Note that \code{"session"} and \code{"reactive_update"} levels are not permitted as
these are runtime-specific levels that should only be set permanently via
\code{options(shiny.otel.collect = ...)} or the \code{SHINY_OTEL_COLLECT} environment
variable, not temporarily during reactive expression creation.
}
\section{Best practice}{


Best practice is to set the collection level for code that \emph{creates} reactive
expressions, not code that \emph{runs} them. For instance:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Disable telemetry for a reactive expression
withOtelCollect("none", \{
  my_reactive <- reactive(\{ ... \})
\})

# Disable telemetry for a render function
withOtelCollect("none", \{
  output$my_plot <- renderPlot(\{ ... \})
\})

#' # Disable telemetry for an observer
withOtelCollect("none", \{
  observe(\{ ... \}))
\})

# Disable telemetry for an entire module
withOtelCollect("none", \{
  my_result <- my_module("my_id")
\})
# Use `my_result` as normal here
}\if{html}{\out{</div>}}

NOTE: It's not recommended to pipe existing reactive objects into
\code{withOtelCollect()} since they won't inherit their intended OTel settings,
leading to confusion.
}

\examples{
\dontrun{
# Temporarily disable telemetry collection
withOtelCollect("none", {
  # Code here won't generate telemetry
  reactive({ input$x + 1 })
})

# Collect reactivity telemetry but not other events
withOtelCollect("reactivity", {
  # Reactive execution will be traced
  observe({ print(input$x) })
})

# Use local variant in a function
my_function <- function() {
  localOtelCollect("none")
  # Rest of function executes without telemetry
  reactive({ input$y * 2 })
}
}

}
\seealso{
See the \code{shiny.otel.collect} option within \code{\link{shinyOptions}}. Setting
this value will globally control OpenTelemetry collection levels.
}
