% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/piFunctions.R
\name{pi.sgm}
\alias{pi.sgm}
\title{Position index segmentation}
\usage{
pi.sgm(
  attTbl,
  ngbList,
  rNumb = FALSE,
  RO,
  mainPI = NULL,
  secPI = NULL,
  cut.mPI = NULL,
  cut.sPI = NULL,
  min.N = NULL,
  plot = FALSE,
  r = NULL
)
}
\arguments{
\item{attTbl}{data.frame, the attribute table returned by the function
\code{\link{attTbl}}.}

\item{ngbList}{list, the list of neighborhoods returned by the function
\code{\link{ngbList}}.}

\item{rNumb}{logic, the neighborhoods of the argument \code{ngbList} are
identified by cell numbers (\code{rNumb=FALSE}) or by row numbers
(\code{rNumb=TRUE}) (see \code{\link{ngbList}}). It is advised to use row
numbers for large rasters.}

\item{RO}{column name, the name of the column with the raster object IDs.}

\item{mainPI}{column name, the name of the column with main position index
values.}

\item{secPI}{column name, the name of the column with secondary position
index values.}

\item{cut.mPI}{numeric, threshold of main position index values. Cells with
values below the threshold are excluded from raster objects.}

\item{cut.sPI}{numeric, threshold of secondary position index values. Cells
with values below the threshold are excluded from raster objects.}

\item{min.N}{numeric, the minimum number of cells a raster object has to have
to be included in the function output.}

\item{plot}{logic, plot the results.}

\item{r}{single or multi-layer raster of the class \code{SpatRaster} (see
\code{help("rast", terra)}) used to compute the attribute table. Required
only if \code{plot = TRUE}.}
}
\value{
The function returns a class vector with raster objects IDs. The
vector has length equal to the number of rows of the attribute table. NA
values are assigned to cells that do not belong to any raster object.
}
\description{
Segment raster objects based on position index values.
}
\details{
Raster objects are segmented based on position index values. Two
different position indices can be passed to the function (\code{mainPI} and
\code{secPI}).
\itemize{
\item Input raster objects are assigned to the same class to flag cells that
are part of raster objects;
\item Cells with values below \code{mainPI} \strong{OR} below \code{mainPI} are
flagged as not being part of any raster object;
\item Each non-continuous group of raster object cells will identify an output
raster object.
\item Only raster objects with at least as many cells as specified by the
argument \code{min.N} are included in the function output.
\item If both \code{mainPI} and \code{secPI} are equal to \code{NULL}, the
function will exclusively filter raster objects based on their size
(\code{min.N}).
}
}
\examples{
# DUMMY DATA
######################################################################################
# LOAD LIBRARIES
library(scapesClassification)
library(terra)

# LOAD THE DUMMY RASTER
r <- list.files(system.file("extdata", package = "scapesClassification"),
                pattern = "dummy_raster\\\\.tif", full.names = TRUE)
r <- terra::rast(r)

# COMPUTE THE ATTRIBUTE TABLE
at <- attTbl(r, "dummy_var")

# COMPUTE THE LIST OF NEIGBORHOODS
nbs <- ngbList(r, attTbl=at)

################################################################################
# COMPUTE RASTER OBJECTS
################################################################################
at$RO <- anchor.seed(at, nbs, silent=TRUE, class = NULL, rNumb=TRUE,
                     cond.filter = "dummy_var > 1",
                     cond.seed   = "dummy_var==max(dummy_var)",
                     cond.growth = "dummy_var<dummy_var[]",
                     lag.growth  = Inf)

# One input raster object
unique(at$RO)

################################################################################
# NORMALIZED RELATIVE POSITION INDEX
################################################################################
at$relPI <- rel.pi(attTbl = at, RO = "RO", el = "dummy_var", type = "n")

################################################################################
# POSITION INDEX SEGMENTATION
################################################################################
RO1 <- pi.sgm(at, nbs,
              RO = "RO",        # Raster objects
              mainPI = "relPI", # PI segmentation layer
              cut.mPI = 0,      # segment on relPI values <= 0
              plot = FALSE, r = r)

################################################################################
# PLOT
################################################################################
# Convert class vectors to raster
r_RO  <- cv.2.rast(r = r, classVector = at$RO)
r_RO1 <- cv.2.rast(r = r, classVector = RO1)

# Plot
oldpar <- par(mfrow = c(1,2))
m <- c(4.5, 0.5, 2, 3.2)

terra::plot(r_RO, type="classes", main="Raster objects - Input", mar=m,
plg=list(x=1, y=1, cex=0.9))

terra::plot(r_RO1, type="classes", main="Raster objects - Output", mar=m,
            plg=list(x=1, y=1, cex=0.9))
text(xyFromCell(r,at$Cell), as.character(round(at$relPI,2))) # visualize relPI
text(0.01, 1, "Cut on relPI <= 0", adj=c(0,1), cex = 0.8)
par(oldpar)

# Two output raster objects
unique(RO1)
}
\seealso{
\code{\link[=attTbl]{attTbl()}}, \code{\link[=ngbList]{ngbList()}}, \code{\link[=rel.pi]{rel.pi()}}, \code{\link[=pi.add]{pi.add()}}
}
