#' Align read to reference sequence intermediated by pathogenic mutation
#'
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences
#' @param pathog.mut.obj A pathogenic mutation object
#' @param ref A reference sequence to be aligned
#' @param ref.start The start position of the reference sequence
#' @param ref.end The end position of the reference sequence
#' @param reads A read sequence to be aligned
#' @param is.scReads A logical value indicating whether the read is a soft-clipped read
#' @param softClippedReads.realign.match A non-negative integer specifying the scoring for a nucleotide match for realigning soft-clipped reads
#' @param softClippedReads.realign.mismatch A non-negative integer specifying the scoring for a nucleotide mismatch for realigning soft-clipped reads
#' @param softClippedReads.realign.gapOpening A non-negative integer specifying the cost for opening a gap in the realignment of soft-clipped reads
#' @param softClippedReads.realign.gapExtension A non-negative integer specifying the incremental cost incurred along the length of the gap in the realignment of soft-clipped reads
#' @param indel.shifting A character variable specifying the direction (left or right) for shifting the start position of indels till it is no longer possible to do so
#' 
#' @return A list of the read realignment including reference sequence, read sequence, cigar, start position of reference sequence, end position of reference sequence
#' 
#' @examples
#' \dontrun{
#'     alignReadToRef()
#' }
#'
#' @noRd
alignReadToRef <- function(
        reference.genome,
        pathog.mut.obj, 
        ref,
        ref.start,
        ref.end,
        reads, 
        is.scReads,
        softClippedReads.realign.match,
        softClippedReads.realign.mismatch,
        softClippedReads.realign.gapOpening,
        softClippedReads.realign.gapExtension,
        indel.shifting
){
    chr.name <- pathog.mut.obj$chr.name
    pathog.mut.type <- pathog.mut.obj$mut.type
    pathog.mut.start <- pathog.mut.obj$mut.start
    pathog.mut.end <- pathog.mut.obj$mut.end
    pathog.mut.ref <- pathog.mut.obj$mut.ref
    pathog.mut.alt <- pathog.mut.obj$mut.alt
    
    mutDNA <- data.frame(
        ref_pos = ref.start:ref.end,
        ref_seq = unlist( strsplit(ref,"") ),
        stringsAsFactors = FALSE
    )
    mutDNA$mut_seq <- mutDNA$ref_seq
    mutDNA$cigar <- rep("M", nrow(mutDNA))
    if(pathog.mut.type=="SNV"){
        mutDNA$mut_seq[ which(mutDNA$ref_pos==pathog.mut.start) ] <- pathog.mut.alt
    }else if(pathog.mut.type=="DEL"){
        mutDNA$mut_seq[ mutDNA$ref_pos %in% pathog.mut.start:pathog.mut.end ] <- "-"
        mutDNA$cigar[ mutDNA$ref_pos %in% pathog.mut.start:pathog.mut.end ] <- "D"
    }else if(pathog.mut.type=="INS"){
        temp.df <- data.frame(
            ref_pos = NA,
            ref_seq = "-",
            mut_seq = unlist( strsplit(pathog.mut.alt,"") ),
            cigar = "I",
            stringsAsFactors = FALSE
        )
        mutDNA <- rbind(
            mutDNA[ 1:which(mutDNA$ref_pos==pathog.mut.start), ],
            temp.df,
            mutDNA[ which(mutDNA$ref_pos==pathog.mut.end):nrow(mutDNA), ] 
        )
    }else if(pathog.mut.type=="DELINS"){
        mutDNA$mut_seq[ mutDNA$ref_pos %in% pathog.mut.start:(pathog.mut.end-1) ] <- "-"
        mutDNA$cigar[ mutDNA$ref_pos %in% pathog.mut.start:(pathog.mut.end-1) ] <- "D"
        temp.df <- data.frame(
            ref_pos = NA,
            ref_seq = "-",
            mut_seq = unlist( strsplit(pathog.mut.alt,"") ),
            cigar = "I",
            stringsAsFactors = FALSE
        )
        mutDNA <- rbind(
            mutDNA[ 1:which(mutDNA$ref_pos==(pathog.mut.end-1)), ],
            temp.df,
            mutDNA[ which(mutDNA$ref_pos==pathog.mut.end):nrow(mutDNA), ] 
        )
    }
    mutant.ref <- paste(mutDNA$mut_seq[mutDNA$mut_seq!="-"], collapse="")
    aligned.ref <- paste(mutDNA$ref_seq, collapse="")
    aligned.mutDNA <- paste(mutDNA$mut_seq, collapse="")
    cigar1 <- encodeCIGAR(aligned.ref, aligned.mutDNA)
    aln.mut2ref <- decodeCIGAR(cigar1, ref, mutant.ref)
    aln.mut2ref$ref_pos <- ref.start + aln.mut2ref$ref_index - 1    
    
    if(is.scReads){
        match <- softClippedReads.realign.match
        mismatch <- softClippedReads.realign.mismatch
        gapOpening <- softClippedReads.realign.gapOpening
        gapExtension <- softClippedReads.realign.gapExtension
    }else{
        match <- 1
        mismatch <- 4
        gapOpening <- 6
        if(pathog.mut.type=="SNV"){
            gapExtension <- 1
        }else{
            gapExtension <- 0
        }
    }
    
    pseudo.aln <- gapShiftingPairwiseAlignment(mutant.ref, reads, indel.shifting, "global", match, mismatch, gapOpening, gapExtension)
    
    cigar2 <- encodeCIGAR(pseudo.aln[1], pseudo.aln[2])
    aln.read2mut <- decodeCIGAR(cigar2, mutant.ref, reads)
    
    flag <- ifelse(aln.mut2ref$reads_seq=="-", "X", "N")
    flag.N <- c()
    flag.XY <- c()
    flag.N.pointer <- 0
    flag.XY.pointer <- 0
    for( i in 1:length(flag) ){
        if( flag[i]=="N" ){
            flag.N.pointer <- flag.N.pointer + 1
            flag.N <- c( flag.N, flag.N.pointer )
            flag.XY <- c( flag.XY, 0 )
        }else if( flag[i]=="X" ){
            flag.XY.pointer <- flag.XY.pointer + 1
            flag.N <- c( flag.N, flag.N.pointer )
            flag.XY <- c( flag.XY, flag.XY.pointer )
        }
    }
    aln.mut2ref$flag <- flag
    aln.mut2ref$flag_N <- flag.N
    aln.mut2ref$flag_XY <- flag.XY
    aln.mut2ref$flag_id <- paste(aln.mut2ref$flag_N, aln.mut2ref$flag, aln.mut2ref$flag_XY, sep="_")
    
    flag <- ifelse(aln.read2mut$ref_seq=="-", "Y", "N")
    flag.N <- c()
    flag.XY <- c()
    flag.N.pointer <- 0
    flag.XY.pointer <- 0
    for( i in 1:length(flag) ){
        if( flag[i]=="N" ){
            flag.N.pointer <- flag.N.pointer + 1
            flag.N <- c( flag.N, flag.N.pointer )
            flag.XY <- c( flag.XY, 0 )
        }else if( flag[i]=="Y" ){
            flag.XY.pointer <- flag.XY.pointer + 1
            flag.N <- c( flag.N, flag.N.pointer )
            flag.XY <- c( flag.XY, flag.XY.pointer )
        }
    }
    aln.read2mut$flag <- flag
    aln.read2mut$flag_N <- flag.N
    aln.read2mut$flag_XY <- flag.XY
    aln.read2mut$flag_id <- paste(aln.read2mut$flag_N, aln.read2mut$flag, aln.read2mut$flag_XY, sep="_")
    
    aln3 <- merge(aln.mut2ref, aln.read2mut, by="flag_id", all=TRUE, sort=FALSE)
    rownames(aln3) <- aln3$flag_id
    
    aln3.id <- as.data.frame(t(sapply(aln3$flag_id, function(x){unlist(strsplit(x,"_"))}))) 
    colnames(aln3.id) <- c("mergedID1","mergedID2","mergedID3")
    aln3.id$flag_id <- rownames(aln3.id)
    aln3.id$mergedID1 <- as.integer(aln3.id$mergedID1)
    aln3.id$mergedID3 <- as.integer(aln3.id$mergedID3)
    
    aln3 <- merge( aln3, aln3.id, by="flag_id", sort=F )
    aln3 <- aln3[ with(aln3, order(mergedID1,mergedID2,mergedID3) ), ]
    
    realn.df <- data.frame( 
        ref_seq = ifelse( is.na(aln3$ref_seq.x), "-", aln3$ref_seq.x), 
        reads_seq = ifelse( is.na(aln3$reads_seq.y), "-", aln3$reads_seq.y), 
        stringsAsFactors = FALSE
    )
    realn.df <- realn.df[ realn.df$ref_seq!="-" | realn.df$reads_seq!="-" , ]
    realn.ref <- paste( realn.df$ref_seq, collapse = "" )
    realn.reads <-  paste( realn.df$reads_seq, collapse = "" )
    
    realn.adj <- adjustIndels(realn.ref, realn.reads, reference.genome, chr.name, ref.start, ref.end)
    
    return( realn.adj )
}
