% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/one_compartment_iv_bolus_nl.R
\name{one_compartment_iv_bolus_nl}
\alias{one_compartment_iv_bolus_nl}
\title{One-Compartment IV Bolus Pharmacokinetic Model (Nonlinear)}
\arguments{
\item{data}{A data frame containing plasma concentration-time data.}

\item{time_col}{Character string specifying the column name for time.}

\item{conc_col}{Character string specifying the column name for plasma concentration.}

\item{dose}{Numeric value specifying the administered IV bolus dose.}

\item{group_col}{Optional character string specifying a grouping variable (e.g., subject, condition).}

\item{plot}{Logical; if TRUE, generates a concentration-time plot with fitted curves.}

\item{annotate}{Logical; if TRUE, annotates the plot with PK parameters (only if <= 2 groups).}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with C0, k_el, t1/2, Vd, CL, RMSE, AIC, and BIC.}
  \item{\code{data}}{Processed data used for fitting and plotting.}
}
}
\description{
Fits plasma concentration-time data to a one-compartment intravenous (IV) bolus
pharmacokinetic model using nonlinear regression. The model assumes instantaneous
drug distribution throughout a single, well-mixed compartment and first-order
elimination kinetics.

Model parameters are estimated by nonlinear least squares:
  - Elimination rate constant (k_el)
  - Initial plasma concentration (C0)
  - Apparent volume of distribution (Vd = Dose / C0)

Secondary pharmacokinetic parameters are derived:
  - Elimination half-life (t1/2 = ln(2)/k_el)
  - Clearance (CL = k_el * Vd)

The function supports optional grouping (e.g., subjects, conditions). Publication-quality
plots with fitted curves are generated, and annotations summarizing key PK parameters
appear in the upper-right corner when <= 2 groups.

Model:
C(t) = C0 * exp(-k_el * t)
}
\examples{
# Example I: Single subject one-compartment IV bolus data
df <- data.frame(
  time = c(0.08, 0.25, 0.5, 1, 2, 4, 6, 8, 12),
  concentration = c(18.2, 16.1, 13.5, 10.2, 6.8, 4.9, 3.6, 2.1, 1.2)
)
one_compartment_iv_bolus_nl(
  data = df,
  time_col = "time",
  conc_col = "concentration",
  dose = 100
)

# Example II: Condition-dependent pharmacokinetics (e.g., pH or physiological state)
df_cond <- data.frame(
  time = rep(c(0.25, 0.5, 1, 2, 4, 6), 2),
  concentration = c(
    17.8, 15.6, 13.1, 9.8, 6.4, 4.8,   # Condition A
    14.9, 13.0, 10.9, 8.0, 5.2, 3.9    # Condition B
  ),
  condition = rep(c("Condition A", "Condition B"), each = 6)
)
one_compartment_iv_bolus_nl(
  data = df_cond,
  time_col = "time",
  conc_col = "concentration",
  dose = 100,
  group_col = "condition"
)

# Example III: Multiple subjects (population-style one-compartment IV bolus pharmacokinetics)
df_subjects <- data.frame(
  time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 6),
  concentration = c(
    18.6, 16.3, 13.9, 10.5, 7.0, 5.1, 3.8,   # Subject 1
    17.9, 15.7, 13.2, 9.9, 6.6, 4.9, 3.6,    # Subject 2
    17.1, 15.0, 12.6, 9.4, 6.3, 4.7, 3.4,    # Subject 3
    16.4, 14.4, 12.0, 9.0, 6.0, 4.4, 3.2,    # Subject 4
    15.8, 13.9, 11.6, 8.7, 5.8, 4.2, 3.1,    # Subject 5
    15.2, 13.3, 11.0, 8.3, 5.5, 4.0, 2.9     # Subject 6
  ),
  subject = rep(paste0("S", 1:6), each = 7)
)
one_compartment_iv_bolus_nl(
  data = df_subjects,
  time_col = "time",
  conc_col = "concentration",
  dose = 100,
  group_col = "subject"
)
}
\references{
Widmark, E. M. P. (1919) Studies in the concentration of indifferent
narcotics in blood and tissues. Acta Medica Scandinavica, 52(1), 87–164.

Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
2nd Edition. Marcel Dekker, New York.

Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
Swedish Pharmaceutical Press, Stockholm.
}
\author{
Paul Angelo C. Manlapaz
}
