% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elnet.R
\name{elnet_cv}
\alias{elnet_cv}
\title{Cross-validation for Least-Squares (Adaptive) Elastic Net Estimates}
\usage{
elnet_cv(
  x,
  y,
  lambda,
  cv_k,
  cv_repl = 1,
  cv_type = "naive",
  cv_metric = c("rmspe", "tau_size", "mape", "auroc"),
  fit_all = TRUE,
  cl = NULL,
  ...
)
}
\arguments{
\item{x}{\code{n} by \code{p} matrix of numeric predictors.}

\item{y}{vector of response values of length \code{n}.
For binary classification, \code{y} should be a factor with 2 levels.}

\item{lambda}{optional user-supplied sequence of penalization levels.
If given and not \code{NULL}, \code{nlambda} and \code{lambda_min_ratio} are ignored.}

\item{cv_k}{number of folds per cross-validation.}

\item{cv_repl}{number of cross-validation replications.}

\item{cv_type}{what kind of cross-validation should be performed:
robust information sharing (\code{ris}) or standard (\code{naive}) CV.}

\item{cv_metric}{only for \code{cv_type='naive'}.
Either a string specifying the performance metric to use, or a function to
evaluate prediction errors in a single CV replication.
If a function, the number of arguments define the data the function receives.
If the function takes a single argument, it is called with a single numeric vector of
prediction errors.
If the function takes two or more arguments, it is called with the predicted values as
first argument and the true values as second argument.
The function must always return a single numeric value quantifying the prediction performance.
The order of the given values corresponds to the order in the input data.}

\item{fit_all}{only for \code{cv_type='naive'}.
If \code{TRUE}, fit the model for all penalization levels.
Can also be any combination of \code{"min"} and \code{"{x}-se"}, in which case only models at the
penalization level with smallest average CV accuracy, or within \code{{x}} standard errors,
respectively.
Setting \code{fit_all} to \code{FALSE} is equivalent to \code{"min"}.
Applies to all \code{alpha} value.}

\item{cl}{a \link[parallel:makeCluster]{parallel} cluster. Can only be used in combination with
\code{ncores = 1}.}

\item{...}{
  Arguments passed on to \code{\link[=elnet]{elnet}}
  \describe{
    \item{\code{alpha}}{elastic net penalty mixing parameter with \eqn{0 \le \alpha \le 1}.
\code{alpha = 1} is the LASSO penalty, and \code{alpha = 0} the Ridge penalty.
Can be a vector of several values, but \code{alpha = 0} cannot be mixed with other values.}
    \item{\code{nlambda}}{number of penalization levels.}
    \item{\code{lambda_min_ratio}}{Smallest value of the penalization level as a fraction of the largest
level (i.e., the smallest value for which all coefficients are zero).
The default depends on the sample size relative to the number of variables and \code{alpha}.
If more observations than variables are available, the default is \code{1e-3 * alpha},
otherwise \code{1e-2 * alpha}.}
    \item{\code{penalty_loadings}}{a vector of positive penalty loadings (a.k.a. weights) for
different penalization of each coefficient.}
    \item{\code{standardize}}{standardize variables to have unit variance.
Coefficients are always returned in original scale.}
    \item{\code{weights}}{a vector of positive observation weights.}
    \item{\code{intercept}}{include an intercept in the model.}
    \item{\code{sparse}}{use sparse coefficient vectors.}
    \item{\code{en_algorithm_opts}}{options for the EN algorithm. See \link{en_algorithm_options}
for details.}
    \item{\code{eps}}{numerical tolerance.}
  }}
}
\value{
a list-like object with the same components as returned by \code{\link[=elnet]{elnet()}},
plus the following:
\describe{
\item{\code{cvres}}{data frame of average cross-validated performance.}
}
}
\description{
Perform (repeated) K-fold cross-validation for \code{\link[=elnet]{elnet()}}.
}
\details{
The built-in CV metrics are
\describe{
\item{\code{"tau_size"}}{\eqn{\tau}-size of the prediction error, computed by
\code{\link[=tau_size]{tau_size()}} (default).}
\item{\code{"mape"}}{Median absolute prediction error.}
\item{\code{"rmspe"}}{Root mean squared prediction error.}
\item{\code{"auroc"}}{Area under the receiver operator characteristic curve (actually 1 - AUROC).
Only sensible for binary responses.}
}
}
\examples{
# Compute the LS-EN regularization path for Freeny's revenue data
# (see ?freeny)
data(freeny)
x <- as.matrix(freeny[ , 2:5])

regpath <- elnet(x, freeny$y, alpha = c(0.5, 0.75))
plot(regpath)
plot(regpath, alpha = 0.75)

# Extract the coefficients at a certain penalization level
coef(regpath, lambda = regpath$lambda[[1]][[5]],
     alpha = 0.75)

# What penalization level leads to good prediction performance?
set.seed(123)
cv_results <- elnet_cv(x, freeny$y, alpha = c(0.5, 0.75),
                       cv_repl = 10, cv_k = 4,
                       cv_measure = "tau")
plot(cv_results, se_mult = 1.5)
plot(cv_results, se_mult = 1.5, what = "coef.path")


# Extract the coefficients at the penalization level with
# smallest prediction error ...
summary(cv_results)
coef(cv_results)
# ... or at the penalization level with prediction error
# statistically indistinguishable from the minimum.
summary(cv_results, lambda = "1.5-se")
coef(cv_results, lambda = "1.5-se")
}
\seealso{
\code{\link[=elnet]{elnet()}} for computing the LS-EN regularization path without cross-validation.

\code{\link[=pense_cv]{pense_cv()}} for cross-validation of S-estimates of regression with elastic net penalty.

\code{\link[=coef.pense_cvfit]{coef.pense_cvfit()}} for extracting coefficient estimates.

\code{\link[=plot.pense_cvfit]{plot.pense_cvfit()}} for plotting the CV performance or the regularization path.

Other functions for computing non-robust estimates: 
\code{\link{elnet}()}
}
\concept{functions for computing non-robust estimates}
