% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_streamline.R
\docType{data}
\name{geom_streamline}
\alias{geom_streamline}
\alias{stat_streamline}
\alias{StatStreamline}
\alias{GeomStreamline}
\title{Streamlines}
\usage{
geom_streamline(
  mapping = NULL,
  data = NULL,
  stat = "streamline",
  position = "identity",
  ...,
  L = 5,
  min.L = 0,
  res = 1,
  S = NULL,
  dt = NULL,
  xwrap = NULL,
  ywrap = NULL,
  skip = 1,
  skip.x = skip,
  skip.y = skip,
  n = NULL,
  nx = n,
  ny = n,
  jitter = 1,
  jitter.x = jitter,
  jitter.y = jitter,
  arrow.angle = 6,
  arrow.length = 0.5,
  arrow.ends = "last",
  arrow.type = "closed",
  arrow = grid::arrow(arrow.angle, grid::unit(arrow.length, "lines"), ends = arrow.ends,
    type = arrow.type),
  lineend = "butt",
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = TRUE
)

stat_streamline(
  mapping = NULL,
  data = NULL,
  geom = "streamline",
  position = "identity",
  ...,
  L = 5,
  min.L = 0,
  res = 1,
  S = NULL,
  dt = NULL,
  xwrap = NULL,
  ywrap = NULL,
  skip = 1,
  skip.x = skip,
  skip.y = skip,
  n = NULL,
  nx = n,
  ny = n,
  jitter = 1,
  jitter.x = jitter,
  jitter.y = jitter,
  arrow.angle = 6,
  arrow.length = 0.5,
  arrow.ends = "last",
  arrow.type = "closed",
  arrow = grid::arrow(arrow.angle, grid::unit(arrow.length, "lines"), ends = arrow.ends,
    type = arrow.type),
  lineend = "butt",
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}}. If specified and
\code{inherit.aes = TRUE} (the default), it is combined with the default mapping
at the top level of the plot. You must supply \code{mapping} if there is no plot
mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{stat}{The statistical transformation to use on the data for this layer.
When using a \verb{geom_*()} function to construct a layer, the \code{stat}
argument can be used to override the default coupling between geoms and
stats. The \code{stat} argument accepts the following:
\itemize{
\item A \code{Stat} ggproto subclass, for example \code{StatCount}.
\item A string naming the stat. To give the stat as a string, strip the
function name of the \code{stat_} prefix. For example, to use \code{stat_count()},
give the stat as \code{"count"}.
\item For more information and other ways to specify the stat, see the
\link[ggplot2:layer_stats]{layer stat} documentation.
}}

\item{position}{A position adjustment to use on the data for this layer. This
can be used in various ways, including to prevent overplotting and
improving the display. The \code{position} argument accepts the following:
\itemize{
\item The result of calling a position function, such as \code{position_jitter()}.
This method allows for passing extra arguments to the position.
\item A string naming the position adjustment. To give the position as a
string, strip the function name of the \code{position_} prefix. For example,
to use \code{position_jitter()}, give the position as \code{"jitter"}.
\item For more information and other ways to specify the position, see the
\link[ggplot2:layer_positions]{layer position} documentation.
}}

\item{...}{Other arguments passed on to \code{\link[ggplot2:layer]{layer()}}'s \code{params} argument. These
arguments broadly fall into one of 4 categories below. Notably, further
arguments to the \code{position} argument, or aesthetics that are required
can \emph{not} be passed through \code{...}. Unknown arguments that are not part
of the 4 categories below are ignored.
\itemize{
\item Static aesthetics that are not mapped to a scale, but are at a fixed
value and apply to the layer as a whole. For example, \code{colour = "red"}
or \code{linewidth = 3}. The geom's documentation has an \strong{Aesthetics}
section that lists the available options. The 'required' aesthetics
cannot be passed on to the \code{params}. Please note that while passing
unmapped aesthetics as vectors is technically possible, the order and
required length is not guaranteed to be parallel to the input data.
\item When constructing a layer using
a \verb{stat_*()} function, the \code{...} argument can be used to pass on
parameters to the \code{geom} part of the layer. An example of this is
\code{stat_density(geom = "area", outline.type = "both")}. The geom's
documentation lists which parameters it can accept.
\item Inversely, when constructing a layer using a
\verb{geom_*()} function, the \code{...} argument can be used to pass on parameters
to the \code{stat} part of the layer. An example of this is
\code{geom_area(stat = "density", adjust = 0.5)}. The stat's documentation
lists which parameters it can accept.
\item The \code{key_glyph} argument of \code{\link[ggplot2:layer]{layer()}} may also be passed on through
\code{...}. This can be one of the functions described as
\link[ggplot2:draw_key]{key glyphs}, to change the display of the layer in the legend.
}}

\item{L, }{typical length of a streamline in x and y units}

\item{min.L}{minimum length of segments to show}

\item{res, }{resolution parameter (higher numbers increases the resolution)}

\item{S}{optional numeric number of timesteps for integration}

\item{dt}{optional numeric size "timestep" for integration}

\item{xwrap, ywrap}{vector of length two used to wrap the circular dimension.}

\item{skip, skip.x, skip.y}{numeric specifying number of gridpoints not to draw
in the x and y direction}

\item{n, nx, ny}{optional numeric indicating the number of points to draw in the
x and y direction (replaces \code{skip} if not \code{NULL})}

\item{jitter, jitter.x, jitter.y}{amount of jitter of the starting points}

\item{arrow.length, arrow.angle, arrow.ends, arrow.type}{parameters passed to
\link[grid:arrow]{grid::arrow}}

\item{arrow}{specification for arrow heads, as created by \code{\link[grid:arrow]{grid::arrow()}}.}

\item{lineend}{Line end style (round, butt, square).}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with
a warning. If \code{TRUE}, missing values are silently removed.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display. To include legend keys for all levels, even
when no data exists, use \code{TRUE}.  If \code{NA}, all levels are shown in legend,
but unobserved levels are omitted.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:annotation_borders]{annotation_borders()}}.}

\item{geom}{The geometric object to use to display the data for this layer.
When using a \verb{stat_*()} function to construct a layer, the \code{geom} argument
can be used to override the default coupling between stats and geoms. The
\code{geom} argument accepts the following:
\itemize{
\item A \code{Geom} ggproto subclass, for example \code{GeomPoint}.
\item A string naming the geom. To give the geom as a string, strip the
function name of the \code{geom_} prefix. For example, to use \code{geom_point()},
give the geom as \code{"point"}.
\item For more information and other ways to specify the geom, see the
\link[ggplot2:layer_geoms]{layer geom} documentation.
}}
}
\description{
Streamlines are paths that are always tangential to a vector field. In the
case of a steady field, it's identical to the path of a massless particle that
moves with the "flow".
}
\details{
Streamlines are computed by simple integration with a forward Euler method.
By default, \code{stat_streamline()} computes \code{dt} and \code{S} from \code{L}, \code{res},
the resolution of the grid and the mean magnitude of the field. \code{S} is
then defined as the number of steps necessary to make a streamline of length
\code{L} under an uniform mean field and \code{dt} is chosen so that each step is no
larger than the resolution of the data (divided by the \code{res} parameter). Be
aware that this rule of thumb might fail in field with very skewed distribution
of magnitudes.

Alternatively, \code{L} and/or \code{res} are ignored if \code{S} and/or \code{dt} are specified
explicitly. This not only makes it possible to fine-tune the result but also
divorces the integration parameters from the properties of the data and makes
it possible to compare streamlines between different fields.

The starting grid is a semi regular grid defined, either by the resolution of the
field and the \code{skip.x} and \code{skip.y} parameters o the \code{nx} and \code{ny} parameters,
jittered by an amount proportional to the resolution of the data and the
\code{jitter.x} and \code{jitter.y} parameters.

It might be important that the units of the vector field are compatible to the units
of the x and y dimensions. For example, passing \code{dx} and \code{dy} in m/s on a
longitude-latitude grid will might misleading results (see \link{spherical}).

Missing values are not permitted and the field must be defined on a
regular grid, for now.
}
\section{Aesthetics}{

\code{stat_streamline} understands the following aesthetics (required aesthetics are in bold)

\itemize{
\item \strong{x}
\item \strong{y}
\item \strong{dx}
\item \strong{dy}
\item \code{alpha}
\item \code{colour}
\item \code{linetype}
\item \code{size}
}
}

\section{Computed variables}{

\describe{
\item{step}{step in the simulation}
\item{dx}{dx at each location of the streamline}
\item{dy}{dy at each location of the streamline}
}
}

\examples{
\dontrun{
library(data.table)
library(ggplot2)
data(geopotential)

geopotential <- copy(geopotential)[date == date[1]]
geopotential[, gh.z := Anomaly(gh), by = .(lat)]
geopotential[, c("u", "v") := GeostrophicWind(gh.z, lon, lat)]

(g <- ggplot(geopotential, aes(lon, lat)) +
    geom_contour2(aes(z = gh.z), xwrap = c(0, 360)) +
    geom_streamline(aes(dx = dlon(u, lat), dy = dlat(v)), L = 60,
                    xwrap = c(0, 360)))

# The circular parameter is particularly important for polar coordinates
g + coord_polar()

# If u and v are not converted into degrees/second, the resulting
# streamlines have problems, specially near the pole.
ggplot(geopotential, aes(lon, lat)) +
    geom_contour(aes(z = gh.z)) +
    geom_streamline(aes(dx = u, dy = v), L = 50)

# The step variable can be mapped to size or alpha to
# get cute "drops". It's important to note that after_stat(dx) (the calculated variable)
# is NOT the same as dx (from the data).
ggplot(geopotential, aes(lon, lat)) +
    geom_streamline(aes(dx = dlon(u, lat), dy = dlat(v), alpha = after_stat(step),
                        color = sqrt(after_stat(dx^2) + after_stat(dy^2)),
                        size = after_stat(step)),
                        L = 40, xwrap = c(0, 360), res = 2, arrow = NULL,
                        lineend = "round") +
    scale_size(range = c(0, 0.6))

# Using topographic information to simulate "rivers" from slope
topo <- GetTopography(295, -55+360, -30, -42, res = 1/20)  # needs internet!
topo[, c("dx", "dy") := Derivate(h ~ lon + lat)]
topo[h <= 0, c("dx", "dy") := 0]

# See how in this example the integration step is too coarse in the
# western montanous region where the slope is much higher than in the
# flatlands of La Pampa at in the east.
ggplot(topo, aes(lon, lat)) +
    geom_relief(aes(z = h), interpolate = TRUE, data = topo[h >= 0]) +
    geom_contour(aes(z = h), breaks = 0, color = "black") +
    geom_streamline(aes(dx = -dx, dy = -dy), L = 10, skip = 3, arrow = NULL,
                    color = "#4658BD") +
    coord_quickmap()
 }

}
\seealso{
Other ggplot2 helpers: 
\code{\link{MakeBreaks}()},
\code{\link{WrapCircular}()},
\code{\link{geom_arrow}()},
\code{\link{geom_contour2}()},
\code{\link{geom_contour_fill}()},
\code{\link{geom_label_contour}()},
\code{\link{geom_relief}()},
\code{\link{guide_colourstrip}()},
\code{\link{map_labels}},
\code{\link{reverselog_trans}()},
\code{\link{scale_divergent}},
\code{\link{scale_longitude}},
\code{\link{stat_na}()},
\code{\link{stat_subset}()}
}
\concept{ggplot2 helpers}
\keyword{datasets}
