
# This defines the triangle function
#' Calculates Triangle
#'
#' This function calculates all group-differences
#'
#' @param mkbo_output placeholder text
#' @param term Specify the model term for which the mKBO results should be presented. Can be a vector of terms to present the summed results for those terms. Should be specified in quotation marks.
#' @param term.cat Specify a factor variable for which to sum the mKBO results across categories. Should be specified in quotation marks.
#' @param components Specify+he decomposition components to be included in the calculation. Can be any combination of c("E", "C", "I"), or "R".
#' @param percentage Specify to express the changes in gap as percentage (default) or in absolute differences (expressed in unites of the dependent variables).
#' @param absolute_gaps If TRUE, the changes in gaps are expressed in absolute terms even when signs change.
#' @return An object of class \code{tibble},  containing absolute or relative group-differences explained by the variables specified in  \code{mKBO}.
#' @importFrom tibble as_tibble
#' @importFrom dplyr %>%
#' @examples
#' mkbo_output <- mkbo("PERNP ~ BACHELOR", group = "RACE", data=pums_subset)
#' mkbo_triangle(mkbo_output, term="BACHELORTRUE")
#' @export

mkbo_triangle <- function(mkbo_output, term=NULL, term.cat=NULL, components = c("E"), percentage=TRUE, absolute_gaps=TRUE)
{

  mkbo_sum <- mkbo_summary(mkbo_output, term, term.cat) %>%
    mutate(components = rowSums(across(all_of(components))))

  dist_observed <- outer(mkbo_sum$M, mkbo_sum$M, "-")
  diag(dist_observed) <- mkbo_sum$M
  dist_counter <- outer(mkbo_sum$M+mkbo_sum$components, mkbo_sum$M+mkbo_sum$components, "-")
  diag(dist_counter) <- mkbo_sum$M+mkbo_sum$components

  if(absolute_gaps) {
    dist_observed <- abs(dist_observed)
    dist_counter <- abs(dist_counter)
  }

  distances <- dist_counter - dist_observed

  ifelse(percentage,
         triangle <- distances/(dist_observed) * 100,
         triangle <- distances)

  triangle[!lower.tri(triangle, diag=TRUE)] <- NA

  triangle <- as_tibble(triangle, .name_repair="minimal")
  colnames(triangle) <- mkbo_sum$group
  triangle$reference_group <- mkbo_sum$group

  return(triangle)
}
