% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_wfs.R
\name{get_wfs}
\alias{get_wfs}
\title{Download data from IGN WFS layer}
\usage{
get_wfs(
  x = NULL,
  layer = NULL,
  predicate = bbox(),
  query = NULL,
  verbose = TRUE
)
}
\arguments{
\item{x}{\code{sf}, \code{sfc} or \code{NULL}. If \code{NULL}, no spatial filter is applied
and \code{query} must be provided.}

\item{layer}{\code{character}; name of the WFS layer. Must correspond to a
layer available on the IGN WFS service (see \code{\link[=get_layers_metadata]{get_layers_metadata()}}).}

\item{predicate}{\code{list}; a spatial predicate definition created with helper
such as \code{bbox()}, \code{intersects()}, \code{within()}, \code{contains()}, \code{touches()},
\code{crosses()}, \code{overlaps()}, \code{equals()}, \code{dwithin()}, \code{beyond()} or
\code{relate()}. See \link{spatial_predicates} for more info.}

\item{query}{\code{character}; an ECQL attribute query. When both \code{x} and \code{query}
are provided, the spatial predicate and the attribute query are combined.}

\item{verbose}{\code{logical}; if \code{TRUE}, display progress information and
other informative message.}
}
\value{
An object of class \code{sf}.
}
\description{
Download features from the IGN Web Feature Service (WFS) using a
spatial predicate, an ECQL attribute query, or both.
}
\details{
\itemize{
\item \code{get_wfs} use ECQL language : a query language created by the
OpenGeospatial Consortium. More info about ECQL language can be
found \href{https://docs.geoserver.org/latest/en/user/filter/ecql_reference.html}{here}.
}
}
\examples{
\dontrun{
library(sf)

# Load a geometry
x <- read_sf(system.file("extdata/penmarch.shp", package = "happign"))

# Retrieve commune boundaries intersecting x
commune <- get_wfs(
  x = x,
  layer = "LIMITES_ADMINISTRATIVES_EXPRESS.LATEST:commune"
)

plot(st_geometry(commune), border = "firebrick")

# Attribute-only query (no spatial filter)

# If unknown, available attributes can be retrieved using `get_wfs_attributes()`
attrs <- get_wfs_attributes("LIMITES_ADMINISTRATIVES_EXPRESS.LATEST:commune")
print(attrs)

plou_communes <- get_wfs(
  x = NULL,
  layer = "LIMITES_ADMINISTRATIVES_EXPRESS.LATEST:commune",
  query = "nom_officiel ILIKE 'PLOU\%'"
)
plot(st_geometry(plou_communes))

# Multiple Attribute-only query (no spatial filter)
plou_inf_2000 <- get_wfs(
  x = NULL,
  layer = "LIMITES_ADMINISTRATIVES_EXPRESS.LATEST:commune",
  query = "nom_officiel ILIKE 'PLOU\%' AND population < 2000"
)
plot(st_geometry(plou_communes))
plot(st_geometry(plou_inf_2000), col = "firebrick", add = TRUE)

# Spatial predicate usage

layer <- "BDCARTO_V5:rond_point"

bbox_feat <- get_wfs(commune, layer, predicate = bbox())
plot(st_geometry(bbox_feat), col = "red")
plot(st_geometry(commune), add = TRUE)

intersects_feat <- get_wfs(commune, layer, predicate = intersects())
plot(st_geometry(intersects_feat), col = "red")
plot(st_geometry(commune), add = TRUE)

dwithin_feat <- get_wfs(commune, layer, predicate = dwithin(5, "kilometers"))
plot(st_geometry(dwithin_feat), col = "red")
plot(st_geometry(commune), add = TRUE)
}

}
\seealso{
\code{\link[=get_layers_metadata]{get_layers_metadata()}}
}
