# ExpoExpPower.R
#' Exponentiated Exponential Power (EEP) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Exponentiated Exponential Power (EEP) distribution.
#'
#' The EEP distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Exponentiated Exponential Power (EEP) distribution has CDF:
#'
#' \deqn{
#' F(x;\,\alpha ,\;\lambda ,\theta ) = {\left[ 
#' {1 - \exp \left\{ {1 - \exp \left( {\lambda {x^\alpha }} \right)} \right\}}
#'  \right]^\theta }\;\;\;;\;\;x > 0
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dgen.exp.power()} — Density function
#'   \item \code{pgen.exp.power()} — Distribution function
#'   \item \code{qgen.exp.power()} — Quantile function
#'   \item \code{rgen.exp.power()} — Random generation
#'   \item \code{hgen.exp.power()} — Hazard function
#' }
#'
#' @usage
#' dgen.exp.power(x, alpha, lambda, theta, log = FALSE)
#' pgen.exp.power(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qgen.exp.power(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rgen.exp.power(n, alpha, lambda, theta)
#' hgen.exp.power(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dgen.exp.power}: numeric vector of (log-)densities
#'   \item \code{pgen.exp.power}: numeric vector of probabilities
#'   \item \code{qgen.exp.power}: numeric vector of quantiles
#'   \item \code{rgen.exp.power}: numeric vector of random variates
#'   \item \code{hgen.exp.power}: numeric vector of hazard values
#' }
#'
#' @references
#' 
#' Sapkota, L.P., & Kumar, V.(2024).  
#' Bayesian Analysis of Exponentiated Exponential Power Distribution under 
#' Hamiltonian Monte Carlo Method, Statistics and Applications. 
#' \emph{Statistics and Applications}, \bold{22(2)}, 231--258.
#' 
#' Srivastava, A.K., & Kumar, V.(2011). 
#' Analysis of Software Reliability Data using Exponential Power Model. 
#' \emph{International Journal of Advanced Computer Science and Applications}, 
#' \bold{2(2)}, 38--45, \doi{10.14569/IJACSA.2011.020208}
#' 	
#' Chen, Z.(1999). Statistical inference about the shape parameter of 
#' the exponential power distribution, \emph{Statistical Papers}, \bold{40}, 459--468.
#' 	
#' Smith, R.M., & Bain, L.J. (1975). 
#' An exponential power life-test distribution. 
#' \emph{IEEE Communications in Statistics}, \bold{4}, 469--481.
#' 
#' @examples
#' x <- seq(0.1, 1, 0.1)
#' dgen.exp.power(x, 1.5, 0.8, 2)
#' pgen.exp.power(x, 1.5, 0.8, 2)
#' qgen.exp.power(0.5, 1.5, 0.8, 2)
#' rgen.exp.power(10, 1.5, 0.8, 2)
#' hgen.exp.power(x, 1.5, 0.8, 2)
#' #Data
#' x <- waiting
#' #ML Estimates    
#' params = list(alpha=0.3407, lambda=0.6068, theta=7.6150)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pgen.exp.power, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qgen.exp.power, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' # Neither plot nor console output; results stored in 'out'
#' out <- gofic(x, params = params,
#'              dfun = dgen.exp.power, pfun = pgen.exp.power, plot=FALSE)
#' print.gofic(out)
#'
#' @name ExpoExpPower
#' @export
NULL


#' @rdname ExpoExpPower
#' @usage NULL
#' @export
dgen.exp.power <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- exp(lambda * (x[valid] ^ alpha))	
	z <- exp(1- u) 	
    pdf[valid] <- alpha *lambda * theta * (x[valid]^(alpha-1)) * u * z * ((1-z)^(theta-1))
    }
	if (log) 
        pdf <- log(pdf)
    return(pdf)   
}

#' @rdname ExpoExpPower
#' @usage NULL
#' @export
pgen.exp.power <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- exp(lambda * (q[valid] ^ alpha))	
	z <- exp(1- u) 	
    cdf[valid] <- (1.0 - z) ^ theta
	}
    if (!lower.tail) 
        cdf <- 1.0 - cdf 
    if (log.p) 
        cdf <- log(cdf)    
    return(cdf)   
}

#' @rdname ExpoExpPower
#' @usage NULL
#' @export
qgen.exp.power <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <- ((1/lambda)*log(1-log(1-p[valid]^(1/theta)))) ^ (1/alpha)
	}
   
    return(qtl)   
}

#' @rdname ExpoExpPower
#' @usage NULL
#' @export
rgen.exp.power <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qgen.exp.power(u, alpha, lambda, theta)
    return(q)
}

#' @rdname ExpoExpPower
#' @usage NULL
#' @export
hgen.exp.power <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid arguments: alpha, lambda, theta, and x must be positive.")  
	 
	nume <- dgen.exp.power(x, alpha, lambda, theta)
	surv <- 1 - pgen.exp.power(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)	
    return(hrf)   
} 

