
#' @title  Gibbs sampler of BCFM
#' @description  It runs a Gibbs sampler for common factors X, factor loadings B, group mean mu, group covariate Omega, idiosyncratic variances sigma^2, group assignment Z and group probabilities probs. This function uses Rcpp.
#'
#' @param data  The dataset
#' @param model.attributes  Model attributes generated by initialize.model.attributes
#' @param hyp.parm  Hyperparameters generated by initialize.hyp.parm
#' @param n.iter  Total number of iterations
#' @param verbose  Print the results by every 10th step
#' @param every  Save every \code{every} iterations
#'
#' @return  List of Gibbs sampler of parameters described in description
#' @export

BCFMcpp <- function(data, model.attributes, hyp.parm, n.iter, every = 1, verbose = FALSE){
  
  n.save <- floor(n.iter / every)
  
  parm <- NULL
  parm$X <- array(0,c(model.attributes$S,model.attributes$L,model.attributes$times))
  parm$B<-matrix(0,model.attributes$R,model.attributes$L)
  
  for(i in 1:model.attributes$L){
    parm$B[i,i] <-1
    parm$B[(i+1):model.attributes$R,i] <- rnorm(model.attributes$R - i,mean = 0,sd = .1)
  }
  parm$mu <- hyp.parm$mu.m
  parm$omega <- hyp.parm$omega.psi # empirical Bayes procedure
  if(model.attributes$L > 1){
    parm$omega[1,,] <- diag(diag(parm$omega[1,,]))
  }
  parm$probs <- sort(rdirichlet(1, rep(1,model.attributes$G)), decreasing = TRUE)
  parm$Z <- matrix(sample(1:model.attributes$G,model.attributes$S*model.attributes$times, prob=rep(1/model.attributes$G, model.attributes$G),replace=TRUE),nrow=model.attributes$S, ncol=model.attributes$times)
  parm$sigma2 <- rep(1,model.attributes$R)
  parm$tau <- rep(1, model.attributes$L)
  
  X.Gibbs <- array(NA, c(n.save, model.attributes$S, model.attributes$L, model.attributes$times))
  B.Gibbs <- array(NA, c(n.save, model.attributes$R, model.attributes$L))
  mu.Gibbs <- array(NA, c(n.save, model.attributes$G, model.attributes$L))
  Omega.Gibbs <- array(NA, c(n.save, model.attributes$G, model.attributes$L, model.attributes$L))
  probs.Gibbs <- matrix(NA, n.save, model.attributes$G)
  Z.Gibbs <- array(NA, c(n.save, model.attributes$S, model.attributes$times))
  sigma2.Gibbs <- matrix(NA, n.save, model.attributes$R)
  tau.Gibbs <- matrix(NA, n.save, model.attributes$L)
  
  g.save <- 1
  
  for(g in 2:n.iter){
    parm$X <- simulated_X(model.attributes, parm, data)
    parm$mu <- simulated_mu(model.attributes,hyp.parm, parm)
    parm$omega <-  simulated_omega(model.attributes, hyp.parm, parm)
    parm$B <- simulated_B(model.attributes, parm, data)
    parm$sigma2 <- simulated_sigma(model.attributes, hyp.parm, parm, data)
    parm$tau <- simulated_tau(model.attributes, hyp.parm, parm)
    if(model.attributes$G > 1){
      parm$Z <- simulated_Z(model.attributes, parm)
      parm$probs <- simulated_probs(model.attributes, parm, hyp.parm)
    }
    
    if(every == 1 & g.save == 1){
      X.Gibbs[g.save,,,] <- parm$X
      B.Gibbs[g.save,,] <- parm$B
      mu.Gibbs[g.save,,] <- parm$mu
      Omega.Gibbs[g.save,,,] <- parm$omega
      probs.Gibbs[g.save,] <- parm$probs
      Z.Gibbs[g.save,,] <- parm$Z
      sigma2.Gibbs[g.save,] <- parm$sigma2
      tau.Gibbs[g.save,] <- parm$tau
      g.save <- g.save + 1
    }
    if(g %% every == 0){
      X.Gibbs[g.save,,,] <- parm$X
      B.Gibbs[g.save,,] <- parm$B
      mu.Gibbs[g.save,,] <- parm$mu
      Omega.Gibbs[g.save,,,] <- parm$omega
      probs.Gibbs[g.save,] <- parm$probs
      Z.Gibbs[g.save,,] <- parm$Z
      sigma2.Gibbs[g.save,] <- parm$sigma2
      tau.Gibbs[g.save,] <- parm$tau
      
      g.save <- g.save + 1
    }
    if(verbose){
      if (g %% 10000 == 0){message("Done #Groups = ", model.attributes$G, ", #Factors = ", model.attributes$L, ", ",g,"th step")}
    }
  }
  
  if(model.attributes$G == 1){
    Z.Gibbs <- array(1, c(n.save, model.attributes$S, model.attributes$times))
    probs.Gibbs <- matrix(1, n.save, 1)
  }
  
  all.Gibbs <- list(X.Gibbs, mu.Gibbs, Omega.Gibbs, B.Gibbs, sigma2.Gibbs, tau.Gibbs, Z.Gibbs, probs.Gibbs)
  names(all.Gibbs) <- c("X", "mu", "Omega", "B", "sigma2", "tau", "Z", "probs")
  return(all.Gibbs)
  
}
