% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spheroid.R
\name{simSpheroidSystem}
\alias{simSpheroidSystem}
\title{Simulation of spheroid system}
\usage{
simSpheroidSystem(theta, lam, size = "const", shape = "const",
  orientation = "rbetaiso", stype = c("prolate", "oblate"), rjoint = NULL,
  box = list(c(0, 1)), mu = c(0, 0, 1), perfect = TRUE, pl = 0,
  label = "N")
}
\arguments{
\item{theta}{simulation parameters}

\item{lam}{mean number of spheroids per unit volume}

\item{size}{name of random generating function for size distribution}

\item{shape}{\code{shape="const"} (default) as a constant shape}

\item{orientation}{name of random generating function for orientation distribution}

\item{stype}{spheroid type}

\item{rjoint}{name of joint random generating function}

\item{box}{simulation box}

\item{mu}{main orientation axis, \code{mu=c(0,0,1)} (default)}

\item{perfect}{logical: \code{perfect=TRUE} (default) simulate perfect}

\item{pl}{optional: print level}

\item{label}{optional: set a label to all generated spheroids}
}
\value{
list of spheroids either of class \code{prolate} or \code{oblate}
}
\description{
Simulation of Poisson spheroid system
}
\details{
The function simulates a Poisson spheroid system according to the supplied
simulation parameter \code{theta} in a predefined simulation box.
The argument \code{size} is of type string and denotes the major-axis length random generating
function name.

Further the function simulates either \code{stype}="prolate" or \code{stype}='oblate' spheroids.
For the directional orientation of the spheroid's major-axis one has the choice of a uniform
(\code{runifdir}), isotropic random planar (\code{rbetaiso}, see reference) or von Mises-Fisher
(\code{rvMisesFisher}) distribution. The simulation box is a list containing of vector arguments
which correspond to the lower and upper points in each direction. If the argument \code{box} has
only one element, i.e. \code{list(c(0,1)}, the same extent is used for the other dimensions.
If \code{rjoint} names a joint random generating function then argument \code{size} is ignored.
For the purpose of exact simulation setting \code{size} equal to \code{rbinorm} declares a bivarite
size-shape distribution which leads to a log normal distributed semi-major axis \code{a} and a scaled
semi-minor axis length \code{c}. If \eqn{[X,Y]} follow a bivariate normal distribution with correlation parameter
\eqn{\rho} then \eqn{a=exp(x)} defines the sample semi-major axis length together with the scaled semi-minor
axis length \eqn{c=a*s} and shape parameter set to \eqn{s=1/(1+exp(-y))}. The parameter \eqn{\rho} defines
the degree of correlation between the semi-axes lengths which must be provided as part of the list of simulation
parameters \code{theta}. The method of exact simulation is tailored to the above described model. For a general
approach please see the given reference below. Other (univariate)  major-axis lengths types include the beta,
gamma, lognormal and uniform distribution where the shape factor which determines the minor-axis length either
follows a beta distribution or is set to a constant. Despite the case of constant size simulations all other
simulations are done as perfect simulations.

The argument \code{pl} denotes the print level of output information during simulation.
Currently, only \code{pl}=0 for no output and \code{pl}>100 for some additional info is implemented.
}
\examples{
\dontrun{
	
# directional distribution
rbetaiso <- function(kappa) {
   phi <- runif(1,0,1)*2*pi
   q <- runif(1,0,1)
   theta=acos((1-2*q)/sqrt(kappa*kappa-(1-2*q)*(1-2*q)*(kappa*kappa-1)))
   list("u"=c(sin(theta)*cos(phi),sin(theta)*sin(phi),cos(theta)),
		   "theta"=theta,"phi"=phi)					
}

# no perfect simualtion here for 'rmulti'
# multivariate size distribution and orientation distribution 
rmulti <- function(m,s,kappa) {	
   dir <- rbetaiso(kappa)
   M <- chol(s, pivot = TRUE)
   M <- M[, order(attr(M, "pivot"))]
   x <- exp(matrix(m,nrow=1) +
          matrix(rnorm(ncol(s)), nrow = 1, byrow = TRUE) \%*\%M)
   a <- min(x)
   b <- max(x)
   
   list("a"=a,"b"=b,"u"=dir$u,"shape"=a/b,
        "theta"=dir$theta, "phi"=dir$phi)

}

set.seed(1234)
lam <- 100
box <- list("xrange"=c(0,5),"yrange"=c(0,5),"zrange"=c(0,5))

# joint (user defined random generating function) simualtion
sigma <- matrix(c(0.1,0.1,0.1,0.25), ncol=2)
theta <- list("m"=c(-3.0,-2.0),"s"=sigma,"kappa"=0.5)
S <- simSpheroidSystem(theta,lam,rjoint="rmulti",box=box,pl=101)

# Spheroids with log normal distributed semi-major axis length
# distribution and independent orientation distribution
theta <- list("size"=list("meanlog"=-2.5,"sdlog"=0.5),
              "shape"=list("s"=0.5),
              "orientation"=list("kappa"=1.5))
  
S <- simSpheroidSystem(theta,lam,size="rlnorm",shape="const",orientation="rbetaiso",
		box=box,pl=101)
			   
# log normal, rbeta shape
theta <- list("size"=list("meanlog"=-2.5,"sdlog"=0.5),
			  "shape"=list("a"=1,"b"=2),
		      "orientation"=list("kappa"=1.5))

S <- simSpheroidSystem(theta,lam,size="rlnorm",shape="rbeta",orientation="rbetaiso",
		box=box,pl=101)

# Spheroids of constant sizes, const shape
theta <- list("size"=list(0.25),
              "shape"=list("s"=0.5),
              "orientation"=list("kappa"=1))
S <- simSpheroidSystem(theta,lam,size="const",shape="const",orientation="rbetaiso",
                       box=box,pl=101)

# constant size, rbeta shape			   
theta <- list("size"=list(0.25),
			  "shape"=list("a"=1,"b"=2),
			  "orientation"=list("kappa"=1.5))	  
			   
S <- simSpheroidSystem(theta,lam,size="const",shape="rbeta",orientation="rbetaiso",
					   box=box,pl=101)			   
			   
			   
# Spheroids exact simulation
param <- list("mx"=-1.0,"my"=-2.5, "sdx"=0.15,"sdy"=0.2,"rho"=0.0,"kappa"=1.0)
theta <- list("size"=list("mx"=param$mx,
					      "sdx"=param$sdx,
						  "my"=param$my,
						  "sdy"=param$sdy,
						  "rho"=param$rho),
			  "orientation"=list("kappa"=param$kappa),
			  "shape"=list())

S <- simSpheroidSystem(theta,lam,size="rbinorm",orientation="rbetaiso",box=box,pl=101)	


## show spheroid system 
#cols <- c("#0000FF","#00FF00","#FF0000","#FF00FF","#FFFF00","#00FFFF")
#spheroids3d(S[1:500],box=box, col=cols)

}

}
\references{
\itemize{
	\item{} {Ohser, J. and Schladitz, K. 3D images of materials structures Wiley-VCH, 2009}
     \item{} {C. Lantu\eqn{\acute{\textrm{e}}}joul. Geostatistical simulation. Models and algorithms.
					Springer, Berlin, 2002. Zbl 0990.86007}
	  }
}

