#' @title Sample (with noise) from disk
#'
#' @description These functions generate uniform samples from a disk in
#'   2-dimensional space, optionally with noise.
#'
#' @details The sample is generated by an area-preserving parameterization of
#'   the disk. This parametrization was derived through the method for sampling
#'   2-manifolds as described by Arvo (2001).

#' @template ref-arvo2001
#' 

#' @name disks
#' @param n Number of observations.
#' @param bins Number of intervals per dimension to stratify by. Default set to
#'   1, which generates a uniform sample.
#' @param sd Standard deviation of (independent multivariate) Gaussian noise.
#' @example inst/examples/ex-disks.r
NULL

#' @rdname disks
#' @export
sample_disk <- function(n, bins = 1L, sd = 0) {
  # generate a uniform sample from [0,1]^2
  unit_square <- sample_strat_square(n, bins)
  # area-preserving parametrization of disk
  res <- apm_disk(unit_square)
  # add noise
  add_noise(res, sd = sd)
}

# area-preserving map from [0,1]^2 to the disk
apm_disk <- function(x) {
  cbind(
    x = sqrt(x[, 1]) * cos(2 * pi * x[, 2]),
    y = sqrt(x[, 1]) * sin(2 * pi * x[, 2])
  )
}
