% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linearIRF.R, R/plotMethods.R, R/printMethods.R
\name{linear_IRF}
\alias{linear_IRF}
\alias{plot.irf}
\alias{print.irf}
\title{Estimate linear impulse response function based on a single regime of a structural STVAR model.}
\usage{
linear_IRF(
  stvar,
  N = 30,
  regime = 1,
  which_cumulative = numeric(0),
  scale = NULL,
  ci = NULL,
  bootstrap_reps = 100,
  ncores = 2,
  robust_method = c("Nelder-Mead", "SANN", "none"),
  maxit_robust = 1000,
  seed = NULL,
  ...
)

\method{plot}{irf}(x, shocks_to_plot, ...)

\method{print}{irf}(x, ..., digits = 2, N_to_print, shocks_to_print)
}
\arguments{
\item{stvar}{an object of class \code{'stvar'} defining a structural or reduced form
STVAR model. For a reduced form model, the shocks are automatically identified by
the lower triangular Cholesky decomposition.}

\item{N}{a positive integer specifying the horizon how far ahead should the
linear impulse responses be calculated.}

\item{regime}{Based on which regime the linear IRF should be calculated?
An integer in \eqn{1,...,M}.}

\item{which_cumulative}{a numeric vector with values in \eqn{1,...,d}
(\code{d=ncol(data)}) specifying which the variables for which the linear impulse
responses should be cumulative. Default is none.}

\item{scale}{should the linear IRFs to some of the shocks be scaled so that they
correspond to a specific instantaneous response of some specific
variable? Provide a length three vector where the shock of interest
is given in the first element (an integer in \eqn{1,...,d}), the variable of
interest is given in the second element (an integer in \eqn{1,...,d}), and
its instantaneous response in the third element (a non-zero real number).
If the linear IRFs of multiple shocks should be scaled, provide a matrix which has one
column for each of the shocks with the columns being the length three vectors described above.}

\item{ci}{a real number in \eqn{(0, 1)} specifying the confidence level of the
confidence intervals calculated via a fixed-design wild residual bootstrap method.
Available only for models that impose linear autoregressive dynamics
(excluding changes in the volatility regime).}

\item{bootstrap_reps}{the number of bootstrap repetitions for estimating confidence bounds.}

\item{ncores}{the number of CPU cores to be used in parallel computing when bootstrapping confidence bounds.}

\item{robust_method}{Should some robust estimation method be used in the estimation before switching
to the gradient based variable metric algorithm? See details.}

\item{maxit_robust}{the maximum number of iterations on the first phase robust estimation, if employed.}

\item{seed}{a real number initializing the seed for the random generator.}

\item{...}{currently not used.}

\item{x}{object of class \code{'irf'} generated by the function \code{linear_IRF}.}

\item{shocks_to_plot}{IRFs of which shocks should be plotted? A numeric vector
with elements in \code{1,...,d}.}

\item{digits}{the number of decimals to print}

\item{N_to_print}{an integer specifying the horizon how far to print the estimates and
confidence intervals. The default is that all the values are printed.}

\item{shocks_to_print}{the responses to which should should be printed?
A numeric vector with elements in \code{1,...,d}. The default is that responses to
all the shocks are printed.}
}
\value{
Returns a class \code{'irf'} list with  with the following elements:
  \describe{
    \item{\code{$point_est}:}{a 3D array \code{[variables, shock, horizon]} containing the point estimates of the IRFs.
       Note that the first slice is for the impact responses and the slice i+1 for the period i. The response of the
       variable 'i1' to the shock 'i2' is subsetted as \code{$point_est[i1, i2, ]}.}
    \item{\code{$conf_ints}:}{bootstrapped confidence intervals for the IRFs in a \code{[variables, shock, horizon, bound]}
       4D array. The lower bound is obtained as \code{$conf_ints[, , , 1]}, and similarly the upper bound as
        \code{$conf_ints[, , , 2]}. The subsetted 3D array is then the bound in a form similar to \code{$point_est}.}
    \item{\code{$all_bootstrap_reps}:}{IRFs from all of the bootstrap replications in a \code{[variables, shock, horizon, rep]}.
       4D array. The IRF from replication i1 is obtained as \code{$all_bootstrap_reps[, , , i1]}, and the subsetted 3D array
       is then the in a form similar to \code{$point_est}.}
    \item{Other elements:}{contains some of the arguments the \code{linear_IRF} was called with.}
  }
}
\description{
\code{linear_IRF} estimates linear impulse response function based on a single regime
  of a structural STVAR model.
}
\details{
If the autoregressive dynamics of the model are linear (i.e., either M == 1 or mean and AR parameters
  are constrained identical across the regimes), confidence bounds can be calculated based on a fixed-design
  wild residual bootstrap method. We employ the method described in Herwartz and Lütkepohl (2014); see also
  the relevant chapters in Kilian and Lütkepohl (2017).

  Employs the estimation function \code{optim} from the package \code{stats} that implements the optimization
  algorithms. The robust optimization method Nelder-Mead is much faster than SANN but can get stuck at a local
  solution. See \code{?optim} and the references therein for further details.

  For model identified by non-Gaussianity, the signs and ordering of the shocks are normalized by assuming
  that the first non-zero element of each column of the impact matrix of Regime 1 is strictly positive and they are
  in a decreasing order. Use the argument \code{scale} to obtain IRFs scaled for specific impact responses.
}
\section{Functions}{
\itemize{
\item \code{plot(irf)}: plot method

\item \code{print(irf)}: print method

}}
\examples{
\donttest{
## These are long running examples that take approximately 10 seconds to run.
## A small number of bootstrap replications is used below to shorten the
## running time (in practice, a larger number of replications should be used).

# p=1, M=1, d=2, linear VAR model with independent Student's t shocks identified
# by non-Gaussianity (arbitrary weight function applied here):
theta_112it <- c(0.644, 0.065, 0.291, 0.021, -0.124, 0.884, 0.717, 0.105, 0.322,
  -0.25, 4.413, 3.912)
mod112 <- STVAR(data=gdpdef, p=1, M=1, params=theta_112it, cond_dist="ind_Student",
 identification="non-Gaussianity", weight_function="threshold", weightfun_pars=c(1, 1))
mod112 <- swap_B_signs(mod112, which_to_swap=1:2)

# Estimate IRFs 20 periods ahead, bootstrapped 90\% confidence bounds based on
# 10 bootstrap replications. Linear model so robust estimation methods are
# not required.
irf1 <- linear_IRF(stvar=mod112, N=20, regime=1, ci=0.90, bootstrap_reps=1,
 robust_method="none", seed=1, ncores=1)
plot(irf1)
print(irf1, digits=3)

# p=1, M=2, d=2, Gaussian STVAR with relative dens weight function,
# shocks identified recursively.
theta_122relg <- c(0.734054, 0.225598, 0.705744, 0.187897, 0.259626, -0.000863,
  -0.3124, 0.505251, 0.298483, 0.030096, -0.176925, 0.838898, 0.310863, 0.007512,
  0.018244, 0.949533, -0.016941, 0.121403, 0.573269)
mod122 <- STVAR(data=gdpdef, p=1, M=2, params=theta_122relg, identification="recursive")

# Estimate IRF based on the first regime 30 period ahead. Scale IRFs so that
# the instantaneous response of the first variable to the first shock is 0.3,
# and the response of the second variable to the second shock is 0.5.
# response of the Confidence bounds
# are not available since the autoregressive dynamics are nonlinear.
irf2 <- linear_IRF(stvar=mod122, N=30, regime=1, scale=cbind(c(1, 1, 0.3), c(2, 2, 0.5)))
plot(irf2)

 # Estimate IRF based on the second regime without scaling the IRFs:
irf3 <- linear_IRF(stvar=mod122, N=30, regime=2)
plot(irf3)

# p=3, M=2, d=3, Students't logistic STVAR model with the first lag of the second
# variable as the switching variable. Autoregressive dynamics restricted linear,
# but the volatility regime varies in time, allowing the shocks to be identified
# by conditional heteroskedasticity.
theta_322 <- c(0.7575, 0.6675, 0.2634, 0.031, -0.007, 0.5468, 0.2508, 0.0217, -0.0356,
 0.171, -0.083, 0.0111, -0.1089, 0.1987, 0.2181, -0.1685, 0.5486, 0.0774, 5.9398, 3.6945,
 1.2216, 8.0716, 8.9718)
mod322 <- STVAR(data=gdpdef, p=3, M=2, params=theta_322, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", mean_constraints=list(1:2),
  AR_constraints=rbind(diag(3*2^2), diag(3*2^2)), identification="heteroskedasticity",
  parametrization="mean")

## Estimate IRFs 30 periods ahead, bootstrapped 90\% confidence bounds based on
# 10 bootstrap replications. Responses of the second variable are accumulated.
irf4 <- linear_IRF(stvar=mod322, N=30, regime=1, ci=0.90, bootstrap_reps=10,
 which_cumulative=2, seed=1)
plot(irf4)
}
}
\references{
\itemize{
   \item Herwartz H. and Lütkepohl H. 2014. Structural vector autoregressions with Markov switching:
     Combining conventional with statistical identification of shocks. \emph{Journal of Econometrics},
     183, pp. 104-116.
   \item Kilian L. and Lütkepohl H. 2017. Structural Vectors Autoregressive Analysis.
         \emph{Cambridge University Press}, Cambridge.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{GFEVD}}, \code{\link{fitSTVAR}}, \code{\link{STVAR}},
  \code{\link{reorder_B_columns}}, \code{\link{swap_B_signs}}
}
