% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unif-alts.R
\name{locdev}
\alias{locdev}
\alias{f_locdev}
\alias{con_f}
\alias{d_locdev}
\alias{r_locdev}
\alias{cutoff_locdev}
\title{Local projected alternatives to uniformity}
\usage{
f_locdev(z, p, uk)

con_f(f, p, N = 320)

d_locdev(x, mu, f, kappa)

r_locdev(n, mu, f, kappa, F_inv = NULL, ...)

cutoff_locdev(p, K_max = 10000, thre = 0.001, type, Rothman_t = 1/3,
  Pycke_q = 0.5, verbose = FALSE, Gauss = TRUE, N = 320, tol = 1e-06)
}
\arguments{
\item{z}{projected evaluation points for \eqn{f}, a vector with entries on
\eqn{[-1, 1]}.}

\item{p}{integer giving the dimension of the ambient space \eqn{R^p} that
contains \eqn{S^{p-1}}.}

\item{uk}{coefficients \eqn{u_{k, p}} associated to the indexes
\code{1:length(uk)}, a vector.}

\item{f}{angular function defined on \eqn{[-1, 1]}. Must be vectorized.}

\item{N}{number of points used in the \link[=Gauss_Legen_nodes]{
Gauss--Legendre quadrature} for computing the Gegenbauer coefficients.
Defaults to \code{320}.}

\item{x}{locations in \eqn{S^{p-1}} to evaluate the density. Either a
matrix of size \code{c(nx, p)} or a vector of length \code{p}. Normalized
internally if required (with a \code{warning} message).}

\item{mu}{a unit norm vector of size \code{p} giving the axis of rotational
symmetry.}

\item{kappa}{the strength of the local alternative, between \code{0}
and \code{1}.}

\item{n}{sample size, a positive integer.}

\item{F_inv}{quantile function associated to \eqn{f}. Computed by
\code{\link{F_inv_from_f}} if \code{NULL} (default).}

\item{...}{further parameters passed to \code{\link{F_inv_from_f}}.}

\item{K_max}{integer giving the truncation of the series. Defaults to
\code{1e4}.}

\item{thre}{proportion of norm \emph{not} explained by the first terms of the
truncated series. Defaults to \code{1e-3}.}

\item{type}{Sobolev statistic. For \eqn{p = 2}, either \code{"Watson"},
\code{"Rothman"}, \code{"Pycke_q"}, or \code{"Hermans_Rasson"}.
For \eqn{p \ge 2}, \code{"Ajne"}, \code{"Gine_Gn"}, \code{"Gine_Fn"},
\code{"Bakshaev"}, \code{"Riesz"}, \code{"PCvM"}, \code{"PAD"}, or
\code{"PRt"}.}

\item{Rothman_t}{\eqn{t} parameter for the Rothman test, a real in
\eqn{(0, 1)}. Defaults to \code{1 / 3}.}

\item{Pycke_q}{\eqn{q} parameter for the Pycke "\eqn{q}-test", a real in
\eqn{(0, 1)}. Defaults to \code{1 / 2}.}

\item{verbose}{output information about the truncation (\code{TRUE} or
\code{1}) and a diagnostic plot (\code{2})? Defaults to \code{FALSE}.}

\item{Gauss}{use a Gauss--Legendre quadrature rule of \code{N} nodes
in the computation of the Gegenbauer coefficients? Otherwise, call
\code{\link{integrate}}. Defaults to \code{TRUE}.}

\item{tol}{tolerance passed to \code{\link{integrate}}'s \code{rel.tol} and
\code{abs.tol} if \code{Gauss = FALSE}. Defaults to \code{1e-6}.}
}
\value{
\itemize{
  \item \code{f_locdev}: angular function evaluated at \code{x}, a vector.
  \item \code{con_f}: normalizing constant \eqn{c_f} of \eqn{f}, a scalar.
  \item \code{d_locdev}: density function evaluated at \code{x}, a vector.
  \item \code{r_locdev}: a matrix of size \code{c(n, p)} containing a random
  sample from the density \eqn{f_{\kappa, \boldsymbol{\mu}}}{
  f_{\kappa, \mu}}.
  \item \code{cutoff_locdev}: vector of coefficients \eqn{\{u_{k, p}\}}
  automatically truncated according to \code{K_max} and \code{thre}
  (see details).
}
}
\description{
Density and random generation for local projected alternatives
to uniformity with densities
\deqn{f_{\kappa, \boldsymbol{\mu}}({\bf x}): =
\frac{1 - \kappa}{\omega_p} + \kappa f({\bf x}'\boldsymbol{\mu})}{
f_{\kappa, \mu}(x) = (1 - \kappa) / \omega_p + \kappa f(x'\mu)}
where
\deqn{f(z) = \frac{1}{\omega_p}\left\{1 + \sum_{k = 1}^\infty u_{k, p}
C_k^{p / 2 - 1}(z)\right\}}{f(x) = (1 / \omega_p)
\{1 + \sum_{k = 1}^\infty u_{k, p} C_k^(p / 2 - 1)(z)\}}
is the \emph{angular function} controlling the local alternative in a
\link[=Gegenbauer]{Gegenbauer series}, \eqn{0\le \kappa \le 1},
\eqn{\boldsymbol{\mu}}{\mu} is a direction on \eqn{S^{p - 1}}, and
\eqn{\omega_p} is the surface area of \eqn{S^{p - 1}}. The sequence
\eqn{\{u_{k, p}\}} is typically such that
\eqn{u_{k, p} = \left(1 + \frac{2k}{p - 2}\right) b_{k, p}}{
u_{k, p} = (1 + 2k / (p - 2)) b_{k, p}} for the Gegenbauer coefficients
\eqn{\{b_{k, p}\}} of the kernel function of a Sobolev statistic (see the
\link[=Sobolev_coefs]{transformation} between the coefficients \eqn{u_{k, p}}
and \eqn{b_{k, p}}).

Also, automatic truncation of the series \eqn{\sum_{k = 1}^\infty u_{k, p}
C_k^{p / 2 - 1}(z)}{\sum_{k = 1}^\infty u_{k, p} C_k^(p / 2 - 1)(z)}
according to the proportion of \link[=Gegenbauer]{"Gegenbauer norm"}
explained.
}
\details{
See the definitions of local alternatives in Prentice (1978) and in
García-Portugués et al. (2023).

The truncation of \eqn{\sum_{k = 1}^\infty u_{k, p} C_k^{p / 2 - 1}(z)}{
\sum_{k = 1}^\infty u_{k, p} C_k^(p / 2 - 1)(z)} is done to the first
\code{K_max} terms and then up to the index such that the first terms
leave unexplained the proportion \code{thre} of the norm of the whole series.
Setting \code{thre = 0} truncates to \code{K_max} terms exactly. If the
series only contains odd or even non-zero terms, then only \code{K_max / 2}
addends are \emph{effectively} taken into account in the first truncation.
}
\examples{
## Local alternatives diagnostics

loc_alt_diagnostic  <- function(p, type, thre = 1e-3, K_max = 1e3) {

  # Coefficients of the alternative
  uk <- cutoff_locdev(K_max = K_max, p = p, type = type, thre = thre,
                      N = 640)

  old_par <- par(mfrow = c(2, 2))

  # Construction of f
  z <- seq(-1, 1, l = 1e3)
  f <- function(z) f_locdev(z = z, p = p, uk = uk)
  plot(z, f(z), type = "l", xlab = expression(z), ylab = expression(f(z)),
       main = paste0("Local alternative f, ", type, ", p = ", p), log = "y")

  # Projected density on [-1, 1]
  f_proj <- function(z) rotasym::w_p(p = p - 1) * f(z) *
    (1 - z^2)^((p - 3) / 2)
  plot(z, f_proj(z), type = "l", xlab = expression(z),
       ylab = expression(omega[p - 1] * f(z) * (1 - z^2)^{(p - 3) / 2}),
       main = paste0("Projected density, ", type, ", p = ", p), log = "y",
       sub = paste("Integral:", round(con_f(f = f, p = p), 4)))

  # Quantile function for projected density
  mu <- c(rep(0, p - 1), 1)
  F_inv <- F_inv_from_f(f = f, p = p, K = 5e2)
  plot(F_inv, xlab = expression(x), ylab = expression(F^{-1}*(x)),
       main = paste0("Quantile function, ", type, ", p = ", p))

  # Sample from the alternative and plot the projected sample
  n <- 5e4
  samp <- r_locdev(n = n, mu = mu, f = f, kappa = 1, F_inv = F_inv)
  plot(z, f_proj(z), col = 2, type = "l",
       main = paste0("Simulated projected data, ", type, ", p = ", p),
       ylim = c(0, 1.75))
  hist(samp \%*\% mu, freq = FALSE, breaks = seq(-1, 1, l = 50), add = TRUE)

  par(old_par)

}
\donttest{
## Local alternatives for the PCvM test

loc_alt_diagnostic(p = 2, type = "PCvM")
loc_alt_diagnostic(p = 3, type = "PCvM")
loc_alt_diagnostic(p = 4, type = "PCvM")
loc_alt_diagnostic(p = 5, type = "PCvM")
loc_alt_diagnostic(p = 11, type = "PCvM")

## Local alternatives for the PAD test

loc_alt_diagnostic(p = 2, type = "PAD")
loc_alt_diagnostic(p = 3, type = "PAD")
loc_alt_diagnostic(p = 4, type = "PAD")
loc_alt_diagnostic(p = 5, type = "PAD")
loc_alt_diagnostic(p = 11, type = "PAD")

## Local alternatives for the PRt test

loc_alt_diagnostic(p = 2, type = "PRt")
loc_alt_diagnostic(p = 3, type = "PRt")
loc_alt_diagnostic(p = 4, type = "PRt")
loc_alt_diagnostic(p = 5, type = "PRt")
loc_alt_diagnostic(p = 11, type = "PRt")
}
}
\references{
García-Portugués, E., Navarro-Esteban, P., Cuesta-Albertos, J. A. (2023)
On a projection-based class of uniformity tests on the hypersphere.
\emph{Bernoulli}, 29(1):181--204. \doi{10.3150/21-BEJ1454}.

Prentice, M. J. (1978). On invariant tests of uniformity for directions and
orientations. \emph{The Annals of Statistics}, 6(1):169--176.
\doi{10.1214/aos/1176344075}
}
