#' R6 Class implementing down-sampling in shiny app
#'
#' @export
#' @docType class
#' @format An \code{R6::R6Class} object
#' @importFrom R6 R6Class
#' @importFrom dplyr %>%
#' @description
#' This class includes high-frequency original data, plotly figure, and shiny.
#' The plotly figure will be made by initializing the instance and using
#' \code{add_trace} method.
#' (note that the method is different from \code{plotly::add_trace}).
#' The easiest way to run shiny app is using \code{show_shiny} method.
#' Or, you can register the figures using \code{register_figures} method
#' then you can run the app using \code{run_server} method.
#' @examples
#'\donttest{
#' library(plotly)
#' data(noise_fluct)
#' p <- plot_ly(noise_fluct) %>%
#'   add_trace(x = ~sec, y = ~level, type = "scatter", mode = "lines")
#' d_app <- shiny_downsampler$new(p)
#' d_app$show_shiny()
#'}

# class decralation -------------------------------------------------------

shiny_downsampler <- R6::R6Class(
  "shiny_downsampler",
  inherit = abstract_downsampler,

# public members ----------------------------------------------------------

  public = list(
    #' @field shiny_session ShinySession R6 instance.
    shiny_session = NULL,
    #' @description
    #' Create a new downsampler
    #' @param figure,is_downsample,n_out,aggregator,legend_options,tz
    #' Arguments pass to the constructor of
    #' the \code{abstract_downsampler} class.
    initialize = function(
      figure = plotly::plot_ly(),
      is_downsample = TRUE,
      n_out = 1000L,
      # aggregator = min_max_ovlp_aggregator$new(),
      aggregator = eLTTB_aggregator$new(),
      legend_options = list(
        downsample_prefix = '<b style="color:sandybrown">[R]</b> ',
        downsample_suffix = "",
        is_aggsize_shown = TRUE,
        agg_prefix = ' <i style="color:#fc9944">~',
        agg_suffix = "</i>"
      ),
      tz = Sys.timezone()
    ) {


      if (is.null(figure)) self$figure <- plotly::plot_ly()
      super$initialize(
        figure,
        is_downsample,
        n_out,
        aggregator,
        legend_options,
        tz
      )
    },



    #' @description
    #' update the trace data according to the relayout order.
    #' @param relayout_order Named list.
    #' The list is generated by converging the dictionary
    #' obtained from \code{plotlyjs_relayout}.
    update_figure_data = function(relayout_order = list()) {

      trace_updates <- private$construct_update_data(relayout_order)
      if (is.null(trace_updates)) return()

      for (tr in trace_updates$trace) {
        if (length(tr$index) == 2) {
          if (!is.null(tr[["yupr"]]) && !is.null(tr[["ylwr"]])) {
            range_tr <- private$get_range_trace(tr)
            tr[["ylwr"]] <- NULL
            tr[["yupr"]] <- NULL
            for (elem in setdiff(names(range_tr), c("index", "uid"))) {
              self$figure$x$data[[tr$index[1] + 1]][[elem]] <- range_tr[[elem]]
            }
            for (elem in setdiff(names(tr), c("index", "uid"))) {
              self$figure$x$data[[tr$index[2] + 1]][[elem]] <- tr[[elem]]
            }
          } else {
            self$figure$x$data[[tr$index[1] + 1]][["x"]] <- NULL
            self$figure$x$data[[tr$index[1] + 1]][["y"]] <- NULL
            self$figure$x$data[[tr$index[1] + 1]][["text"]] <- NULL
            self$figure$x$data[[tr$index[1] + 1]][["hovertext"]] <- NULL
            self$figure$x$data[[tr$index[1] + 1]][["name"]] <- "-"
          }
        } else {
          for (elem in setdiff(names(tr), c("index", "uid"))) {
            self$figure$x$data[[tr$index + 1]][[elem]] <- tr[[elem]]
          }
        }
      }
    },

    #' @description
    #' Easily output the shiny app.
    #' @param shiny_options Named list.
    #' Arguments passed to \code{shinyApp} as the options.
    show_shiny = function(shiny_options = list()) {
      # Define UI
      ui <- fluidPage(
        plotlyOutput(outputId = "fig", width = "800px", height = "600px")
      )

      # Define server logic
      server <- function(input, output, session) {

        output$fig <- renderPlotly(self$figure)

        observeEvent(plotly::event_data("plotly_relayout"), {
          updatePlotlyH(
            session, "fig", plotly::event_data("plotly_relayout"), self
            )
        })

      }

      # Run the application
      shinyApp(ui = ui, server = server, options = shiny_options)
    }

  ), # end of the public
  private = list(
  )
) # end of the class definition
