% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table_spec.r
\name{create_table}
\alias{create_table}
\title{Create a table}
\usage{
create_table(
  x,
  show_cols = "all",
  use_attributes = "all",
  width = NULL,
  first_row_blank = FALSE,
  n_format = upcase_parens,
  headerless = FALSE,
  borders = "none",
  header_bold = FALSE,
  continuous = FALSE
)
}
\arguments{
\item{x}{The data frame or tibble from which to create the table object.}

\item{show_cols}{This parameter gives control over which columns in the 
input data to display on the report by default.  Valid values are
'all', 'none', a vector of quoted column names, or a vector of 
column positions.  'all' means show all columns, 
unless overridden by the column definitions.  
'none' means don't show any 
columns unless specified in the column definitions.  If a vector of column
names or positions is supplied, those columns will be shown in the report 
in the order specified, whether or not a definition is supplied. See the 
\code{\link{define}} function for additional information on how to
show/hide report columns.}

\item{use_attributes}{Whether or not to use any formatting attributes assigned
to the columns on the input data frame.  Valid values are 'all', 'none', or
a vector of attribute names to use.  Possible attributes that may be used
are 'label', 'format', 'width', and 'justify'.  By default, any of these
attribute values will be applied to the table.  For example, if you assign
a label to the 'label' attribute of a data frame column, pass that data 
frame into \code{create_table}, and don't override the label value on a 
\code{define} function, the label will appear as a column header on the
table.  The \code{use_attributes} parameter allows you to control this default
behavior, and use or ignore data frame attributes as desired.}

\item{width}{The expected width of the table in the report units of 
measure.  By default, the width setting is NULL, and columns will be sized
according to the width of the data and labels.  If the width parameter is 
set, the function will attempt to size the table to the specified width.
If the sum of the column widths is less than the specified width, the 
function will adjust the columns widths proportionally to fit the specified
width.  If the sum of the column widths is wider than the table width 
parameter value, the table width parameter will be ignored.}

\item{first_row_blank}{Whether to place a blank row under the table header.
Valid values are TRUE or FALSE.  Default is FALSE.}

\item{n_format}{The formatting function to apply to the header "N=" label. 
The default formatting function is \code{\link{upcase_parens}}.}

\item{headerless}{Whether to create a headerless table. A headerless
table displays the table data only. Default is FALSE, meaning the table
will have a header.}

\item{borders}{Whether and where to place a border. Valid values are 'top',
'bottom', 'left', 'right', 'all', 'none', 'outside', 'inside', and 'body'.  
Default is 'none'.  The 'left', 'right', 'outside', 'inside', and 'body'
border specifications only apply to RTF, HTML, PDF, and DOCX reports.  
The 'body' border 
specification means put borders around only the body of the table.}

\item{header_bold}{Whether or not the column headers on the tables should
be bolded.  Valid values are TRUE and FALSE.  The default is FALSE.}

\item{continuous}{If a table crosses multiple pages, it is normally
broken into a separate table for each page, and the titles and footnotes
are repeated on each page. When the "continuous" parameter is TRUE, the
table will instead be a single table, and the titles and footnotes will
not be repeated on each page.  This parameter currently only works for RTF
outputs.}
}
\description{
The \code{create_table} function creates a table object to which 
further specifications can be added.  The object can be added to a report
using the \code{\link{add_content}} function. The object is implemented as an 
S3 object of class 'table_spec'.
}
\details{
A table object is a container to hold information about a table.  The 
only required information for a table is the table data.  All other 
parameters and functions are optional.

By default, the table will display all columns in the data frame.  To change
this default, use the \code{show_cols} parameter.  Setting this parameter
to 'none' will display none of the columns in the data, unless they are
explicitly defined with a \code{\link{define}} function.  

The \code{show_cols} parameter also accepts a vector of column positions
or column names. When a vector is supplied, \code{create_table} will 
display only those columns on the report, in the order encountered in the 
vector.  The \code{show_cols} parameter is the only mechanism in 
\code{create_table} to modify the column order. Otherwise, modify the 
order prior to sending the data to \code{create_table} using the many options
available in Base R or supplemental packages.
}
\section{Setting Formatting Attributes}{

Formatting attributes can be controlled in three ways.  By default, formatting
attributes assigned to the data frame will be passed through to the 
reporting functions.  The reporting functions will recognize the 'label',
'format', 'width', and 'justify' attributes. In other words, you can control 
the column label, width, format, and alignment of your report columns simply by 
assigning those attributes to your data frame. The advantage of using
attributes assigned to data frame columns is that you can store those 
attributes permanently with the data frame, and those attributes will
not have to be re-specified for each report.  To ignore attributes assigned
to the data frame, set the \code{use_attributes} parameter to 'none'.

Secondly, attributes can be specified using the \code{\link{column_defaults}}
function.  This function allows the user to apply a default set of parameters
to one or more columns.  If no columns are specified in the \code{var} 
or \code{from} and \code{to} parameter of this function, the defaults 
will apply to all columns.  Any default parameter value can be overridden 
by the \code{\link{define}} function.

Lastly, the \code{\link{define}} function provides the most control over 
column parameters.  This function provides a significant amount of 
functionality that cannot be specified elsewhere.  See the 
\code{\link{define}} function for additional information.  The \code{define}
function will also override any formatting attributes assigned to the 
data frame, or anything set by the \code{column_defaults} function.
}

\section{Additional Functionality}{

The \code{create_table} function also provides the capabilities to create
a "headerless" table.  A headerless table is useful when combining two tables 
into one report.  The example below illustrates use of a headerless table.

Since the purpose of the \strong{reporter} package is to create statistical 
reports, the \code{create_table} function makes it easy to add population
counts to the table header.  These population counts are added to column
labels and spanning header labels using the \code{n} parameter on the 
\code{\link{define}} or \code{\link{spanning_header}} functions.  The 
population count is formatted according to the  
\code{n_format} parameter on \code{create_table}. The \strong{reporter} 
package provides four population count formatting functions.  
You may create your own formatting function 
if one of these functions does not meet your needs.  See 
\code{\link{upcase_parens}} for further details.
}

\examples{
library(reporter)
library(magrittr)

# Create temp file path
tmp <- file.path(tempdir(), "mtcars.txt")

#Subset cars data
dat <- mtcars[1:10, 1:7]

# Calculate means for all columns
dat_sum <- data.frame(all_cars = "All cars average", as.list(sapply(dat, mean)), 
                      stringsAsFactors = FALSE)

# Get vehicle names into first column
dat_mod <- data.frame(vehicle = rownames(dat), dat, stringsAsFactors = FALSE)
                      
# Create table for averages
tbl1 <- create_table(dat_sum) \%>\% 
        titles("Table 1.0", "MTCARS Sample Data") \%>\% 
        column_defaults(width = .5) \%>\% 
        define(all_cars, label = "", width = 2) \%>\% 
        define(mpg, format = "\%.1f") \%>\% 
        define(disp, format = "\%.1f") \%>\% 
        define(hp, format = "\%.0f") \%>\% 
        define(qsec, format = "\%.2f")

# Create table for modified data
tbl2 <- create_table(dat_mod, headerless = TRUE) \%>\% 
        column_defaults(width = .5) \%>\% 
        define(vehicle, width = 2) 

# Create the report object
rpt <- create_report(tmp) \%>\% 
  add_content(tbl1, align = "left", page_break = FALSE) \%>\% 
  add_content(tbl2, align = "left") 

# Write the report to the file system
write_report(rpt)

# Write report to console
writeLines(readLines(tmp, encoding = "UTF-8"))

#                                 Table 1.0
#                             MTCARS Sample Data
# 
#                             mpg    cyl   disp     hp   drat     wt   qsec
# -------------------------------------------------------------------------
# All cars average           20.4    5.8  208.6    123  3.538  3.128  18.58
# 
# Mazda RX4                    21      6    160    110    3.9   2.62  16.46
# Mazda RX4 Wag                21      6    160    110    3.9  2.875  17.02
# Datsun 710                 22.8      4    108     93   3.85   2.32  18.61
# Hornet 4 Drive             21.4      6    258    110   3.08  3.215  19.44
# Hornet Sportabout          18.7      8    360    175   3.15   3.44  17.02
# Valiant                    18.1      6    225    105   2.76   3.46  20.22
# Duster 360                 14.3      8    360    245   3.21   3.57  15.84
# Merc 240D                  24.4      4  146.7     62   3.69   3.19     20
# Merc 230                   22.8      4  140.8     95   3.92   3.15   22.9
# Merc 280                   19.2      6  167.6    123   3.92   3.44   18.3
# 
}
\seealso{
\code{\link{create_report}} to create a report, 
\code{\link{create_plot}} to create a plot,
\code{\link{create_text}} to create text content, and 
\code{\link{add_content}} to append content to a report.  Also see
the \code{\link{titles}}, \code{\link{footnotes}}, and \code{\link{page_by}}
functions to add those items to the table if desired.

Other table: 
\code{\link{column_defaults}()},
\code{\link{define}()},
\code{\link{print.table_spec}()},
\code{\link{spanning_header}()},
\code{\link{stub}()}
}
\concept{table}
