###########################################################################################################
###########################################################################################################
## 
## Name:		rags2ridges
## Authors:		Carel F.W. Peeters & Wessel N. van Wieringen
##			Molecular Biostatistics Unit
##			Dept. of Epidemiology & Biostatistics
##			VU University medical center
##			Amsterdam, the Netherlands
## Email:		cf.peeters@vumc.nl
## 
## Version:		1.3
## Last Update:	21/07/2014
## Description:	Ridge estimation, with supporting functions, for high-dimensional precision matrices
##
## Publications:	[1] Wessel N. van Wieringen & Carel F.W. Peeters (2014)
##			"Ridge Estimation of Inverse Covariance Matrices for High-Dimensional Data"
##			arXiv:1403.0904 [stat.ME]. 
## 			[2] Carel F.W. Peeters & Wessel N. van Wieringen (in preparation)
##			"The Spectral Condition Number Plot for Regularization Parameter Selection"
##
###########################################################################################################
###########################################################################################################




##---------------------------------------------------------------------------------------------------------
## 
## Hidden support functions
##
##---------------------------------------------------------------------------------------------------------

.trace <- function(M){
	#####################################################################################################
	# - Internal function to compute the trace of a matrix
	# - Faster support function (as opposed to 'matrix.trace') when input M is already forced to 'matrix'
	# - M > matrix input
	#####################################################################################################

	return(sum(diag(M)))
}



.is.int <- function(x, tolerance = .Machine$double.eps){
	#####################################################################################################
	# - Logical function that checks if a number is an integer within machine precision
	# - x         > input number
	# - tolerance > tolerance threshold for determining integer quality
	#####################################################################################################

	abs(x - round(x)) < tolerance
}



.LL <- function(S, P){
	#####################################################################################################
	# - Function that computes the value of the (negative) log-likelihood
	# - S > sample covariance matrix
	# - P > precision matrix (possibly regularized inverse of covariance or correlation matrix)
	#####################################################################################################

    	LL <- -log(det(P)) + .trace(S %*% P) 
    	return(LL)
}



.FrobeniusLoss <- function(O, P){
	#####################################################################################################
	# - Function computing Frobenius loss
	# - O > Estimated (possibly regularized) precision matrix
	# - P > True (population) precision matrix
	#####################################################################################################

	return(sum(abs(O - P)^2))
}



.QuadraticLoss <- function(O, C){
	#####################################################################################################
	# - Function computing Quadratic loss
	# - O > Estimated (possibly regularized) precision matrix
	# - C > True (population) covariance matrix
	#####################################################################################################

	return((sum(abs((O %*% C - diag(ncol(O))))^2)))
}



.eigShrink <- function(dVec, lambda, const = 0){
	#####################################################################################################
	# - Function that shrinks the eigenvalues in an eigenvector
	# - Shrinkage is that of rotation equivariant alternative ridge estimator
	# - Main use is in avoiding expensive matrix square root when choosing a target that leads to a
	#   rotation equivariant version of the alternative ridge estimator
	# - dVec   > numeric vector containing the eigenvalues of a matrix S
	# - lambda > penalty parameter
	# - const  > a constant, default = 0
	#####################################################################################################

	Evector <- (dVec - lambda * const)
	return(sqrt(lambda + Evector^2/4) + Evector/2)
}



.ridgeSi <- function(S, lambda, type = "Alt", target = default.target(S)){
	#####################################################################################################
	# - Hidden function that calculates Ridge estimators of a covariance matrix
	# - Function is mirror image main routine 'ridgeS'
	# - Main use is to circumvent (unnecessary) inversion (especially in 'conditionNumberPlot' function)
	# - S       > sample covariance matrix
	# - lambda  > penalty parameter (choose in accordance with type of Ridge estimator)
	# - type    > must be one of {"Alt", "ArchI", "ArchII"}, default = "Alt"
	# - Alt     > van Wieringen-Peeters alternative ridge estimator of a covariance matrix
	# - ArchI   > Archetypal I ridge estimator of a covariance matrix
	# - ArchII  > Archetypal II ridge estimator of a covariance matrix
	# - target  > target (precision terms) for Type I estimators, default = default.target(S)
	#
	# - NOTES:
	# - When type = "Alt" and target is p.d., one obtains the van Wieringen-Peeters type I estimator
	# - When type = "Alt" and target is null-matrix, one obtains the van Wieringen-Peeters type II est.
	# - When target is not the null-matrix it is expected to be p.d. for the vWP type I estimator
	# - The target is always expected to be p.d. in case of the archetypal I estimator
	# - When type = "Alt" and target is null matrix or of form c * diag(p), a rotation equivariant
	#   estimator ensues. In these cases the expensive matrix square root can be circumvented
	#####################################################################################################

	# Alternative estimator
	if (type == "Alt"){
		if (all(target == 0)){
			Spectral  <- eigen(S, symmetric = TRUE)
			Eigshrink <- .eigShrink(Spectral$values, lambda)
			C_Alt     <- Spectral$vectors %*% diag(Eigshrink) %*% t(Spectral$vectors)
			colnames(C_Alt) = rownames(C_Alt) <- colnames(S)
			return(C_Alt)
		} else if (all(target[!diag(nrow(target))] == 0) & (length(unique(diag(target))) == 1)){
			varPhi    <- unique(diag(target))
			Spectral  <- eigen(S, symmetric = TRUE)
			Eigshrink <- .eigShrink(Spectral$values, lambda, const = varPhi)
			C_Alt     <- Spectral$vectors %*% diag(Eigshrink) %*% t(Spectral$vectors)
			colnames(C_Alt) = rownames(C_Alt) <- colnames(S)
			return(C_Alt)
		} else {
			D     <- (S - lambda * target)
			C_Alt <- D/2 + sqrtm((D %*% D)/4 + lambda * diag(nrow(S)))
			return(C_Alt)
		}
	}

	# Archetypal I
	if (type == "ArchI"){
		C_ArchI <- (1-lambda) * S + lambda * solve(target)
		return(C_ArchI)
	}

	# Archetypal II
	if (type == "ArchII"){
		C_ArchII <- S + lambda * diag(nrow(S))
		return(C_ArchII)
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Support functions
##
##---------------------------------------------------------------------------------------------------------

symm <- function(M){
	#####################################################################################################
	# - Large objects that are symmetric sometimes fail to be recognized as such by R due to
	#   rounding under machine precision. This function symmetrizes for computational purposes
	#   matrices that are symmetric in numeric ideality
	# - M > symmetric (in numeric ideality) square matrix
	#####################################################################################################

	# Dependencies
	# require("base")

	if (!is.matrix(M)){
		stop("M should be a matrix")
	}
	else if (nrow(M) != ncol(M)){
		stop("M should be a square matrix")
	}
	else {
		# Symmetrize
		Msym <- (M + t(M))/2

		# Return
		return(Msym)
	}
}



adjacentMat <- function(M, diag = FALSE){
	#####################################################################################################
	# - Function that transforms a real matrix into an adjacency matrix
	# - Intended use: Turn sparsified precision matrix into an adjacency matrix for undirected graph
	# - M    > (sparsified precision) matrix
	# - diag > logical indicating if the diagonal elements should be retained
	#####################################################################################################

	# Dependencies
	# require("base")

	if (!is.matrix(M)){
		stop("M should be a matrix")
	}
	else if (nrow(M) != ncol(M)){
		stop("M should be square matrix")
	}
	else {
		# Create adjacency matrix
		AM <- M
		AM[AM != 0] <- 1
		diag(AM) <- 0

		if (diag){
			diag(AM) <- 1
		}

		# Return
		return(AM)
	}
}



covML <- function(Y){
	#####################################################################################################
	# - function that gives the maximum likelihood estimate of the covariance matrix
	# - Y   > (raw) data matrix, assumed to have variables in columns
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("stats")

	if (!is.matrix(Y)){
		stop("Input (Y) should be a matrix")
	}
	else {
		Ys  <- scale(Y, center = TRUE, scale = FALSE)
		Sml <- (t(Ys) %*% Ys)/nrow(Ys)
		return(Sml)
	}
}



evaluateS <- function(S, verbose = TRUE){
	#####################################################################################################
	# - Function evualuating various properties of an input matrix
	# - Intended use is to evaluate the various properties of what is assumed to be a covariance matrix
	# - Another use is to evaluate the various properties of a (regularized) precision matrix
	# - S       > sample covariance/correlation matrix or (regularized) precision matrix
	# - verbose > logical indicating if output should be printed on screen
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("stats")
	
	if (!is.matrix(S)){
		stop("S should be a matrix")
	} 
	else if (nrow(S) != ncol(S)){
		stop("S should be a square matrix")
	}
	else if (class(verbose) != "logical"){
		stop("Input (verbose) is of wrong class")
	} 
	else {
		Sproperties <- list()

		# Is S symmetric?
		Sproperties$symm <- isSymmetric(S)

		# Are eigenvalues S real and positive?
		evs                   <- eigen(S)$values 
		Sproperties$realEigen <- all(Im(evs) == 0) 
		Sproperties$posEigen  <- all(evs > 0)

		# Is S diagonally dominant?
		Sproperties$diagDom <- all(abs(cov2cor(S)[upper.tri(S)]) < 1)

		# Trace and determinant S
		Sproperties$trace <- sum(diag(S))
		Sproperties$det   <- det(S)

		# Spectral condition number S
		Sproperties$condNumber <- abs(max(evs) / min(evs))	

		if (verbose){
			cat("Properties of input matrix:", "\n")
			cat("----------------------------------------", "\n")
			cat(paste("       symmetric : ", Sproperties$symm, sep=""), "\n")
			cat(paste("eigenvalues real : ", Sproperties$realEigen, sep=""), "\n")
			cat(paste(" eigenvalues > 0 : ", Sproperties$posEigen, sep=""), "\n")
			cat(paste("  diag. dominant : ", Sproperties$diagDom, sep=""), "\n")
	 		cat("", "\n")
			cat(paste("           trace : ", round(Sproperties$trace, 5), sep=""), "\n")
			cat(paste("     determinant : ", round(Sproperties$det, 5), sep=""), "\n")
			cat(paste(" l2 cond. number : ", round(Sproperties$condNumber, 5), sep=""), "\n")
			cat("----------------------------------------", "\n")
		}

		# Return
		return(Sproperties)
	}
}



pcor <- function(P, pc = TRUE){
	#####################################################################################################
	# - Function computing partial correlation/standardized precision matrix from a precision matrix
	# - P  > precision matrix (possibly regularized inverse of covariance or correlation matrix)
	# - pc > logical indicating if the partial correlation matrix should be computed
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("stats")

	if (!is.matrix(P)){
		stop("P should be a matrix")
	}
	else if (!isSymmetric(P)){
		stop("P should be a symmetric matrix")
	}
	else {
		# Compute partial correlation matrix
		if (pc){
			P       <- -P
			diag(P) <- -diag(P)
			Pcor    <- cov2cor(P)
			return(Pcor)
		} 

		# Compute standardized precision matrix
		else {
			SP <- cov2cor(P)
			return(SP)
		}
	}
}



default.target <- function(S, type = "DAIE", fraction = 1e-04, const){ 
	#####################################################################################################
	# - Function that generates a (data-driven) default target for usage in ridge-type shrinkage estimation
	# - The target that is generated is to be understood in precision terms
	# - See function 'ridgeS'
	# S        > sample covariance/correlation matrix
	# type     > character determining the type of default target; default = "DAIE" (see notes below)
	# fraction > fraction of largest eigenvalue below which an eigenvalue is considered zero 
	#            Only when type = "DAIE"
	# const    > numeric constant that represents the partial variance. Only when type = "DCPV"
	#
	# Notes:
	# - The van Wieringen-Peeters type I estimator and the archetypal I estimator utilize a p.d. target
	# - DAIE: diagonal average inverse eigenvalue
	#   Diagonal matrix with average of inverse nonzero eigenvalues of S as entries 
	# - DUPV: diagonal unit partial variance
	#   Diagonal matrix with unit partial variance as entries (identity matrix)
	# - DAPV: diagonal average partial variance
	#   Diagonal matrix with average of inverse variances of S as entries
	# - DCPV: diagonal constant partial variance
	#   Diagonal matrix with constant partial variance as entries. Allows one to use other constant than
	#   [DAIE, DUPV, DAPV, and in a sense Null]
	# - DEPV: diagonal empirical partial variance
	#   Diagonal matrix with the inverse variances of S as entries
	# - Null: Null matrix
	#   Matrix with only zero entries
	# - All but DEPV and Null lead to rotation equivariant alternative and archetype I ridge estimators
	# - Null also leads to a rotation equivariant alternative Type II estimator
	#####################################################################################################

	# Dependencies
	# require("base")

	if (!is.matrix(S)){
		stop("Input (S) should be a matrix")
	} 
	else if (!isSymmetric(S)){
		stop("Input (S) should be a symmetric matrix")
	}
	else if (class(type) != "character"){
		stop("Input (type) is of wrong class")
	}
	else if (!(type %in% c("DAIE", "DUPV", "DAPV", "DCPV", "DEPV", "Null"))){
		stop("type should be one of {'DAIE', 'DUPV', 'DAPV', 'DCPV', 'DEPV', 'Null'}")
	}
	else {
		# Compute and return a default target matrix
		# Diagonal matrix with average of inverse nonzero eigenvalues of S as entries 
		if (type == "DAIE"){
			if (class(fraction) != "numeric"){
				stop("Input (fraction) is of wrong class")
			} else if (length(fraction) != 1){
				stop("Length fraction must be one")
			} else if (fraction < 0 | fraction > 1){
				stop("Input (fraction) is expected to be in the interval [0,1]")
			} else {
				Eigs   <- eigen(S, symmetric = TRUE, only.values = TRUE)$values
				const  <- mean(1/(Eigs[Eigs >= Eigs[1]*fraction]))
				target <- const * diag(ncol(S))
			}
		}

		# Diagonal matrix with unit partial variance as entries
		if (type == "DUPV"){
			target <- diag(ncol(S))
		}

		# Diagonal matrix with average empirical partial variances as entries
		if (type == "DAPV"){
			apv    <- mean(1/diag(S))
			target <- apv * diag(ncol(S))
		}

		# Diagonal matrix with constant partial variance as entries
		if (type == "DCPV"){
			if (class(const) != "numeric"){
				stop("Input (const) is of wrong class")
			} else if (length(const) != 1){
				stop("Length 'const' must be one")
			} else if (const <= 0 | const > .Machine$double.xmax){
				stop("Input (const) is expected to be in the interval (0, Inf)")
			} else {
				target <- const * diag(ncol(S))
			}
		}

		# Diagonal matrix with empirical partial variances as entries
		if (type == "DEPV"){
			target <- diag(1/diag(S))
		}

		# Null matrix
		if (type == "Null"){
			target <- matrix(0, ncol(S), nrow(S))
		}

		# Return
		colnames(target) = rownames(target) <- rownames(S)
		return(target)
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Function for Ridge Estimators of the Precision matrix
##
##---------------------------------------------------------------------------------------------------------

ridgeS <- function(S, lambda, type = "Alt", target = default.target(S)){
	#####################################################################################################
	# - Function that calculates Ridge estimators of a precision matrix
	# - S       > sample covariance matrix
	# - lambda  > penalty parameter (choose in accordance with type of Ridge estimator)
	# - type    > must be one of {"Alt", "ArchI", "ArchII"}, default = "Alt"
	# - Alt     > van Wieringen-Peeters alternative ridge estimator of a precision matrix
	# - ArchI   > Archetypal I ridge estimator of a precision matrix
	# - ArchII  > Archetypal II ridge estimator of a precision matrix
	# - target  > target (precision terms) for Type I estimators, default = default.target(S)
	#
	# - NOTES:
	# - When type = "Alt" and target is p.d., one obtains the van Wieringen-Peeters type I estimator
	# - When type = "Alt" and target is null-matrix, one obtains the van Wieringen-Peeters type II est.
	# - When target is not the null-matrix it is expected to be p.d. for the vWP type I estimator
	# - The target is always expected to be p.d. in case of the archetypal I estimator
	# - When type = "Alt" and target is null matrix or of form c * diag(p), a rotation equivariant
	#   estimator ensues. In these cases the expensive matrix square root can be circumvented
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("expm")

	if (!isSymmetric(S)){
		stop("S should be a covariance matrix")
	}
	else if (lambda <= 0){
		stop("lambda should be positive")
	}
	else if (!(type %in% c("Alt", "ArchI", "ArchII"))){
		stop("type should be one of {'Alt', 'ArchI', 'ArchII'}")
	}
	else{
		# Calculate Ridge estimator
		# Alternative estimator
		if (type == "Alt"){
			if (!isSymmetric(target)){
				stop("Shrinkage target should be symmetric")
			} else if (dim(target)[1] != dim(S)[1]){
				stop("S and target should be of the same dimension")
			} else if (!all(target == 0) & any(eigen(target, symmetric = TRUE, only.values = T)$values <= 0)){
				stop("When target is not a null-matrix it should be p.d. for this type of ridge estimator")
			} else if (all(target == 0)){
				Spectral  <- eigen(S, symmetric = TRUE)
				Eigshrink <- .eigShrink(Spectral$values, lambda)
				P_Alt     <- solve(Spectral$vectors %*% diag(Eigshrink) %*% t(Spectral$vectors))
				colnames(P_Alt) = rownames(P_Alt) <- colnames(S)
				return(P_Alt)
			} else if (all(target[!diag(nrow(target))] == 0) & (length(unique(diag(target))) == 1)){
				varPhi    <- unique(diag(target))
				Spectral  <- eigen(S, symmetric = TRUE)
				Eigshrink <- .eigShrink(Spectral$values, lambda, const = varPhi)
				P_Alt     <- solve(Spectral$vectors %*% diag(Eigshrink) %*% t(Spectral$vectors))
				colnames(P_Alt) = rownames(P_Alt) <- colnames(S)
				return(P_Alt)
			} else {
				E     <- (S - lambda * target)
				P_Alt <- solve(E/2 + sqrtm((E %*% E)/4 + lambda * diag(nrow(S))))
				return(P_Alt)
			}
		}

		# Archetypal I
		if (type == "ArchI"){
			if (lambda > 1){
				stop("lambda should be in (0,1] for this type of Ridge estimator")
			} else if (!isSymmetric(target)){
				stop("Shrinkage target should be symmetric")
			} else if (dim(target)[1] != dim(S)[1]){
				stop("S and target should be of the same dimension")
			} else if (any(eigen(target, symmetric = TRUE, only.values = T)$values <= 0)){
				stop("Target should always be p.d. for this type of ridge estimator")
			} else {
				P_ArchI <- solve((1-lambda) * S + lambda * solve(target))
				return(P_ArchI)
			}
		}

		# Archetypal II
		if (type == "ArchII"){
			P_ArchII <- solve(S + lambda * diag(nrow(S)))
			return(P_ArchII)
		}
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Functions for Penalty Parameter selection
##
##---------------------------------------------------------------------------------------------------------

optPenalty.LOOCV <- function(Y, lambdaMin, lambdaMax, step, type = "Alt", target = default.target(covML(Y)), 
                         output = "light", graph = TRUE, verbose = TRUE){ 
	#####################################################################################################
	# - Function that selects the optimal penalty parameter by leave-one-out cross-validation
	# - Y           > (raw) Data matrix, variables in columns
	# - lambdaMin   > minimum value penalty parameter (dependent on 'type')
	# - lambdaMax   > maximum value penalty parameter (dependent on 'type')
	# - step        > determines the coarseness in searching the grid [lambdaMin, lambdaMax]
	# - type        > must be one of {"Alt", "ArchI", "ArchII"}, default = "Alt"
	# - target      > target (precision terms) for Type I estimators, default = default.target(covML(Y))
	# - output      > must be one of {"all", "light"}, default = "light"
	# - graph       > Optional argument for visualization optimal penalty selection, default = TRUE
	# - verbose     > logical indicating if intermediate output should be printed on screen
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("stats")
	# require("graphics")

	if (class(verbose) != "logical"){
		stop("Input (verbose) is of wrong class")
	} 
	if (verbose){
		cat("Perform input checks...", "\n")
	}
	if (!is.matrix(Y)){
		stop("Y should be a matrix")
	} 
	else if (class(lambdaMin) != "numeric"){
		stop("Input (lambdaMin) is of wrong class")
	} 
	else if (length(lambdaMin) != 1){
		stop("lambdaMin must be a scalar")
	} 
	else if (lambdaMin <= 0){
		stop("lambdaMin must be positive")
	} 
	else if (class(lambdaMax) != "numeric"){
		stop("Input (lambdaMax) is of wrong class")
	} 
	else if (length(lambdaMax) != 1){
		stop("lambdaMax must be a scalar")
	} 
	else if (lambdaMax <= lambdaMin){
		stop("lambdaMax must be larger than lambdaMin")
	}
	else if (class(step) != "numeric"){
		stop("Input (step) is of wrong class")
	}
	else if (!.is.int(step)){
		stop("step should be integer")
	}
	else if (step <= 0){
		stop("step should be a positive integer")
	}
	else if (!(output %in% c("all", "light"))){
		stop("output should be one of {'all', 'light'}")
	}
	else if (class(graph) != "logical"){
		stop("Input (graph) is of wrong class")
	}
	else {
		# Set preliminaries
		LLs     <- numeric()
		lambdas <- seq(lambdaMin, lambdaMax, len = step)

		# Calculate CV scores
		if (verbose){cat("Calculating cross-validated negative log-likelihoods...", "\n")}
		for (k in 1:length(lambdas)){
			slh <- numeric()
        		for (i in 1:nrow(Y)){
				S   <- covML(Y[-i,])
				slh <- c(slh, .LL(t(Y[i,,drop = F]) %*% Y[i,,drop = F], ridgeS(S, lambdas[k], type = type, target = target)))
			}
			
			LLs <- c(LLs, mean(slh))
			if (verbose){cat(paste("lambda = ", lambdas[k], " done", sep = ""), "\n")}
		}

		# Visualization
		optLambda <- min(lambdas[which(LLs == min(LLs))])
		if (graph){
			if (type == "Alt"){Main = "Alternative ridge estimator"}
			if (type == "ArchI"){Main = "Archetypal I ridge estimator"}
			if (type == "ArchII"){Main = "Archetypal II ridge estimator"}
			plot(log(lambdas), type = "l", log(LLs), axes = FALSE, xlab = "ln(penalty value)", ylab = "ln(cross-validated neg. log-likelihood)", main = Main)
			axis(2, ylim = c(log(min(LLs)),log(max(LLs))), col = "black", lwd = 1)
			axis(1, col = "black", lwd = 1)
			abline(h = log(min(LLs)), v = log(optLambda), col = "red")
			legend("topright", legend = c(paste("min. LOOCV -LL: ", round(min(LLs),3), sep = ""), 
			paste("Opt. penalty: ", optLambda, sep = "")), cex = .8)
		}

		# Return
		S <- covML(Y)
		if (output == "all"){
			return(list(optLambda = optLambda, optPrec = ridgeS(S, optLambda, type = type, target = target), lambdas = lambdas, LLs = LLs))
		}
		if (output == "light"){
			return(list(optLambda = optLambda, optPrec = ridgeS(S, optLambda, type = type, target = target)))
		}
	}
}



optPenalty.aLOOCV <- function(Y, lambdaMin, lambdaMax, step, type = "Alt", target = default.target(covML(Y)), 
                              output = "light", graph = TRUE, verbose = TRUE){
	#####################################################################################################
	# - Function that selects the optimal penalty parameter by approximate leave-one-out cross-validation
	# - Y           > (raw) Data matrix, variables in columns
	# - lambdaMin   > minimum value penalty parameter (dependent on 'type')
	# - lambdaMax   > maximum value penalty parameter (dependent on 'type')
	# - step        > determines the coarseness in searching the grid [lambdaMin, lambdaMax]
	# - type        > must be one of {"Alt", "ArchI", "ArchII"}, default = "Alt"
	# - target      > target (precision terms) for Type I estimators, default = default.target(covML(Y))
	# - output      > must be one of {"all", "light"}, default = "light"
	# - graph       > Optional argument for visualization optimal penalty selection, default = TRUE
	# - verbose     > logical indicating if intermediate output should be printed on screen
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("graphics")

	if (class(verbose) != "logical"){
		stop("Input (verbose) is of wrong class")
	} 
	if (verbose){
		cat("Perform input checks...", "\n")
	}
	if (!is.matrix(Y)){
		stop("Y should be a matrix")
	} 
	else if (class(lambdaMin) != "numeric"){
		stop("Input (lambdaMin) is of wrong class")
	} 
	else if (length(lambdaMin) != 1){
		stop("lambdaMin must be a scalar")
	} 
	else if (lambdaMin <= 0){
		stop("lambdaMin must be positive")
	} 
	else if (class(lambdaMax) != "numeric"){
		stop("Input (lambdaMax) is of wrong class")
	} 
	else if (length(lambdaMax) != 1){
		stop("lambdaMax must be a scalar")
	} 
	else if (lambdaMax <= lambdaMin){
		stop("lambdaMax must be larger than lambdaMin")
	}
	else if (class(step) != "numeric"){
		stop("Input (step) is of wrong class")
	}
	else if (!.is.int(step)){
		stop("step should be integer")
	}
	else if (step <= 0){
		stop("step should be a positive integer")
	}
	else if (!(output %in% c("all", "light"))){
		stop("output should be one of {'all', 'light'}")
	}
	else if (class(graph) != "logical"){
		stop("Input (graph) is of wrong class")
	}
	else {
		# Set preliminaries
		S       <- covML(Y)
		n       <- nrow(Y)
		lambdas <- seq(lambdaMin, lambdaMax, len = step)
		aLOOCVs <- numeric()

		# Calculate approximate LOOCV scores
		if (verbose){cat("Calculating approximate LOOCV scores...", "\n")}
		if (type == "Alt" & all(target == 0)){
			if (!isSymmetric(target)){
				stop("Shrinkage target should be symmetric")
			} else if (dim(target)[1] != dim(S)[1]){
				stop("Covariance matrix based on data input (Y) and target should be of the same dimension")
			} else {
				Spectral <- eigen(S, symmetric = TRUE)
				for (k in 1:length(lambdas)){
					Eigshrink <- .eigShrink(Spectral$values, lambdas[k])
					P         <- solve(Spectral$vectors %*% diag(Eigshrink) %*% t(Spectral$vectors))
					nLL       <- .5 * .LL(S, P)
					isum      <- numeric()

        				for (i in 1:n){
						S1   <- t(Y[i,,drop = FALSE]) %*% Y[i,,drop = FALSE]
						isum <- c(isum, sum((solve(P) - S1) * (P %*% (S - S1) %*% P)))
					}

					aLOOCVs <- c(aLOOCVs, nLL + 1/(2 * n^2 - 2 * n) * sum(isum))
					if (verbose){cat(paste("lambda = ", lambdas[k], " done", sep = ""), "\n")}
				}
			}					
		} else if (type == "Alt" & all(target[!diag(nrow(target))] == 0) & (length(unique(diag(target))) == 1)){
			if (!isSymmetric(target)){
				stop("Shrinkage target should be symmetric")
			} else if (dim(target)[1] != dim(S)[1]){
				stop("Covariance matrix based on data input (Y) and target should be of the same dimension")
			} else if (any(diag(target) <= 0)){
				stop("target should be p.d.")
			} else {
				varPhi   <- unique(diag(target))
				Spectral <- eigen(S, symmetric = TRUE)
				for (k in 1:length(lambdas)){
					Eigshrink <- .eigShrink(Spectral$values, lambdas[k], const = varPhi)
					P         <- solve(Spectral$vectors %*% diag(Eigshrink) %*% t(Spectral$vectors))
					nLL       <- .5 * .LL(S, P)
					isum      <- numeric()

        				for (i in 1:n){
						S1   <- t(Y[i,,drop = FALSE]) %*% Y[i,,drop = FALSE]
						isum <- c(isum, sum((solve(P) - S1) * (P %*% (S - S1) %*% P)))
					}

					aLOOCVs <- c(aLOOCVs, nLL + 1/(2 * n^2 - 2 * n) * sum(isum))
					if (verbose){cat(paste("lambda = ", lambdas[k], " done", sep = ""), "\n")}
				}
			}
		} else {
			for (k in 1:length(lambdas)){
				P    <- ridgeS(S, lambdas[k], type = type, target = target)
				nLL  <- .5 * .LL(S, P)
				isum <- numeric()

        			for (i in 1:n){
					S1   <- t(Y[i,,drop = FALSE]) %*% Y[i,,drop = FALSE]
					isum <- c(isum, sum((solve(P) - S1) * (P %*% (S - S1) %*% P)))
				}

				aLOOCVs <- c(aLOOCVs, nLL + 1/(2 * n^2 - 2 * n) * sum(isum))
				if (verbose){cat(paste("lambda = ", lambdas[k], " done", sep = ""), "\n")}
			}
		}

		# Visualization
		optLambda <- min(lambdas[which(aLOOCVs == min(aLOOCVs))])
		if (graph){
			if (type == "Alt"){Main = "Alternative ridge estimator"}
			if (type == "ArchI"){Main = "Archetypal I ridge estimator"}
			if (type == "ArchII"){Main = "Archetypal II ridge estimator"}
			plot(log(lambdas), type = "l", log(aLOOCVs), axes = FALSE, xlab = "ln(penalty value)", ylab = "ln(Approximate LOOCV neg. log-likelihood)", main = Main)
			axis(2, ylim = c(log(min(aLOOCVs)),log(max(aLOOCVs))), col = "black", lwd = 1)
			axis(1, col = "black", lwd = 1)
			abline(h = log(min(aLOOCVs)), v = log(optLambda), col = "red")
			legend("topright", legend = c(paste("min. approx. LOOCV -LL: ", round(min(aLOOCVs),3), sep = ""), 
			paste("Opt. penalty: ", optLambda, sep = "")), cex = .8)
		}

		# Return
		if (output == "all"){
			return(list(optLambda = optLambda, optPrec = ridgeS(S, optLambda, type = type, target = target), lambdas = lambdas, aLOOCVs = aLOOCVs))
		}
		if (output == "light"){
			return(list(optLambda = optLambda, optPrec = ridgeS(S, optLambda, type = type, target = target)))
		}
	}
}



conditionNumberPlot <- function(S, lambdaMin, lambdaMax, step, type = "Alt", target = default.target(S),
				        norm = "2", rlDist = FALSE, verticle = FALSE, value, main = TRUE, 
                                nOutput = FALSE, verbose = TRUE){
	#####################################################################################################
	# - Function that visualizes the spectral condition number against the regularization parameter
	# - Can be used to heuristically determine the (minimal) value of the penalty parameter
	# - The ridges are rotation equivariant, meaning they work by shrinking the eigenvalues
	# - Maximum shrinkage implies that all eigenvalues will be equal
	# - Ratio of maximum and minimum eigenvalue of P can then function as a heuristic 
	# - It's point of stabilization can give an acceptable value for the penalty
	# - The ratio boils down to the (spectral) condition number of a matrix
	# - S         > sample covariance/correlation matrix
	# - lambdaMin > minimum value penalty parameter (dependent on 'type')
	# - lambdaMax > maximum value penalty parameter (dependent on 'type')
	# - step      > determines the coarseness in searching the grid [lambdaMin, lambdaMax]
	# - type      > must be one of {"Alt", "ArchI", "ArchII"}, default = "Alt"
	# - target    > target (precision terms) for Type I estimators, default = default.target(S)
	# - norm      > indicates the norm under which the condition number is to be estimated
	# - rlDist    > logical indicating if relative distance to set of singular matrices should also be
	#               plotted. Default = FALSE
	# - verticle  > optional argument for visualization verticle line in graph output, default = FALSE
	#               Can be used to indicate the value of, e.g., the optimal penalty as indicated by some
	#               routine. Can be used to assess if this optimal penalty will lead to a 
	#               well-conditioned estimate
	# - value     > indicates constant on which to base verticle line when verticle = TRUE
	# - main      > logical indicating if plot should contain type of estimator as main title
	# - nOutput   > logical indicating if numeric output should be given (lambdas and condition numbers)
	# - verbose   > logical indicating if intermediate output should be printed on screen
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("graphics")
	# require("Hmisc")

	if (class(verbose) != "logical"){
		stop("Input (verbose) is of wrong class")
	} 
	if (verbose){
		cat("Perform input checks...", "\n")
	}
	if (!is.matrix(S)){
		stop("S should be a matrix")
	} 
	else if (!isSymmetric(S)){
		stop("S should be a covariance matrix")
	}
	else if (class(lambdaMin) != "numeric"){
		stop("Input (lambdaMin) is of wrong class")
	} 
	else if (length(lambdaMin) != 1){
		stop("lambdaMin must be a scalar")
	} 
	else if (lambdaMin <= 0){
		stop("lambdaMin must be positive")
	} 
	else if (class(lambdaMax) != "numeric"){
		stop("Input (lambdaMax) is of wrong class")
	} 
	else if (length(lambdaMax) != 1){
		stop("lambdaMax must be a scalar")
	} 
	else if (lambdaMax <= lambdaMin){
		stop("lambdaMax must be larger than lambdaMin")
	}
	else if (class(step) != "numeric"){
		stop("Input (step) is of wrong class")
	}
	else if (!.is.int(step)){
		stop("step should be integer")
	}
	else if (step <= 0){
		stop("step should be a positive integer")
	}
	else if (!(type %in% c("Alt", "ArchI", "ArchII"))){
		stop("type should be one of {'Alt', 'ArchI', 'ArchII'}")
	}
	else if (!isSymmetric(target)){
		stop("Shrinkage target should be symmetric")
	} 
	else if (dim(target)[1] != dim(S)[1]){
		stop("S and target should be of the same dimension")
	}
	else if (type == "Alt" & !all(target == 0) & any(eigen(target, symmetric = TRUE, only.values = T)$values <= 0)){
		stop("When target is not a null-matrix it should be p.d.")
	}
	else if (type == "ArchI" & lambdaMax > 1){
		stop("lambda should be in (0,1] for this type of Ridge estimator")
	}
	else if (type == "ArchI" & any(eigen(target, symmetric = TRUE, only.values = T)$values <= 0)){
		stop("Target should be p.d.")
	}
	else if (!(norm %in% c("2", "1"))){
		stop("norm should be one of {'2', '1'}")
	}
	else if (class(rlDist) != "logical"){
		stop("Input (rlDist) is of wrong class")
	}
	else if (class(verticle) != "logical"){
		stop("Input (verticle) is of wrong class")
	}
	else if (class(main) != "logical"){
		stop("Input (main) is of wrong class")
	}
	else if (class(nOutput) != "logical"){
		stop("Input (nOutput) is of wrong class")
	}
	else {
		# Set preliminaries
		lambdas <- seq(lambdaMin, lambdaMax, len = step)
		condNR  <- numeric()

		if (norm == "2"){
			# Calculate spectral condition number ridge estimate on lambda grid
			if (verbose){cat("Calculating spectral condition numbers...", "\n")}
			if (type == "Alt" & all(target == 0)){
				Spectral <- eigen(S, symmetric = TRUE, only.values = TRUE)$values
				for (k in 1:length(lambdas)){
					Eigshrink <- .eigShrink(Spectral, lambdas[k])
					condNR[k] <- as.numeric(max(Eigshrink)/min(Eigshrink))
				}
			} else if (type == "Alt" & all(target[!diag(nrow(target))] == 0) & (length(unique(diag(target))) == 1)){
				varPhi   <- unique(diag(target))
				Spectral <- eigen(S, symmetric = TRUE, only.values = TRUE)$values
				for (k in 1:length(lambdas)){
					Eigshrink <- .eigShrink(Spectral, lambdas[k], const = varPhi)
					condNR[k] <- as.numeric(max(Eigshrink)/min(Eigshrink))
				}
			} else {
				for (k in 1:length(lambdas)){
					P         <- .ridgeSi(S, lambdas[k], type = type, target = target)
					Eigs      <- eigen(P, symmetric = TRUE, only.values = TRUE)$values
					condNR[k] <- as.numeric(max(Eigs)/min(Eigs))
					if (verbose){cat(paste("lambda = ", lambdas[k], " done", sep = ""), "\n")}
				}
			}
		}

		if (norm == "1"){
			# Calculate approximation to condition number under 1-norm
			if (verbose){cat("Approximating condition number under 1-norm...", "\n")}
			for (k in 1:length(lambdas)){
				P         <- .ridgeSi(S, lambdas[k], type = type, target = target)
				condNR[k] <- as.numeric(1/rcond(P, norm = "O"))
				if (verbose){cat(paste("lambda = ", lambdas[k], " done", sep = ""), "\n")}
			}
		}

		# Visualization
		if (main){
			if (type == "Alt"){Main = "Alternative ridge estimator"}
			if (type == "ArchI"){Main = "Archetypal I ridge estimator"}
			if (type == "ArchII"){Main = "Archetypal II ridge estimator"}
		}
		if (!main){Main = " "}
		if (norm == "2"){Ylab = "spectral condition number"}
		if (norm == "1"){Ylab = "condition number under 1-norm"}
		if (rlDist){par(mar = c(5,4,4,5)+.1)}
		plot(log(lambdas), type = "l", condNR, axes = FALSE, col = "blue4", xlab = "ln(penalty value)", ylab = Ylab, main = Main)
		axis(2, ylim = c(0,max(condNR)), col = "black", lwd = 1)
		axis(1, col = "black", lwd = 1)
		minor.tick(nx = 10, ny = 0, tick.ratio = .4)
		if (rlDist){
			RlDist <- 1/condNR
			par(new=TRUE)
			plot(log(lambdas), RlDist,, axes = FALSE, type = "l", col = "green3", xaxt = "n", yaxt = "n", xlab = "", ylab = "")
			axis(4, col = "black", lwd = 1)
			mtext("relative distance to singular matrix", side = 4, line = 3)
			legend("top", col=c("blue4","green3"), lty = 1, legend = c("Condition number", "Relative distance"), cex = .8)
		}
		if (verticle){
			if (missing(value)){
				stop("Need to specify input (value)")
			} else if (class(value) != "numeric"){
				stop("Input (value) is of wrong class")
			} else if (length(value) != 1){
				stop("Input (value) must be a scalar")
			} else if (value <= 0){
				stop("Input (value) must be positive")
			} else {
				abline(v = log(value), col = "red")
			}
		}

		# Possible output
		if (nOutput){
			return(list(lambdas = lambdas, conditionNumbers = condNR))
		}
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Test for Vanishing Partial Correlations
##
##---------------------------------------------------------------------------------------------------------

sparsify <- function(P, threshold = c("absValue", "localFDR"), absValueCut = .25, FDRcut = .8, verbose = TRUE){
	#####################################################################################################
	# - Function that sparsifies/determines support of a partial correlation matrix
	# - Support can be determined by absolute value thresholding or by local FDRs thresholding
	# - Local FDR operates on the nonredundant non-diagonal elements of a partial correlation matrix
	# - Function is to some extent a wrapper around certain 'GeneNet' and 'fdrtool' functions
	# - P           > (possibly shrunken) precision matrix
	# - threshold   > signifies type of thresholding: based on either absolute value or local FDR testing
	# - absValueCut > cut-off for partial correlation elements selection based on absolute value 
	#                 thresholding. Only when threshold = 'absValue'. Default = .25
	# - FDRcut      > cut-off for partial correlation element selection based on local FDR thresholding
	#                 Only when threshold = 'localFDR'. Default = .8
	# - verbose     > logical indicating if intermediate output should be printed on screen
	#                 Only when threshold = 'localFDR'. Default = TRUE
	#
	# NOTES:
	# - Input (P) may be the partial correlation matrix or the standardized precision matrix. These are 
	#   identical up to the signs of off-diagonal elements. Either can be used as it has no effect
	#   on the thresholding operator and the ensuing sparsified result
	# - Input (P) may also be the unstandardized precision matrix. The function converts it to the 
	#   partial correlation matrix
	# - The function evualtes if the input (P) is a partial correlation/standardized precision matrix or 
	#   an unstandardized precision matrix. If the input amounts to the latter both the sparsified partial
	#   correlation matrix and the corresponding sparsified precision matrix are given as output.
	#   Otherwise, the ouput consists of the sparsified partial correlation/standardized precision matrix
	#####################################################################################################

	# Dependencies
	# require("base")
	# require("stats")
	# require("corpcor")
	# require("longitudinal")
	# require("fdrtool")
	# require("GeneNet")

	if (!is.matrix(P)){
		stop("P should be a matrix")
	}
	else if (!isSymmetric(P)){
		stop("P should be a symmetric matrix")
	}
	else if (!evaluateS(P, verbose = FALSE)$posEigen){
		stop("P is expected to be positive definite")
	}
	else if (missing(threshold)){
		stop("Need to specify type of sparsification ('absValue' or 'localFDR')")
	}
	else if (!(threshold %in% c("absValue", "localFDR"))){
		stop("Input (threshold) should be one of {'absValue', 'localFDR'}")
	}
	else {
		# Obtain partial correlation matrix
		if (all(length(unique(diag(P))) == 1 & unique(diag(P)) == 1)){
			stan = TRUE
			PC  <- P
		} else {
			stan = FALSE
      		PC  <- pcor(P)
		}

		# Obtain sparsified matrix
		if (threshold == "absValue"){
			if (class(absValueCut) != "numeric"){
				stop("Input (absValueCut) is of wrong class")
			} else if (length(absValueCut) != 1){
				stop("Input (absValueCut) must be a scalar")
			} else if (absValueCut < 0 | absValueCut > 1){
				stop("Input (absValueCut) must be in the interval [0,1]")
			} else {
				PC0 <- PC
				PC0[!(abs(PC0) >= absValueCut)] <- 0
				if (!stan){
					P0 <- P
					P0[PC0 == 0] <- 0
				}
			}
		}

		if (threshold == "localFDR"){
			if (class(FDRcut) != "numeric"){
				stop("Input (FDRcut) is of wrong class")
			} else if (length(FDRcut) != 1){
				stop("Input (FDRcut) must be a scalar")
			} else if (FDRcut < 0 | FDRcut > 1){
				stop("Input (FDRcut) must be in the interval [0,1]")
			} else if (class(verbose) != "logical"){
				stop("Input (verbose) is of wrong class")
			} else {
				if (verbose){
					PCtest <- ggm.test.edges(PC, fdr = TRUE, direct = FALSE, plot = TRUE)
					PCnet  <- extract.network(PCtest, cutoff.ggm = FDRcut); print(PCnet)
				} else {
					PCtest <- ggm.test.edges(PC, fdr = TRUE, direct = FALSE, plot = FALSE)
					PCnet  <- extract.network(PCtest, cutoff.ggm = FDRcut)
				}
				EdgeSet <- PCnet[,2:3]
				PC0     <- diag(nrow(PC))
				for(k in 1:dim(EdgeSet)[1]){
					PC0[EdgeSet[k,1],EdgeSet[k,2]] = PC0[EdgeSet[k,2],EdgeSet[k,1]] <- PC[EdgeSet[k,1],EdgeSet[k,2]]
				} 
				if (!stan){
					P0 <- P
					P0[PC0 == 0] <- 0
				}
			}
		}

		# Return
		if (stan){
			colnames(PC0) = rownames(PC0) <- colnames(P)
			return(PC0)
		}
		if (!stan){
			colnames(PC0) = rownames(PC0) <- colnames(P)
			colnames(P0)  = rownames(P0)  <- colnames(P)
			return(list(sparseParCor = PC0, sparsePrecision = P0))
		}
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Functions for Loss/Entropy/Fit Evaluation
##
##---------------------------------------------------------------------------------------------------------

loss <- function(E, T, precision = TRUE, type = c("frobenius", "quadratic")){
	#####################################################################################################
	# - Function evualuating various loss functions on the precision
	# - E         > Estimated (possibly regularized) precision matrix
	# - T         > True (population) covariance or precision matrix
	# - precision > Logical indicating if T is a precision matrix (when TRUE)
	# - type      > character indicating which loss function is to be used
	#####################################################################################################

	if (!is.matrix(E)){
		stop("Input (E) is of wrong class")
	} 
	else if (!isSymmetric(E)){
		stop("E should be a symmetric matrix")
	}
	else if (!is.matrix(T)){
		stop("Input (T) is of wrong class")
	} 
	else if (!isSymmetric(T)){
		stop("T should be a symmetric matrix")
	}
	else if (dim(E)[1] != dim(T)[1]){
		stop("E and T should be of the same dimension")
	}
	else if (class(precision) != "logical"){
		stop("Input (precision) is of wrong class")
	}
	else if (missing(type)){
		stop("Need to specify loss type ('frobenius' or 'quadratic')")
	}
	else if (!(type %in% c("frobenius", "quadratic"))){
		stop("type should be one of {'frobenius', 'quadratic'}")
	}
	else {
		# Frobenius loss
		if (type == "frobenius"){
			if (precision)  {loss <- .FrobeniusLoss(E, T)}
			if (!precision) {loss <- .FrobeniusLoss(E, solve(T))}
		}

		# Quadratic loss
		if (type == "quadratic"){
			if (precision)  {loss <- .QuadraticLoss(E, solve(T))}
			if (!precision) {loss <- .QuadraticLoss(E, T)}
		}

		# Return
		return(loss)
	}
}



KLdiv <- function(Mtest, Mref, Stest, Sref, symmetric = FALSE){
	#####################################################################################################
	# - Function that calculates the Kullback-Leibler divergence between two normal distributions
	# - Mtest     > mean vector approximating m.v. normal distribution
	# - Mref      > mean vector 'true'/reference m.v. normal distribution
	# - Stest     > covariance matrix approximating m.v. normal distribution
	# - Sref      > covariance matrix 'true'/reference m.v. normal distribution
	# - symmetric > logical indicating if original symmetric version of KL div. should be calculated
	#####################################################################################################

	# Dependencies
	# require("base")

	if (class(Mtest) != "numeric"){
		stop("Input (Mtest) is of wrong class")
	} 
	else if (class(Mref) != "numeric"){
		stop("Input (Mref) is of wrong class")
	} 
	else if (length(Mtest) != length(Mref)){
		stop("Mtest and Mref should be of same length")
	} 
	else if (!is.matrix(Stest)){
		stop("Input (Stest) is of wrong class")
	} 
	else if (!is.matrix(Sref)){
		stop("Input (Sref) is of wrong class")
	} 
	else if (!isSymmetric(Stest)){
		stop("Stest should be symmetric")
	}
	else if (!isSymmetric(Sref)){
		stop("Sref should be symmetric")
	}
	else if (dim(Stest)[1] != length(Mtest)){
		stop("Column and row dimension of Stest should correspond to length Mtest")
	}
	else if (dim(Sref)[1] != length(Mref)){
		stop("Column and row dimension of Sref should correspond to length Mref")
	}
	else if (class(symmetric) != "logical"){
		stop("Input (symmetric) is of wrong class")
	} 
	else {
		# Evaluate KL divergence
		KLd <- (sum(diag(solve(Stest) %*% Sref)) + t(Mtest - Mref) %*% solve(Stest) %*% (Mtest - Mref) 
			  - nrow(Sref) - log(det(Sref)) + log(det(Stest)))/2

		# Evaluate (original) symmetric version KL divergence
		if (symmetric){
			KLd <- KLd + (sum(diag(solve(Sref) %*% Stest)) + t(Mref - Mtest) %*% solve(Sref) %*% (Mref - Mtest) 
				 	  - nrow(Sref) - log(det(Stest)) + log(det(Sref)))/2
		}

	# Return
	return(as.numeric(KLd))
	}
}



evaluateSfit <- function(Phat, S, diag = FALSE, fileType = "pdf", nameExt = "", dir = getwd()){
	############################################################################################
	# - Function aiding the visual inspection of the fit of the estimated (possibly regularized)
	#   precision matrix vis-a-vis the sample covariance matrix
	# - Phat     > (regularized) estimate of the precision matrix
	# - S        > sample covariance matrix
	# - diag     > logical determining treatment diagonal elements for plots
	# - fileType > signifies filetype of output
	# - nameExt  > character giving extension of default output names.
	#              Circumvents overwriting of output when working in single directory
	# - dir      > specifies the directory in which the visual output is stored
	#############################################################################################

	# Dependencies
	# require("base")
	# require("graphics")

	if (!is.matrix(Phat)){
		stop("Input (Phat) should be a matrix")
	}
	else if (!isSymmetric(Phat)){
		stop("Input (Phat) should be a symmetric matrix")
	}
	else if (all(diag(Phat) == 1)){
		stop("Input (Phat) should be a nonstandardized precision matrix")
	}
	else if (!is.matrix(S)){
		stop("Input (S) should be a matrix")
	}
	else if (!isSymmetric(S)){
		stop("Input (S) should be a symmetric matrix")
	}
	else if (all(diag(S) == 1)){
		stop("Input (S) should be a nonstandardized covariance matrix")
	}
	else if (class(diag) != "logical"){
		stop("Input (diag) is of wrong class")
	}
	else if (missing(fileType)){
		stop("Need to specify type of output file ('pdf' or 'eps')")
	}
	else if (!(fileType %in% c("pdf", "eps"))){
		stop("fileType should be one of {'pdf', 'eps'}")
	}
	else if (class(nameExt) != "character"){
		stop("Input (nameExt) is of wrong class")
	}
	else if (class(dir) != "character"){
		stop("Specify directory for output as 'character'")
	}
	else {
		# Obtain estimated covariance matrix
		Shat <- solve(Phat)


		print("Visualizing covariance fit")
		# plot 1: QQ-plot of covariances
		if (fileType == "pdf"){pdf(paste("QQplot_covariances_", nameExt, ".pdf"))}
		if (fileType == "eps"){setEPS(); postscript(paste("QQplot_covariances_", nameExt, ".eps"))}
		if (diag){cObs <- as.numeric(S[upper.tri(S, diag = TRUE)]); cFit <- as.numeric(Shat[upper.tri(Shat, diag = TRUE)])}
		if (!diag){cObs <- as.numeric(S[upper.tri(S)]); cFit <- as.numeric(Shat[upper.tri(Shat)])}      
		op <- par(pty = "s") 
		qqplot(x = cObs, y = cFit, pch = 20, xlab = "sample covariances", ylab = "fits", main = "QQ-plot, covariances")
		lines(seq(min(cFit, cObs), max(cFit, cObs), length.out = 100), seq(min(cFit, cObs), max(cFit, cObs), length.out = 100), col = "grey", lty = 2)
		par(op); dev.off()

		# plot 2: Comparison of covariances by heatmap
		if (fileType == "pdf"){pdf(paste("heatmap_covariances_", nameExt, ".pdf"))}
		if (fileType == "eps"){setEPS(); postscript(paste("heatmap_covariances_", nameExt, ".eps"))}
		op  <- par(pty = "s")  
		slh <- S
		slh[lower.tri(slh)] <- Shat[lower.tri(Shat)]
		gplot <- edgeHeat(slh, diag = diag, legend = FALSE, main = "Covariances")
		print(gplot); par(op); dev.off()   


		print("Visualizing correlation fit")
    		# plot 3: QQ-plot of correlations
		if (fileType == "pdf"){pdf(paste("QQplot_correlations_", nameExt, ".pdf"))}
		if (fileType == "eps"){setEPS(); postscript(paste("QQplot_correlations_", nameExt, ".eps"))}
		if (diag){cObs <- as.numeric(cov2cor(S)[upper.tri(S, diag = TRUE)]); cFit <- as.numeric(cov2cor(Shat)[upper.tri(Shat, diag = TRUE)])} 
		if (!diag){cObs <- as.numeric(cov2cor(S)[upper.tri(S)]); cFit <- as.numeric(cov2cor(Shat)[upper.tri(Shat)])}
		op <- par(pty = "s") 
		qqplot(x = cObs, y = cFit, pch = 20, xlab = "sample correlations", ylab = "fits", main = "QQ-plot, correlations")
		lines(seq(min(cFit, cObs), max(cFit, cObs), length.out = 100), seq(min(cFit, cObs), max(cFit, cObs), length.out = 100), col = "grey", lty = 2)
		par(op); dev.off()

		# plot 4: Comparison of correlations by heatmap
		if (fileType == "pdf"){pdf(paste("heatmap_correlations_", nameExt, ".pdf"))}
		if (fileType == "eps"){setEPS(); postscript(paste("heatmap_correlations_", nameExt, ".eps"))}
		op  <- par(pty = "s")  
		slh <- cov2cor(S)
		slh[lower.tri(slh)] <- cov2cor(Shat)[lower.tri(Shat)]
		gplot <- edgeHeat(slh, diag = diag, legend = FALSE, main = "Correlations")
		print(gplot); par(op); dev.off()   


		print("Visualizing partial correlation fit")
    		# If sample covariance matrix non-singular, also evaluate partial correlation fit
		if (evaluateS(S, verbose = FALSE)$posEigen){

			# plot 5: QQ-plot of partial correlations
			if (fileType == "pdf"){pdf(paste("QQplot_partCorrelations_", nameExt, ".pdf"))}
			if (fileType == "eps"){setEPS(); postscript(paste("QQplot_partCorrelations_", nameExt, ".eps"))}
			if (diag){cObs <- as.numeric(pcor(solve(S))[upper.tri(S)], diag = TRUE); cFit <- as.numeric(pcor(Phat)[upper.tri(Phat, diag = TRUE)])}
			if (!diag){cObs <- as.numeric(pcor(solve(S))[upper.tri(S)]); cFit <- as.numeric(pcor(Phat)[upper.tri(Phat)])}
			op <- par(pty = "s")  
			qqplot(x = cObs, y = cFit, pch = 20, xlab = "sample partial correlations", ylab = "fits", main = "QQ-plot, partial correlations")
			lines(seq(min(cFit, cObs), max(cFit, cObs), length.out = 100), seq(min(cFit, cObs), max(cFit, cObs), length.out = 100), col = "grey", lty = 2)
			par(op); dev.off()

			# plot 6: Comparison of partial correlations by heatmap
			if (fileType == "pdf"){pdf(paste("heatmap_partCorrelations_", nameExt, ".pdf"))}
			if (fileType == "eps"){setEPS(); postscript(paste("heatmap_partCorrelations_", nameExt, ".eps"))}
			op  <- par(pty = "s")  
			slh <- pcor(solve(S))
			slh[lower.tri(slh)] <- pcor(Phat)[lower.tri(Phat)]
			gplot <- edgeHeat(slh, diag = diag, legend = FALSE, main = "Partial correlations")
			print(gplot); par(op); dev.off()

		} else {
			print("sample covariance matrix is singular: partial correlation fit not visualized")
		}
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Functions for Visualization
##
##---------------------------------------------------------------------------------------------------------

ridgePathS <- function (S, lambdaMin, lambdaMax, step, type = "Alt", target = default.target(S), 
                        plotType = "pcor", diag = FALSE, verticle = FALSE, value, verbose = TRUE){
	#####################################################################################################
	# - Function that visualizes the regularization path
	# - Regularization path may be visualized for (partial) correlations, covariances and precision elements
	# - S         > sample covariance/correlation matrix
	# - lambdaMin > minimum value penalty parameter (dependent on 'type')
	# - lambdaMax > maximum value penalty parameter (dependent on 'type')
	# - step      > determines the coarseness in searching the grid [lambdaMin, lambdaMax]
	# - type      > must be one of {"Alt", "ArchI", "ArchII"}, default = "Alt"
	# - target    > target (precision terms) for Type I estimators, default = default.target(S)
	# - plotType  > specificies the elements for which the regularization path is to be visualized.
	#               Must be one of {"pcor", "cor", "cov", "prec"}, default = "pcor"
	# - diag      > logical indicating if the diagonal elements should be retained for plotting, 
	#               default = FALSE
	# - verticle  > optional argument for visualization verticle line in graph output, default = FALSE
	#               Can be used to indicate the value of, e.g., the optimal penalty as indicated by some
	#               routine. Can be used to assess the whereabouts of this optimal penalty along the 
	#               regularization path
	# - value     > indicates constant on which to base verticle line when verticle = TRUE
	# - verbose   > logical indicating if intermediate output should be printed on screen
	#####################################################################################################

	# Dependencies
	# require("base")

	if (class(verbose) != "logical"){
		stop("Input (verbose) is of wrong class")
	}
	if (verbose){
		cat("Perform input checks...", "\n")
	}
	if (!is.matrix(S)){
		stop("input (S) should be a matrix")
	}
	if (!isSymmetric(S)){
		stop("Input (S) should be a covariance matrix")
	}
	else if (class(lambdaMin) != "numeric"){
		stop("Input (lambdaMin) is of wrong class")
	}
	else if (length(lambdaMin) != 1){
		stop("Input (lambdaMin) must be a scalar")
	}
	else if (lambdaMin <= 0){
		stop("Input (lambdaMin) must be positive")
	}
	else if (class(lambdaMax) != "numeric"){
		stop("Input (lambdaMax) is of wrong class")
	}
	else if (length(lambdaMax) != 1){
		stop("Input (lambdaMax) must be a scalar")
	}
	else if (lambdaMax <= lambdaMin){
		stop("lambdaMax must be larger than lambdaMin")
	}
	else if (class(step) != "numeric") {
		stop("Input (step) is of wrong class")
	}
	else if (!.is.int(step)){
		stop("Input (step) should be integer")
	}
	else if (step <= 0){
		stop("Input (step) should be a positive integer")
	}
	else if (class(plotType) != "character") {
		stop("Input (plotType) is of wrong class")
	}
	else if (!(plotType %in% c("pcor", "cor", "cov", "prec"))){
		stop("Input (plotType) should be one of {'pcor', 'cor', 'cov', 'prec'}")
	}
	else if (length(nchar(plotType)) != 1){
		stop("Input (plotType) should be exactly one of {'pcor', 'cor', 'cov', 'prec'}")
	}
	if (class(diag) != "logical") {
		stop("Input (diag) is of wrong class")
	}
	else if (class(verticle) != "logical"){
		stop("Input (verticle) is of wrong class")
	}
	else {
		# Set preliminaries
		lambdas  <- seq(lambdaMin, lambdaMax, len = step)
		YforPlot <- numeric()

		# Calculate paths
		if (verbose){cat("Calculating...", "\n")}
        	for (k in 1:length(lambdas)){
            	P <- ridgeS(S, lambdas[k], type = type, target = target)
	    		if (plotType=="pcor"){YforPlot <- cbind(YforPlot, pcor(P)[upper.tri(P)])}
	    		if (plotType=="prec"){YforPlot <- cbind(YforPlot, P[upper.tri(P, diag = diag)])}
	    		if (plotType=="cov"){YforPlot <- cbind(YforPlot, solve(P)[upper.tri(P, diag = diag)])}
	    		if (plotType=="cor"){YforPlot <- cbind(YforPlot, cov2cor(solve(P))[upper.tri(P)])}
            	if (verbose){cat(paste("lambda = ", lambdas[k], " done", sep = ""), "\n")}
        	}

		# Visualize
        	if (plotType=="cor"){ylabel <- "penalized correlation"}
        	if (plotType=="cov"){ylabel <- "penalized covariances"}
        	if (plotType=="pcor"){ylabel <- "penalized partial correlation"}
        	if (plotType=="prec"){ylabel <- "penalized precision elements"}
		if (type == "Alt"){Main = "Alternative ridge estimator"}
		if (type == "ArchI"){Main = "Archetypal I ridge estimator"}
		if (type == "ArchII"){Main = "Archetypal II ridge estimator"}

        	plot(YforPlot[1,] ~ log(lambdas), axes = FALSE, xlab = "ln(penalty value)", ylab = ylabel, main = Main, col = "white", ylim = c(min(YforPlot), max(YforPlot)))
        	for (k in 1:nrow(YforPlot)){lines(YforPlot[k,] ~ log(lambdas), col = k, lty = k)}
		axis(2, col = "black", lwd = 1)
		axis(1, col = "black", lwd = 1)
		if (verticle){
			if (missing(value)){
				stop("Need to specify input (value)")
			} else if (class(value) != "numeric"){
				stop("Input (value) is of wrong class")
			} else if (length(value) != 1){
				stop("Input (value) must be a scalar")
			} else if (value <= 0){
				stop("Input (value) must be positive")
			} else {
				abline(v = log(value), col = "red")
			}
		}
	}
}



if(getRversion() >= "2.15.1") utils::globalVariables(c("X1", "X2", "value"))

edgeHeat <- function(M, lowColor = "blue", highColor = "red", textsize = 10, diag = TRUE, legend = TRUE, main = ""){
	#####################################################################################################
	# - function that visualizes precision matrix as a heatmap
	# - can be used to assess (visually) the performance of set of graphical modeling techniques
	# - M         > Precision matrix
	# - lowColor  > determines color scale in the negative range, default = "blue"
	# - highColor > determines color scale in the positive range, default = "red"
	# - textsize  > set textsize row and column labels, default = 10
	# - diag      > logical determining treatment diagonal elements M. If FALSE, then the diagonal
	#               elements are given the midscale color of white; only when M is a square matrix
	# - legend    > optional inclusion of color legend, default = TRUE
	# - main      > character specifying the main title, default = ""
	#####################################################################################################

	# Dependencies
	#require("ggplot2")
	#require("reshape")

	if (!is.matrix(M)){
		stop("Supply 'M' as matrix")
	}
	else if (class(lowColor) != "character"){
		stop("Input (lowColor) is of wrong class")
	}
	else if (length(lowColor) != 1){
		stop("Length lowColor must be one")
	}
	else if (class(highColor) != "character"){
		stop("Input (highColor) is of wrong class")
	}
	else if (length(highColor) != 1){
		stop("Length highColor must be one")
	}
	else if (class(textsize) != "numeric"){
			stop("Input (textsize) is of wrong class")
	}
	else if (length(textsize) != 1){
			stop("Length textsize must be one")
	}
	else if (textsize <= 0){
			stop("textsize must be positive")
	} 
	else if (class(diag) != "logical"){
		stop("Input (diag) is of wrong class")
	}
	else if (class(legend) != "logical"){
		stop("Input (legend) is of wrong class")
	}
	else if (class(main) != "character"){
		stop("Input (main) is of wrong class")
	}
	else {
		# Put matrix in data format
		if (nrow(M) == ncol(M) & !diag) {diag(M) <- 0}
		Mmelt    <- melt(M)
		Mmelt$X1 <- factor(as.character(Mmelt$X1), levels = unique(Mmelt$X1), ordered = TRUE)
		Mmelt$X2 <- factor(as.character(Mmelt$X2), levels = unique(Mmelt$X2), ordered = TRUE)
	
		# Visualize
		if (legend){
			ggplot(Mmelt, aes(X2, X1, fill = value)) + geom_tile() + 
         	 	 	 scale_fill_gradient2("", low = lowColor,  mid = "white", high = highColor, midpoint = 0) +
		 	 	 theme(axis.ticks = element_blank()) +
				 theme(axis.text.y = element_text(size = textsize)) +
				 theme(axis.text.x = element_text(angle = -90, vjust = .5, size = textsize)) +
	   	 	 	 xlab(" ") + ylab(" ") +
         		 	 ylim(rev(levels(Mmelt$X1))) +
				 ggtitle(main)
		} else {
			ggplot(Mmelt, aes(X2, X1, fill = value)) + geom_tile() + 
         	 	 	 scale_fill_gradient2("", low = lowColor,  mid = "white", high = highColor, midpoint = 0) +
		 	 	 theme(axis.ticks = element_blank()) +
				 theme(axis.text.y = element_text(size = textsize)) +
				 theme(axis.text.x = element_text(angle = -90, vjust = .5, size = textsize)) +
	   	 	 	 xlab(" ") + ylab(" ") +
         		 	 ylim(rev(levels(Mmelt$X1))) +
				 ggtitle(main) +
			 	 theme(legend.position = "none")
		}
	}
}



Ugraph <- function(M, type = c("plain", "fancy", "weighted"), lay = layout.circle, Vsize = 15, Vcex = 1, 
			 Vcolor = "orangered", VBcolor = "darkred", VLcolor = "black", prune = FALSE, legend = FALSE, 
			 label = "", Lcex = 1.3, PTcex = 4, cut = .5, scale = 10, pEcolor = "black", nEcolor = "grey", 
			 main = ""){
	#####################################################################################################
	# - Function that visualizes the sparsified precision matrix as an undirected graph
	# - Function is partly a wrapper around certain 'igraph' functions
	# - M       > (Possibly sparsified) precision matrix
	# - type    > graph type: 'plain' gives plain undirected graph. 'fancy' gives undirected graph in
	#		  which dashed lines indicate negative partial correlations while solid lines indicate
	#		  positive partial correlations, and in which black lines indicate strong edges. 'weighted'
	#             gives an undirected graph in which edge thickness indicates the strenght of the partial
	#             correlations. Grey lines then indicate negative partial correlations while black lines
	#             represent positive partial correlations.
	# - lay     > determines layout of the graph. All layouts in 'layout{igraph}' are accepted.
	#		  Default = layout.circle, giving circular layout
	# - Vsize   > gives vertex size, default = 15
	# - Vcex    > gives size vertex labels, default = 1
	# - Vcolor  > gives vertex color, default = "orangered"
	# - VBcolor > gives color of the vertex border, default = "darkred"
	# - VLcolor > gives color of the vertex labels, default = "black"
	# - prune   > logical indicating if vertices of degree 0 should be removed
	# - legend  > optional inclusion of color legend, default = FALSE
	# - label   > character label for the endogenous variables, default = ""; only when legend = TRUE
	# - Lcex    > scaling legend box, default = 1.3; only when legend = TRUE
	# - PTcex   > scaling node in legend box, default = 4; only when legend = TRUE
	# - cut     > cut-off for indication of strong edge, default = .5; only when type = "fancy"
	# - scale   > scale factor for visualizing strenght of edges, default = 10; only when type = "weighted"
	# - pEcolor > gives edge color for edges tied to positive precision elements, default = "black"; only
	#             when type = "weighted"
	# - nEcolor > gives edge color for edges tied to negative precision elements, default = "grey"; only
	#             when type = "weighted"
	# - main    > character specifying heading figure, default = "" 
	#####################################################################################################

	# Dependencies
	# require("igraph")
	# require("reshape")

	if (!is.matrix(M)){
		stop("M should be a matrix")
	}
	else if (nrow(M) != ncol(M)){
		stop("M should be square matrix")
	}
	else if (missing(type)){
		stop("Need to specify graph type ('plain' or 'fancy' or 'weighted')")
	}
	else if (!(type %in% c("plain", "fancy", "weighted"))){
		stop("type should be one of {'plain', 'fancy', 'weighted'}")
	}
	else if (class(Vsize) != "numeric"){
		stop("Input (Vsize) is of wrong class")
	}
	else if (length(Vsize) != 1){
		stop("Length Vsize must be one")
	}
	else if (Vsize <= 0){
		stop("Vsize must be positive")
	}
	else if (class(Vcex) != "numeric"){
		stop("Input (Vcex) is of wrong class")
	}
	else if (length(Vcex) != 1){
		stop("Length Vcex must be one")
	}
	else if (Vcex <= 0){
		stop("Vcex must be positive")
	}
	else if (class(Vcolor) != "character"){
		stop("Input (Vcolor) is of wrong class")
	}
	else if (length(Vcolor) != 1){
		stop("Length Vcolor must be one")
	}
	else if (class(VBcolor) != "character"){
		stop("Input (VBcolor) is of wrong class")
	}
	else if (length(VBcolor) != 1){
		stop("Length VBcolor must be one")
	}
	else if (class(VLcolor) != "character"){
		stop("Input (VLcolor) is of wrong class")
	}
	else if (length(VLcolor) != 1){
		stop("Length VLcolor must be one")
	}
	else if (class(prune) != "logical"){
		stop("Input (prune) is of wrong class")
	}
	else if (class(legend) != "logical"){
		stop("Input (legend) is of wrong class")
	}
	else if (class(main) != "character"){
		stop("Input (main) is of wrong class")
	}
	else {
		# Preliminaries
		AM <- adjacentMat(M)
		GA <- graph.adjacency(AM, mode = "undirected")
		if (prune){GA <- delete.vertices(GA, which(degree(GA) < 1))}

		# Plain graph
		if (type == "plain"){
			plot(GA, layout = lay, vertex.size = Vsize, vertex.label.family = "sans", vertex.label.cex = Vcex, 
			     vertex.color = Vcolor, vertex.frame.color = VBcolor, vertex.label.color = VLcolor, main = main)
		}

		# Fancy graph
		if (type == "fancy"){
			if (class(cut) != "numeric"){
				stop("Input (cut) is of wrong class")
			} else if (length(cut) != 1){
				stop("Length cut must be one")
			} else if (cut <= 0){
				stop("cut must be positive")
			} else {
				Names <- colnames(M)
				colnames(M) = rownames(M) <- seq(1, ncol(M), by = 1)
				Mmelt <- melt(M)
				Mmelt <- Mmelt[Mmelt$X1 > Mmelt$X2,]
				Mmelt <- Mmelt[Mmelt$value != 0,]
				E(GA)$weight <- Mmelt$value
				E(GA)$color  <- "grey"
				E(GA)[E(GA)$weight < 0]$style <- "dashed"
				E(GA)[E(GA)$weight > 0]$style <- "solid"		
				E(GA)[abs(E(GA)$weight) > cut]$color <- "black"
				plot(GA, layout = lay, vertex.size = Vsize, vertex.label.family = "sans", vertex.label.cex = Vcex, 
				     vertex.color = Vcolor, vertex.frame.color = VBcolor, vertex.label.color = VLcolor, 
				     edge.color = E(GA)$color, edge.lty = E(GA)$style, main = main)
			}
		}

		# Weighted graph
		if (type == "weighted"){
			if (class(scale) != "numeric"){
				stop("Input (scale) is of wrong class")
			} else if (length(scale) != 1){
				stop("Length scale must be one")
			} else if (scale <= 0){
				stop("scale must be positive")
			} else if (class(pEcolor) != "character"){
				stop("Input (pEcolor) is of wrong class")
			} else if (length(pEcolor) != 1){
				stop("Length pEcolor must be one")
			} else if (class(nEcolor) != "character"){
				stop("Input (nEcolor) is of wrong class")
			} else if (length(nEcolor) != 1){
				stop("Length nEcolor must be one")
			} else {
				Names <- colnames(M)
				colnames(M) = rownames(M) <- seq(1, ncol(M), by = 1)
				Mmelt <- melt(M)
				Mmelt <- Mmelt[Mmelt$X1 > Mmelt$X2,]
				Mmelt <- Mmelt[Mmelt$value != 0,]
				E(GA)$weight <- Mmelt$value
				E(GA)[E(GA)$weight < 0]$color <- nEcolor
				E(GA)[E(GA)$weight > 0]$color <- pEcolor
				plot(GA, layout = lay, vertex.size = Vsize, vertex.label.family = "sans", vertex.label.cex = Vcex, 
				     vertex.color = Vcolor, vertex.frame.color = VBcolor, vertex.label.color = VLcolor, 
				     edge.color = E(GA)$color, edge.width = scale*abs(E(GA)$weight), main = main)
			}
		}
				
		# Legend
		if (legend){
			if (class(label) != "character"){
				stop("Input (label) is of wrong class")
			} else if (length(label) != 1){
				stop("Length label must be one")
			} else if (class(Lcex) != "numeric"){
				stop("Input (Lcex) is of wrong class")
			} else if (length(Lcex) != 1){
				stop("Length Lcex must be one")
			} else if (Lcex <= 0){
				stop("Lcex must be positive")
			} else if (class(PTcex) != "numeric"){
				stop("Input (PTcex) is of wrong class")
			} else if (length(PTcex) != 1){
				stop("Length PTcex must be one")
			} else if (PTcex <= 0){
				stop("PTcex must be positive")
			} else{
				legend("bottomright", label, pch = 20, col = Vcolor, cex = Lcex, pt.cex = PTcex)
			}
		}
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Function for Network Statistics
##
##---------------------------------------------------------------------------------------------------------

GGMnetworkStats <- function(sparseP){
	############################################################################################
	# - Function that calculates various network statistics from a sparse matrix
	# - Input matrix is assumed to be a sparse precision of partial correlation matrix
	# - The sparse precision matrix is taken to represent a conditional independence graph
	# - sparseP > sparse precision/partial correlation matrix
	# 
	# - NOTES (network statistics produced):
	# - Node degree
	# - Betweenness centrality
	# - Closeness centrality
	# - Number of negative edges for each node
	# - Number of positive edges for each node
	# - Mutual information of each variate with all other variates
	# - Variance of each variate (based on inverse sparsified precision matrix)
	# - Partial variance of each variate (= 1 when input matrix is partial correlation matrix)
	# - Future versions of this function may include additional statistics
	#
	# - REFERENCE:
	# - Newman, M.E.J. (2010), "Networks: an introduction", Oxford University Press
	#############################################################################################

	# Dependencies
	# require("base")
	# require("igraph")

	if (!is.matrix(sparseP)){
		stop("Input (sparseP) should be a matrix")
	}
	else if (!isSymmetric(sparseP)){
		stop("Input (sparseP) should be a symmetric matrix")
	}
	else if (!evaluateS(sparseP, verbose = FALSE)$posEigen){
		stop("Input (sparseP) is expected to be positive definite")
	}
	else{
		# Some warnings
		if (all(sparseP != 0)){warning("Given input (sparseP) implies a saturated conditional independence graph")}
		if (all(sparseP[!diag(nrow(sparseP))] == 0)){warning("Given input (sparseP) implies an empty conditional independence graph")}

      	# Obtain corresponding sample covariance matrix
    		pvars <- 1/diag(sparseP)
    		S     <- solve(sparseP)

    		# Calculate nodes' mutual information
    		MI    <- unlist(lapply(1:nrow(S), function(j, S){ log(det(S[-j,-j])) - log(det(S[-j,-j] - S[-j,j,drop=FALSE] %*% S[j,-j,drop=FALSE] / S[j,j])) }, S = S))
		Nodes <- colnames(sparseP)
		MI    <- data.frame(Nodes, MI)
    	
		# Signs of edges
		diag(sparseP) <- 0 
		nPos <- apply(sign(sparseP), 2, function(Z){ sum(Z == 1) }) 
		nNeg <- apply(sign(sparseP), 2, function(Z){ sum(Z == -1) })

    		# Adjacency to graphical object
    		AM  <- adjacentMat(sparseP)
    		CIG <- graph.adjacency(AM, mode = "undirected")

		# Return
    		return(list(degree = degree(CIG), betweenness = betweenness(CIG), closeness = closeness(CIG),  
		 	 nNeg = nNeg, nPos = nPos, mutualInfo = MI, variance = diag(S), partialVariance = pvars))
	}
}




##---------------------------------------------------------------------------------------------------------
## 
## Miscellaneous
##
##---------------------------------------------------------------------------------------------------------

.TwoCents <- function(){
	#####################################################################################################
	# - Unsolicited Advice
	#####################################################################################################

	cat("
            ##########################
            ##########################
            Get ridge or die trying. 
                              - 2Cents 
            ##########################
            ########################## \n")
}




##---------------------------------------------------------------------------------------------------------
## 
## NOTES
##
##---------------------------------------------------------------------------------------------------------

## Updates from version 1.1 to 1.2
#- Inclusion function for ML estimation of the sample covariance matrix: 'covML'
#- Inclusion function for approximate leave-one-out cross-validation: 'optPenalty.aLOOCV'
#- Inclusion function 'conditionNumber' to visualize the spectral condition number over the regularization path
#- Inclusion function 'evaluateS' to evaluate basic properties of a covariance matrix
#- Inclusion function 'KLdiv' that calculates the Kullback-Leibler divergence between two normal distributions
#- Inclusion option to suppress on-screen output in 'sparsify' function
#- Corrected small error in 'optPenaltyCV' function
#- Both 'optPenaltyCV' and 'optPenalty.aLOOCV' now utilize 'covML' instead of 'cov'
#- Default output option in 'optPenaltyCV' (as in 'optPenalty.aLOOCV') is now "light"


## Updates from version 1.2 to 1.3
#- Inclusion hidden function '.ridgeSi' for usage in 'conditionNumberPlot' function
#- Inclusion hidden function '.eigShrink' for usage in (a.o.) 'ridgeS' function
#- Inclusion function calculating various network statistics from a sparse matrix: 'GGMnetworkStats'
#- Inclusion function for visual inspection fit of regularized precision matrix to sample covariance matrix: 'evaluateSfit'
#- Inclusion function for visualization of regularization paths: 'ridgePathS'
#- Inclusion function for default target matrix generation: 'default.target'
#- New features updated 'evaluateS' function:
	# The printed output of the 'evaluateS' function is now aligned 
	# Calculation spectral condition number has been improved
#- 'conditionNumber' function now called 'conditionNumberPlot'. Updated function has new features:
	# Main plot can now be obtained with either the spectral (l2) or the (approximation to) l1 condition number
	# Main plot can now be amended with plot of the relative distance to the set of singular matrices
	# The title of the main plot can now be suppressed
	# One can now obtain numeric output from the function: lambdas and condition numbers
#- New features updated 'sparsify' function:
	# Changed 'type = c("threshold", "localFDR")' to 'threshold = c("absValue", "localFDR")' (clarifying nomenclature)
	# Changed 'threshold' to 'absValueCut' (clarifying nomenclature)
	# Will now output both sparsified partial correlation/standardized precision matrix and the sparsified precison matrix >>
	# when input consists of the unstandardized precision matrix
#- New features updated 'ridgeS' function:
	# Contains an improved evaluation of the target matrix possibly being a null matrix
	# Now evaluates if a rotation equivariant alternative estimator ensues for a given target matrix
	# When rotation equivariant alternative estimator ensues, computation is sped up considerably by circumventing the matrix square root
#- 'optPenaltyCV' function now called 'optPenalty.LOOCV', for sake of (naming) consistency. Updated function has new features:
	# 'targetScale' option has been removed
	# Replaced 'log' in optional graph by 'ln'
	# Visual layout of optional graph now more in line with recommendations by Tufte
#- New features updated 'optPenalty.aLOOCV' function:
	# Replaced 'log' in optional graph by 'ln'
	# Visual layout of optional graph now more in line with recommendations by Tufte
#- Computation optimal penalty in 'conditionNumberPlot', 'optPenalty.aLOOCV' and 'optPenalty.LOOCV' functions sped up considerably:
	# For rotation equivariant alternative estimator
	# By usage new ridgeS and avoidance of redundant eigendecompositions
#- Default target in 'ridgeS', 'conditionNumberPlot', 'optPenalty.aLOOCV' and 'optPenalty.LOOCV' now "DAIE" option in 'default.target'



