% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_power.R
\name{get_power}
\alias{get_power}
\title{Get NASA POWER Data From the POWER API}
\usage{
get_power(
  community,
  pars,
  temporal_api = NULL,
  lonlat,
  dates = NULL,
  site_elevation = NULL,
  wind_elevation = NULL,
  wind_surface = NULL,
  temporal_average = NULL,
  time_standard = "LST"
)
}
\arguments{
\item{community}{A character vector providing community name: \dQuote{ag},
\dQuote{re} or \dQuote{sb}.  See argument details for more.}

\item{pars}{A character vector of solar, meteorological or climatology
parameters to download.  When requesting a single point of x, y
coordinates, a maximum of twenty (20) \code{pars} can be specified at one time,
for \dQuote{daily}, \dQuote{monthly} and \dQuote{climatology}
\code{temporal_api}s.  If the \code{temporal_api} is specified as \dQuote{hourly}
only 15 \code{pars} can be specified in a single query.  See \code{temporal_api} for
more.}

\item{temporal_api}{Temporal \acronym{API} end-point for data being queried,
supported values are \dQuote{hourly}, \dQuote{daily}, \dQuote{monthly} or
\dQuote{climatology}.  See argument details for more.}

\item{lonlat}{A numeric vector of geographic coordinates for a cell or region
entered as x, y coordinates.  See argument details for more.}

\item{dates}{A character vector of start and end dates in that order,\cr
\emph{e.g.}, \code{dates = c("1983-01-01", "2017-12-31")}.
Not used when\cr \code{temporal_api} is set to \dQuote{climatology}.
See argument details for more.}

\item{site_elevation}{A user-supplied value for elevation at a single point
in metres.  If provided this will return a corrected atmospheric pressure
value adjusted to the elevation provided.  Only used with \code{lonlat} as a
single point of x, y coordinates, not for use with \dQuote{global} or with
a regional request.}

\item{wind_elevation}{A user-supplied value for elevation at a single point
in metres.  Wind Elevation values in Meters are required to be between 10m
and 300m.  Only used with \code{lonlat} as a single point of x, y coordinates,
not for use with \dQuote{global} or with a regional request.  If this
parameter is provided, the \code{wind-surface} parameter is required with the
request, see
\url{https://power.larc.nasa.gov/docs/methodology/meteorology/wind/}.}

\item{wind_surface}{A user-supplied wind surface for which the corrected
wind-speed is to be supplied.  See \code{wind-surface} section for more detail.}

\item{temporal_average}{Deprecated. This argument has been superseded by
\code{temporal_api} to align with the new \acronym{POWER} \acronym{API}
terminology.}

\item{time_standard}{\acronym{POWER} provides two different time standards:
\itemize{
\item Universal Time Coordinated (\acronym{UTC}): is the standard time measure
that used by the world.
\item Local Solar Time (\acronym{LST}): A 15 degree swath that represents
solar noon at the middle longitude of the swath.
Defaults to \code{LST}.
}}
}
\value{
A data frame as a \code{POWER.Info} class, an extension of the
\link[tibble:tibble]{tibble::tibble}, object of \acronym{POWER} data including location, dates
(not including \dQuote{climatology}) and requested parameters.  A decorative
header of metadata is included in this object.
}
\description{
Get \acronym{POWER} global meteorology and surface solar energy
climatology data and return a tidy data frame \code{\link[tibble:tibble]{tibble::tibble()}}
object.  All options offered by the official \acronym{POWER} \acronym{API}
are supported.  Requests are formed to submit one request per point.
There is no need to make synchronous requests for multiple parameters for
a single point or regional request.  See section on \dQuote{Rate Limiting}
for more.
}
\note{
The associated metadata shown in the decorative header are not saved
if the data are exported to a file format other than a native \R data
format, \emph{e.g.}, .Rdata, .rda or .rds.
}
\section{Argument details for \dQuote{community}}{
 There are three valid
values, one must be supplied. This  will affect the units of the parameter
and the temporal display of time series data.

\describe{
\item{ag}{Provides access to the Agroclimatology Archive, which
contains industry-friendly parameters formatted for input to crop models.}

\item{sb}{Provides access to the Sustainable Buildings Archive, which
contains industry-friendly parameters for the buildings community to include
parameters in multi-year monthly averages.}

\item{re}{Provides access to the Renewable Energy Archive, which contains
parameters specifically tailored to assist in the design of solar and wind
powered renewable energy systems.}
}
}

\section{Argument details for \code{temporal_api}}{
 There are four valid values.
\describe{
\item{hourly}{The hourly average of \code{pars} by hour, day, month and year,
the time zone is LST by default.}
\item{daily}{The daily average of \code{pars} by day, month and year.}
\item{monthly}{The monthly average of \code{pars} by month and year.}
\item{climatology}{Provide parameters as 22-year climatologies (solar)
and 30-year climatologies (meteorology); the period climatology and
monthly average, maximum, and/or minimum values.}
}
}

\section{Argument details for \code{lonlat}}{

\describe{
\item{For a single point}{To get a specific cell, 1/2 x 1/2 degree, supply a
length-two numeric vector giving the decimal degree longitude and latitude
in that order for data to download,\cr
\emph{e.g.}, \code{lonlat = c(-179.5, -89.5)}.}

\item{For regional coverage}{To get a region, supply a length-four numeric
vector as lower left (lon, lat) and upper right (lon, lat) coordinates,
\emph{e.g.}, \code{lonlat = c(xmin, ymin, xmax, ymax)} in that order for a
given region, \emph{e.g.}, a bounding box for the south western corner of
Australia: \code{lonlat = c(112.5, -55.5, 115.5, -50.5)}.  *Maximum area
processed is 4.5 x 4.5 degrees (100 points).}

\item{For global coverage}{To get global coverage for \dQuote{climatology},
supply \dQuote{global} while also specifying \dQuote{climatology} for the
\code{temporal_api}.}
}
}

\section{Argument details for \code{dates}}{
 if one date only is provided, it
will be treated as both the start date and the end date and only a single
day's values will be returned, \emph{e.g.}, \code{dates = "1983-01-01"}.  When
\code{temporal_api} is set to \dQuote{monthly}, use only two year values (YYYY),
\emph{e.g.} \code{dates = c(1983, 2010)}.  This argument should not be used when
\code{temporal_api} is set to \dQuote{climatology} and will be ignored if set.
}

\section{\code{wind_surface}}{
 There are 17 surfaces that may be used for corrected
wind-speed values using the following equation:
\deqn{WSC_hgt = WS_10m\times(\frac{hgt}{WS_50m})^\alpha}{WSC_hgt = WS_10m*(hgt/WS_50m)^\alpha}
Valid surface types are described here.

\describe{
\item{vegtype_1}{35-m broadleaf-evergreen trees (70\% coverage)}
\item{vegtype_2}{20-m broadleaf-deciduous trees (75\% coverage)}
\item{vegtype_3}{20-m broadleaf and needleleaf trees (75\% coverage)}
\item{vegtype_4}{17-m needleleaf-evergreen trees (75\% coverage)}
\item{vegtype_5}{14-m needleleaf-deciduous trees (50\% coverage)}
\item{vegtype_6}{Savanna:18-m broadleaf trees (30\%) & groundcover}
\item{vegtype_7}{0.6-m perennial groundcover (100\%)}
\item{vegtype_8}{0.5-m broadleaf shrubs (variable \%) & groundcover}
\item{vegtype_9}{0.5-m broadleaf shrubs (10\%) with bare soil}
\item{vegtype_10}{Tundra: 0.6-m trees/shrubs (variable \%) & groundcover}
\item{vegtype_11}{Rough bare soil}
\item{vegtype_12}{Crop: 20-m broadleaf-deciduous trees (10\%) & wheat}
\item{vegtype_20}{Rough glacial snow/ice}
\item{seaice}{Smooth sea ice}
\item{openwater}{Open water}
\item{airportice}{Airport: flat ice/snow}
\item{airportgrass}{Airport: flat rough grass}
}
}

\section{Rate limiting}{
 The POWER API endpoints limit queries to prevent
server overloads due to repetitive and rapid requests.  If you find that
the \acronym{API} is throttling your queries, I suggest that you
investigate the use of \code{limit_rate()} from \CRANpkg{ratelimitr} to create
self-limiting functions that will respect the rate limits that the
\acronym{API} has in place.  It is considered best practice to check the
POWER website](https://power.larc.nasa.gov/docs/services/api/#rate-limiting)
for the latest rate limits as they differ between temporal \acronym{API}s
and may change over time as the project matures.
}

\examples{
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

# Fetch daily "ag" community temperature, relative humidity and
# precipitation for January 1 1985 at Kingsthorpe, Queensland, Australia
ag_d <- get_power(
  community = "ag",
  lonlat = c(151.81, -27.48),
  pars = c("RH2M", "T2M", "PRECTOTCORR"),
  dates = "1985-01-01",
  temporal_api = "daily"
)

ag_d

# Fetch single point climatology for air temperature
ag_c_point <- get_power(
  community = "ag",
  pars = "T2M",
  c(151.81, -27.48),
  temporal_api = "climatology"
)

ag_c_point

# Fetch interannual solar cooking parameters for a given region
sse_i <- get_power(
  community = "re",
  lonlat = c(112.5, -55.5, 115.5, -50.5),
  dates = c("1984", "1985"),
  temporal_api = "monthly",
  pars = c("CLRSKY_SFC_SW_DWN", "ALLSKY_SFC_SW_DWN")
)

sse_i
\dontshow{\}) # examplesIf}
}
\references{
\url{https://power.larc.nasa.gov/docs/methodology/}
\url{https://power.larc.nasa.gov}
}
\author{
Adam H. Sparks \email{adamhsparks@gmail.com}
}
