% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elliptic.r
\name{elliptic}
\alias{deviance.elliptic}
\alias{elliptic}
\alias{fitted.elliptic}
\alias{print.elliptic}
\alias{residuals.elliptic}
\title{Nonlinear Multivariate Elliptically-contoured Repeated Measurements Models
with AR(1) and Two Levels of Variance Components}
\usage{
elliptic(response = NULL, model = "linear", distribution = "normal",
  times = NULL, dose = NULL, ccov = NULL, tvcov = NULL, nest = NULL,
  torder = 0, interaction = NULL, transform = "identity",
  link = "identity", autocorr = "exponential", pell = NULL, preg = NULL,
  covfn = NULL, pvar = var(y), varfn = NULL, par = NULL, pre = NULL,
  delta = NULL, shfn = FALSE, common = FALSE, twins = FALSE,
  envir = parent.frame(), print.level = 0, ndigit = 10, gradtol = 1e-05,
  steptol = 1e-05, iterlim = 100, fscale = 1, stepmax = 10 * sqrt(theta
  \%*\% theta), typsize = abs(c(theta)))

\method{deviance}{elliptic}(object, ...)

\method{fitted}{elliptic}(object, recursive = FALSE, ...)

\method{residuals}{elliptic}(object, recursive = FALSE, ...)

\method{print}{elliptic}(x, digits = max(3, .Options$digits - 3),
  correlation = TRUE, ...)
}
\arguments{
\item{response}{A list of two or three column matrices with response values,
times, and possibly nesting categories, for each individual, one matrix or
dataframe of response values, or an object of class, \code{response}
(created by \code{\link[rmutil]{restovec}}) or \code{repeated} (created by
\code{\link[rmutil]{rmna}} or \code{\link[rmutil]{lvna}}). If the
\code{repeated} data object contains more than one response variable, give
that object in \code{envir} and give the name of the response variable to be
used here.}

\item{model}{The model to be fitted for the location. Builtin choices are
(1) \code{linear} for linear models with time-varying covariate; if
\code{torder > 0}, a polynomial in time is automatically fitted; (2)
\code{logistic} for a four-parameter logistic growth curve; (3) \code{pkpd}
for a first-order one-compartment pharmacokinetic model. Otherwise, set this
to a function of the parameters or a formula beginning with ~, specifying
either a linear regression function for the location parameter in the
Wilkinson and Rogers notation or a general function with named unknown
parameters that describes the location, returning a vector the same length
as the number of observations, in which case \code{ccov} and \code{tvcov}
cannot be used.}

\item{distribution}{Multivariate \code{normal}, \code{power exponential},
\code{Student t}, or skew \code{Laplace} distribution. The latter is not an
elliptical distribution. Note that the latter has a different
parametrization of the skew (family) parameter than the univariate skew
Laplace distribution in \code{\link[rmutil]{dskewlaplace}}:
\eqn{skew=\frac{\sigma(1-\nu^2)}{\sqrt{2}\nu}}{skew = (s * (1 - f^2)) /
(sqrt(2) * f)}. Here, zero skew yields a symmetric distribution.}

\item{times}{When \code{response} is a matrix, a vector of possibly
unequally spaced times when they are the same for all individuals or a
matrix of times. Not necessary if equally spaced. Ignored if response has
class, \code{response} or \code{repeated}.}

\item{dose}{A vector of dose levels for the \code{pkpd model}, one per
individual.}

\item{ccov}{A vector or matrix containing time-constant baseline covariates
with one line per individual, a model formula using vectors of the same
size, or an object of class, \code{tccov} (created by
\code{\link[rmutil]{tcctomat}}). If response has class, \code{repeated},
with a \code{linear}, \code{logistic}, or \code{pkpd} model, the covariates
must be specified as a Wilkinson and Rogers formula unless none are to be
used. For the \code{pkpd} and \code{logistic} models, all variables must be
binary (or factor variables) as different values of all parameters are
calculated for all combinations of these variables (except for the logistic
model when a time-varying covariate is present). It cannot be used when
model is a function.}

\item{tvcov}{A list of vectors or matrices with time-varying covariates for
each individual (one column per variable), a matrix or dataframe of such
covariate values (if only one covariate), or an object of class,
\code{tvcov} (created by \code{\link[rmutil]{tvctomat}}). If times are not
the same as for responses, the list can be created with
\code{\link[rmutil]{gettvc}}. If response has class, \code{repeated}, with a
\code{linear}, \code{logistic}, or \code{pkpd} model, the covariates must be
specified as a Wilkinson and Rogers formula unless none are to be used. Only
one time-varying covariate is allowed except for the \code{linear model}; if
more are required, set \code{model} equal to the appropriate mean function.
This argument cannot be used when model is a function.}

\item{nest}{When \code{response} is a matrix, a vector of length equal to
the number of responses per individual indicating which responses belong to
which nesting category. Categoriess must be consecutive increasing integers.
This option should always be specified if nesting is present. Ignored if
response has class, \code{repeated}.}

\item{torder}{When the \code{linear model} is chosen, order of the
polynomial in time to be fitted.}

\item{interaction}{Vector of length equal to the number of time-constant
covariates, giving the levels of interactions between them and the
polynomial in time in the \code{linear model}.}

\item{transform}{Transformation of the response variable: \code{identity},
\code{exp}, \code{square}, \code{sqrt}, or \code{log}.}

\item{link}{Link function for the location: \code{identity}, \code{exp},
\code{square}, \code{sqrt}, or \code{log}. For the \code{linear model}, if
not the \code{identity}, initial estimates of the regression parameters must
be supplied (intercept, polynomial in time, time-constant covariates,
time-varying covariates, in that order).}

\item{autocorr}{The form of the autocorrelation function: \code{exponential}
is the usual \eqn{\rho^{|t_i-t_j|}}{rho^|t_i-t_j|}; \code{gaussian} is
\eqn{\rho^{(t_i-t_j)^2}}{rho^((t_i-t_j)^2)}; \code{cauchy} is
\eqn{1/(1+\rho(t_i-t_j)^2)}{1/(1+rho(t_i-t_j)^2)}; \code{spherical} is
\eqn{((|t_i-t_j|\rho)^3-3|t_i-t_j|\rho+2)/2}{((|t_i-t_j|rho)^3-3|t_i-t_j|rho+2)/2}
for \eqn{|t_i-t_j|\leq1/\rho}{|t_i-t_j|<=1/rho} and zero otherwise;
\code{IOU} is the integrated Ornstein-Uhlenbeck process, \eqn{(2\rho
\min(t_i,t_j)+\exp(-\rho t_i) }{(2rho min(t_i,t_j)+exp(-rho t_i) +exp(-rho
t_j)-1 -exp(rho|ti-t_j|))/2rho^3}\eqn{+\exp(-\rho t_j)-1
-\exp(\rho|ti-t_j|))/2\rho^3}{(2rho min(t_i,t_j)+exp(-rho t_i) +exp(-rho
t_j)-1 -exp(rho|ti-t_j|))/2rho^3}.}

\item{pell}{Initial estimate of the power parameter of the multivariate
power exponential distribution, of the degrees of freedom parameter of the
multivariate Student t distribution, or of the asymmetry parameter of the
multivariate Laplace distribution. If not supplied for the latter, asymmetry
depends on the regression equation in \code{model}.}

\item{preg}{Initial parameter estimates for the regression model. Only
required for \code{linear model} if the \code{link} is not the
\code{identity} or a variance (dispersion) function is fitted.}

\item{covfn}{Either a function or a formula beginning with ~, specifying how
the covariance depends on covariates: either a linear regression function in
the Wilkinson and Rogers notation or a general function with named unknown
parameters.}

\item{pvar}{Initial parameter estimate for the variance or dispersion. If
more than one value is provided, the log variance/dispersion depends on a
polynomial in time. With the \code{pkpd model}, if four values are supplied,
a nonlinear regression for the variance/dispersion is fitted.}

\item{varfn}{The builtin variance (dispersion) function has the
variance/dispersion proportional to a function of the location: pvar*v(mu) =
\code{identity} or \code{square}. If pvar contains two initial values, an
additive constant is included: pvar(1)+pvar(2)*v(mu). Otherwise, either a
function or a formula beginning with ~, specifying either a linear
regression function in the Wilkinson and Rogers notation or a general
function with named unknown parameters for the log variance can be supplied.
If it contains unknown parameters, the keyword \code{mu} may be used to
specify a function of the location parameter.}

\item{par}{If supplied, an initial estimate for the autocorrelation
parameter.}

\item{pre}{Zero, one or two parameter estimates for the variance components,
depending on the number of levels of nesting. If covfn is specified, this
contains the initial estimates of the regression parameters.}

\item{delta}{Scalar or vector giving the unit of measurement for each
response value, set to unity by default. For example, if a response is
measured to two decimals, \code{delta=0.01}. Ignored if response has class,
\code{response} or \code{repeated}.}

\item{shfn}{If TRUE, the supplied variance (dispersion) function depends on
the mean function. The name of this mean function must be the last argument
of the variance/dispersion function.}

\item{common}{If TRUE, \code{mu} and \code{varfn} must both be either
functions with, as argument, a vector of parameters having some or all
elements in common between them so that indexing is in common between them
or formulae with unknowns. All parameter estimates must be supplied in
\code{preg}. If FALSE, parameters are distinct between the two functions and
indexing starts at one in each function.}

\item{twins}{Only possible when there are two observations per individual
(e.g. twin data). If TRUE and \code{covfn} is supplied, allows the
covariance to vary across pairs of twins with the diagonal "variance" of the
covariance matrix remaining constant.}

\item{envir}{Environment in which model formulae are to be interpreted or a
data object of class, \code{repeated}, \code{tccov}, or \code{tvcov}; the
name of the response variable should be given in \code{response}. If
\code{response} has class \code{repeated}, it is used as the environment.}

\item{print.level}{Arguments for nlm.}

\item{ndigit}{Arguments for nlm.}

\item{gradtol}{Arguments for nlm.}

\item{steptol}{Arguments for nlm.}

\item{iterlim}{Arguments for nlm.}

\item{fscale}{Arguments for nlm.}

\item{stepmax}{Arguments for nlm.}

\item{typsize}{Arguments for nlm.}

\item{object}{An object of class, \code{elliptic}.}

\item{...}{additional arguments.}

\item{recursive}{If TRUE, recursive residuals or fitted values are given;
otherwise, marginal ones. In all cases, raw residuals are returned, not
standardized by the standard deviation (which may be changing with
covariates or time).}

\item{x}{An object of class, \code{elliptic}.}

\item{digits}{number of digits to print.}

\item{correlation}{logical; print correlations.}
}
\value{
A list of class \code{elliptic} is returned that contains all of the
relevant information calculated, including error codes.
}
\description{
\code{elliptic} fits special cases of the multivariate
elliptically-contoured distribution, the multivariate normal, Student t, and
power exponential distributions. The latter includes the multivariate normal
(power=1), a multivariate Laplace (power=0.5), and the multivariate uniform
(power -> infinity) distributions as special cases. As well, another form of
multivariate skew Laplace distribution is also available.
}
\details{
With two levels of nesting, the first is the individual and the second will
consist of clusters within individuals.

For clustered (non-longitudinal) data, where only random effects will be
fitted, \code{times} are not necessary.

This function is designed to fit linear and nonlinear models with
time-varying covariates observed at arbitrary time points. A continuous-time
AR(1) and zero, one, or two levels of nesting can be handled. Recall that
zero correlation (all zeros off-diagonal in the covariance matrix) only
implies independence for the multivariate normal distribution.

Nonlinear regression models can be supplied as formulae where parameters are
unknowns in which case factor variables cannot be used and parameters must
be scalars. (See \code{\link[rmutil]{finterp}}.)

Recursive fitted values and residuals are only available for the
multivariate normal distribution with a linear model without a variance
function and with either an AR(1) of \code{exponential} form and/or one
level of random effect. In these cases, marginal and individual profiles can
be plotted using \code{\link[rmutil]{mprofile}} and
\code{\link[rmutil]{iprofile}} and residuals with
\code{\link[rmutil]{plot.residuals}}.
}
\section{Methods (by generic)}{
\itemize{
\item \code{deviance}: Deviance method

\item \code{fitted}: Fitted method

\item \code{residuals}: Residuals method

\item \code{print}: Print method
}}
\examples{

# linear models
y <- matrix(rnorm(40),ncol=5)
x1 <- gl(2,4)
x2 <- gl(2,1,8)
# independence with time trend
elliptic(y, ccov=~x1, torder=2)
# AR(1)
elliptic(y, ccov=~x1, torder=2, par=0.1)
elliptic(y, ccov=~x1, torder=3, interact=3, par=0.1)
# random intercept
elliptic(y, ccov=~x1+x2, interact=c(2,0), torder=3, pre=2)
#
# nonlinear models
time <- rep(1:20,2)
dose <- c(rep(2,20),rep(5,20))
mu <- function(p) exp(p[1]-p[3])*(dose/(exp(p[1])-exp(p[2]))*
	(exp(-exp(p[2])*time)-exp(-exp(p[1])*time)))
shape <- function(p) exp(p[1]-p[2])*time*dose*exp(-exp(p[1])*time)
conc <- matrix(rnorm(40,mu(log(c(1,0.3,0.2))),sqrt(shape(log(c(0.1,0.4))))),
	ncol=20,byrow=TRUE)
conc[,2:20] <- conc[,2:20]+0.5*(conc[,1:19]-matrix(mu(log(c(1,0.3,0.2))),
	ncol=20,byrow=TRUE)[,1:19])
conc <- ifelse(conc>0,conc,0.01)
# with builtin function
# independence
elliptic(conc, model="pkpd", preg=log(c(0.5,0.4,0.1)), dose=c(2,5))
# AR(1)
elliptic(conc, model="pkpd", preg=log(c(0.5,0.4,0.1)), dose=c(2,5),
	par=0.1)
# add variance function
elliptic(conc, model="pkpd", preg=log(c(0.5,0.4,0.1)), dose=c(2,5),
	par=0.1, varfn=shape, pvar=log(c(0.5,0.2)))
# multivariate power exponential distribution
elliptic(conc, model="pkpd", preg=log(c(0.5,0.4,0.1)), dose=c(2,5),
	par=0.1, varfn=shape, pvar=log(c(0.5,0.2)), pell=1,
	distribution="power exponential")
# multivariate Student t distribution
elliptic(conc, model="pkpd", preg=log(c(0.5,0.4,0.1)), dose=c(2,5),
	par=0.1, varfn=shape, pvar=log(c(0.5,0.2)), pell=5,
	distribution="Student t")
# multivariate Laplace distribution
elliptic(conc, model="pkpd", preg=log(c(0.5,0.4,0.1)), dose=c(2,5),
	par=0.1, varfn=shape, pvar=log(c(0.5,0.2)),
	distribution="Laplace")
# or equivalently with user-specified function
# independence
elliptic(conc, model=mu, preg=log(c(0.5,0.4,0.1)))
# AR(1)
elliptic(conc, model=mu, preg=log(c(0.5,0.4,0.1)), par=0.1)
# add variance function
elliptic(conc, model=mu, preg=log(c(0.5,0.4,0.1)), par=0.1,
	varfn=shape, pvar=log(c(0.5,0.2)))
# multivariate power exponential distribution
elliptic(conc, model=mu, preg=log(c(0.5,0.4,0.1)), par=0.1,
	varfn=shape, pvar=log(c(0.5,0.2)), pell=1,
	distribution="power exponential")
# multivariate Student t distribution
elliptic(conc, model=mu, preg=log(c(0.5,0.4,0.1)), par=0.1,
	varfn=shape, pvar=log(c(0.5,0.2)), pell=5,
	distribution="Student t")
# multivariate Laplace distribution
elliptic(conc, model=mu, preg=log(c(0.5,0.4,0.1)), par=0.1,
	varfn=shape, pvar=log(c(0.5,0.2)), pell=5,
	distribution="Laplace")
# or with user-specified formula
# independence
elliptic(conc, model=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*time)-exp(-exp(absorption)*time)),
	preg=list(absorption=log(0.5),elimination=log(0.4),
	volume=log(0.1)))
# AR(1)
elliptic(conc, model=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*time)-exp(-exp(absorption)*time)),
	preg=list(absorption=log(0.5),elimination=log(0.4),volume=log(0.1)),
	par=0.1)
# add variance function
elliptic(conc, model=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*time)-exp(-exp(absorption)*time)),
	preg=list(absorption=log(0.5),elimination=log(0.4),volume=log(0.1)),
	varfn=~exp(b1-b2)*time*dose*exp(-exp(b1)*time),
	par=0.1, pvar=list(b1=log(0.5),b2=log(0.2)))
# variance as function of the mean
elliptic(conc, model=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*time)-exp(-exp(absorption)*time)),
	preg=list(absorption=log(0.5),elimination=log(0.4),volume=log(0.1)),
	varfn=~d*log(mu),shfn=TRUE,par=0.1, pvar=list(d=1))
# multivariate power exponential distribution
elliptic(conc, model=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*time)-exp(-exp(absorption)*time)),
	preg=list(absorption=log(0.5),elimination=log(0.4),volume=log(0.1)),
	varfn=~exp(b1-b2)*time*dose*exp(-exp(b1)*time),
	par=0.1, pvar=list(b1=log(0.5),b2=log(0.2)), pell=1,
	distribution="power exponential")
# multivariate Student t distribution
elliptic(conc, model=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*time)-exp(-exp(absorption)*time)),
	preg=list(absorption=log(0.5),elimination=log(0.4),volume=log(0.1)),
	varfn=~exp(b1-b2)*time*dose*exp(-exp(b1)*time),
	par=0.1, pvar=list(b1=log(0.5),b2=log(0.2)), pell=5,
	distribution="Student t")
# multivariate Laplace distribution
elliptic(conc, model=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*time)-exp(-exp(absorption)*time)),
	preg=list(absorption=log(0.5),elimination=log(0.4),volume=log(0.1)),
	varfn=~exp(b1-b2)*time*dose*exp(-exp(b1)*time),
	par=0.1, pvar=list(b1=log(0.5),b2=log(0.2)), pell=5,
	distribution="Laplace")
#
# generalized logistic regression with square-root transformation
# and square  link
time <- rep(seq(10,200,by=10),2)
mu <- function(p) {
	yinf <- exp(p[2])
	yinf*(1+((yinf/exp(p[1]))^p[4]-1)*exp(-yinf^p[4]
		*exp(p[3])*time))^(-1/p[4])}
y <- matrix(rnorm(40,sqrt(mu(c(2,1.5,0.05,-2))),0.05)^2,ncol=20,byrow=TRUE)
y[,2:20] <- y[,2:20]+0.5*(y[,1:19]-matrix(mu(c(2,1.5,0.05,-2)),
	ncol=20,byrow=TRUE)[,1:19])
y <- ifelse(y>0,y,0.01)
# with builtin function
# independence
elliptic(y, model="logistic", preg=c(2,1,0.1,-1), trans="sqrt",
	link="square")
# the same model with AR(1)
elliptic(y, model="logistic", preg=c(2,1,0.1,-1), trans="sqrt",
	link="square", par=0.4)
# the same model with AR(1) and one component of variance
elliptic(y, model="logistic", preg=c(2,1,0.1,-1),
	trans="sqrt", link="square", pre=1, par=0.4)
# or equivalently with user-specified function
# independence
elliptic(y, model=mu, preg=c(2,1,0.1,-1), trans="sqrt",
	link="square")
# the same model with AR(1)
elliptic(y, model=mu, preg=c(2,1,0.1,-1), trans="sqrt",
	link="square", par=0.4)
# the same model with AR(1) and one component of variance
elliptic(y, model=mu, preg=c(2,1,0.1,-1),
	trans="sqrt", link="square", pre=1, par=0.4)
# or equivalently with user-specified formula
# independence
elliptic(y, model=~exp(yinf)*(1+((exp(yinf-y0))^b4-1)*
	exp(-exp(yinf*b4+b3)*time))^(-1/b4),
	preg=list(y0=2,yinf=1,b3=0.1,b4=-1), trans="sqrt", link="square")
# the same model with AR(1)
elliptic(y, model=~exp(yinf)*(1+((exp(yinf-y0))^b4-1)*
	exp(-exp(yinf*b4+b3)*time))^(-1/b4),
	preg=list(y0=2,yinf=1,b3=0.1,b4=-1), trans="sqrt",
	link="square", par=0.1)
# add one component of variance
elliptic(y, model=~exp(yinf)*(1+((exp(yinf-y0))^b4-1)*
	exp(-exp(yinf*b4+b3)*time))^(-1/b4),
	preg=list(y0=2,yinf=1,b3=0.1,b4=-1),
	trans="sqrt", link="square", pre=1, par=0.1)
#
# multivariate power exponential and Student t distributions for outliers
y <- matrix(rcauchy(40,mu(c(2,1.5,0.05,-2)),0.05),ncol=20,byrow=TRUE)
y[,2:20] <- y[,2:20]+0.5*(y[,1:19]-matrix(mu(c(2,1.5,0.05,-2)),
	ncol=20,byrow=TRUE)[,1:19])
y <- ifelse(y>0,y,0.01)
# first with normal distribution
elliptic(y, model="logistic", preg=c(1,1,0.1,-1))
elliptic(y, model="logistic", preg=c(1,1,0.1,-1), par=0.5)
# then power exponential
elliptic(y, model="logistic", preg=c(1,1,0.1,-1), pell=1,
	distribution="power exponential")
elliptic(y, model="logistic", preg=c(1,1,0.1,-1), par=0.5, pell=1,
	distribution="power exponential")
# finally Student t
elliptic(y, model="logistic", preg=c(1,1,0.1,-1), pell=1,
	distribution="Student t")
elliptic(y, model="logistic", preg=c(1,1,0.1,-1), par=0.5, pell=1,
	distribution="Student t")

}
\author{
J.K. Lindsey
}
\references{
Lindsey, J.K. (1999) Multivariate elliptically-contoured
distributions for repeated measurements. Biometrics 55, 1277-1280.

Kotz, S., Kozubowski, T.J., and Podgorski, K. (2001) The Laplace
Distribution and Generalizations. A Revisit with Applications to
Communications, Economics, Engineering, and Finance. Basel: Birkhauser, Ch.
6.
}
\seealso{
\code{\link[growth]{carma}}, \code{\link[rmutil]{dpowexp}},
\code{\link[rmutil]{dskewlaplace}}, \code{\link[rmutil]{finterp}},
\code{\link[repeated]{gar}}, \code{\link[rmutil]{gettvc}},
\code{\link[repeated]{gnlmix}}, \code{\link[repeated]{glmm}},
\code{\link[repeated]{gnlmm}}, \code{\link[gnlm]{gnlr}},
\code{\link[rmutil]{iprofile}}, \code{\link[repeated]{kalseries}},
\code{\link[rmutil]{mprofile}}, \code{\link[growth]{potthoff}},
\code{\link[rmutil]{read.list}}, \code{\link[rmutil]{restovec}},
\code{\link[rmutil]{rmna}}, \code{\link[rmutil]{tcctomat}},
\code{\link[rmutil]{tvctomat}}.
}
\keyword{models}

