% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg.R
\name{summary.gkwreg}
\alias{summary.gkwreg}
\title{Summary Method for Generalized Kumaraswamy Regression Models}
\usage{
\method{summary}{gkwreg}(object, conf.level = 0.95, ...)
}
\arguments{
\item{object}{An object of class \code{"gkwreg"}, typically the result of a
call to \code{\link{gkwreg}}.}

\item{conf.level}{Numeric. The desired confidence level for constructing
confidence intervals for the regression coefficients. Default is 0.95.}

\item{...}{Additional arguments, currently ignored by this method.}
}
\value{
An object of class \code{"summary.gkwreg"}, which is a list containing
the following components:
\item{call}{The original function call that created the \code{object}.}
\item{family}{Character string specifying the distribution family.}
\item{coefficients}{A data frame (matrix) containing the coefficient estimates,
standard errors, z-values, and p-values.}
\item{conf.int}{A matrix containing the lower and upper bounds of the confidence
intervals for the coefficients (if standard errors are available).}
\item{link}{A list of character strings specifying the link functions used.}
\item{fitted_parameters}{A list containing the mean values of the estimated
distribution parameters.}
\item{residuals}{A named numeric vector containing summary statistics for the
response residuals.}
\item{nobs}{Number of observations used in the fit.}
\item{npar}{Total number of estimated regression coefficients.}
\item{df.residual}{Residual degrees of freedom.}
\item{loglik}{The maximized log-likelihood value.}
\item{aic}{Akaike Information Criterion.}
\item{bic}{Bayesian Information Criterion.}
\item{rmse}{Root Mean Squared Error of the residuals.}
\item{efron_r2}{Efron's pseudo-R-squared value.}
\item{mean_absolute_error}{Mean Absolute Error of the residuals.}
\item{convergence}{Convergence code from the optimizer.}
\item{iterations}{Number of iterations reported by the optimizer.}
\item{conf.level}{The confidence level used for calculating intervals.}
}
\description{
Computes and returns a detailed statistical summary for a fitted Generalized
Kumaraswamy (GKw) regression model object of class \code{"gkwreg"}.
}
\details{
This method provides a comprehensive summary of the fitted \code{gkwreg} model.
It calculates z-values and p-values for the regression coefficients based on
the estimated standard errors (if available) and computes confidence intervals
at the specified \code{conf.level}. The summary includes:
\itemize{
\item The model call.
\item The distribution family used.
\item A table of coefficients including estimates, standard errors, z-values,
and p-values. Note: Significance stars are typically added by the
corresponding \code{print.summary.gkwreg} method.
\item Confidence intervals for the coefficients.
\item Link functions used for each parameter.
\item Mean values of the fitted distribution parameters (\eqn{\alpha, \beta, \gamma, \delta, \lambda}).
\item A five-number summary (Min, Q1, Median, Q3, Max) plus the mean of the
response residuals.
\item Key model fit statistics (Log-likelihood, AIC, BIC, RMSE, Efron's R^2).
\item Information about model convergence and optimizer iterations.
}
If standard errors were not computed (e.g., \code{hessian = FALSE} in the
original \code{gkwreg} call), the coefficient table will only contain estimates,
and confidence intervals will not be available.
}
\examples{
\donttest{
set.seed(123)
n <- 100
x1 <- runif(n, -2, 2)
x2 <- rnorm(n)
alpha_coef <- c(0.8, 0.3, -0.2)
beta_coef <- c(1.2, -0.4, 0.1)
eta_alpha <- alpha_coef[1] + alpha_coef[2] * x1 + alpha_coef[3] * x2
eta_beta <- beta_coef[1] + beta_coef[2] * x1 + beta_coef[3] * x2
alpha_true <- exp(eta_alpha)
beta_true <- exp(eta_beta)
# Use stats::rbeta as a placeholder if rkw is unavailable
y <- stats::rbeta(n, shape1 = alpha_true, shape2 = beta_true)
y <- pmax(pmin(y, 1 - 1e-7), 1e-7)
df <- data.frame(y = y, x1 = x1, x2 = x2)

# Fit a Kumaraswamy regression model
kw_reg <- gkwreg(y ~ x1 + x2 | x1 + x2, data = df, family = "kw")

# Generate detailed summary using the summary method
summary_kw <- summary(kw_reg)

# Print the summary object (uses print.summary.gkwreg)
print(summary_kw)

# Extract coefficient table directly from the summary object
coef_table <- coef(summary_kw) # Equivalent to summary_kw$coefficients
print(coef_table)
}

}
\seealso{
\code{\link{gkwreg}}, \code{\link{print.summary.gkwreg}},
\code{\link[stats]{coef}}, \code{\link[stats]{confint}}
}
\author{
Lopes, J. E.
}
\keyword{models}
\keyword{regression}
\keyword{summary}
