#' @name network
#' @title Generate Haplotype Net Relationshop with Haplotype Result
#' @description computes a haplotype network with haplotype summary result
#' @seealso
#' \code{\link[geneHapR:plotHapNet]{plotHapNet()}} and
#' \code{\link[geneHapR:hap_summary]{hap_summary()}}.
#' @usage
#' get_hapNet(hapSummary,
#'            AccINFO = AccINFO,
#'            groupName = groupName,
#'            na.label = "Unknown")
#' @inherit plotHapNet examples
#' @importFrom pegas haploNet
#' @importFrom stringdist stringdist
#' @references
#' Mark P.J. van der Loo (2014)
#' <doi:10.32614/RJ-2014-011>;
#'
#' E. Paradis (2010) <doi:10.1093/bioinformatics/btp696>
#' @param hapSummary object of `hapSummary` class, generated by `hap_summary()`
#' @param AccINFO data.frame, specified groups of each accession.
#' Used for pie plot. If missing, pie will not draw in plotHapNet.
#' Or you can supplied a hap_group mattrix with `plot(hapNet, pie = hap_group)`.
#' @param groupName the group name used for pie plot,
#' should be in `AccINFO` column names, default as the first column name
#' @param na.label the label of `NA`s
#' @return haplonet class
#' @export
get_hapNet <-
    function(hapSummary,
             AccINFO = AccINFO,
             groupName = groupName,
             na.label = "Unknown") {
        if (!inherits(hapSummary, "hapSummary"))
            stop("'hapSummary' should be of 'hapSummary' class")
        d <- getStringDist(hapSummary)
        hapNet <- pegas::haploNet(as.haplotype(hapSummary), d)
        if (!missing(AccINFO)) {
            if (missing(groupName))
                groupName <- colnames(AccINFO)[1]
            hapGroup <- getHapGroup(hapSummary,
                                    AccINFO = AccINFO,
                                    groupName = groupName,
                                    na.label = na.label)
            attr(hapNet, "hapGroup") <- hapGroup
        }
        return(hapNet)
    }


#' @name plotHapNet
#' @title plotHapNet
#' @importFrom graphics lines locator strheight text legend
#' @importFrom stats median
#' @usage
#' plotHapNet(hapNet,
#'            size = "freq",
#'            scale = 1,
#'            cex = 0.8,
#'            cex.legend = 0.6,
#'            col.link = 1,
#'            link.width = 1,
#'            show.mutation = 1,
#'            backGround = backGround,
#'            hapGroup = hapGroup,
#'            legend = FALSE,
#'            main = main,
#'            labels = TRUE,
#'            ...)
#' @param hapNet an object of class "haploNet"
#' @param size a numeric vector giving the diameter of the circles
#'  representing the haplotypes: this is in the same unit than the
#'  links and eventually recycled.
#' @param scale a numeric indicate the ratio of the scale of the links
#' representing the number of steps on the scale of the circles
#' representing the haplotypes or a character one of `c('log10', 'log2')`
#' indicate the scale method by `log10(size)` or `log2(size)`, respectively.
#' Default as 1
#' @param col.link a character vector specifying the colours of the links;
#' eventually recycled.
#' @param link.width a numeric vector giving the width of the links;
#' eventually recycled.
#' @param show.mutation an integer value:
#'
#' if 0, nothing is drawn on the links;
#'
#' if 1, the mutations are shown with small segments on the links;
#'
#' if 2, they are shown with small dots;
#'
#' if 3, the number of mutations are printed on the links.
#' @param backGround a color vector with length equal to number of
#' Accession types
#' @param hapGroup a matrix used to draw pie charts for each haplotype;
#' its number of rows must be equal to the number of haplotypes
#' @param cex character expansion factor relative to current par("cex")
#' @param cex.legend same as `cex`, but for text in legend
#' @param scale a numeric indicate the ratio of the scale of the links
#' representing the number of steps on the scale of the circles
#' representing the haplotypes or a character one of `c('log10', 'log2')`
#' indicate the scale method by `log10(size)` or `log2(size)`, respectively.
#' Default as 1
#' @param legend a logical specifying whether to draw the legend,
#' or a vector of length two giving the coordinates where to draw the legend;
#' `FALSE` by default.
#' If `TRUE`, the user is asked to click where to draw the legend.
#' @param labels a logical specifying whether to identify the haplotypes
#'  with their labels (default as TRUE)
#' @param ... other parameters will pass to `plot` function
#' @inheritParams graphics::title
#' @importFrom graphics title
#' @seealso
#' \code{\link[geneHapR:hap_summary]{hap_summary()}} and
#' \code{\link[geneHapR:get_hapNet]{get_hapNet()}}.
#' @examples
#'
#' \donttest{
#' data("geneHapR_test")
#' hapSummary <- hap_summary(hapResult)
#'
#' # calculate haploNet
#' hapNet <- get_hapNet(hapSummary,
#'                      AccINFO = AccINFO, # accession types
#'                      groupName = colnames(AccINFO)[2])
#'
#' # plot haploNet
#' plot(hapNet)
#'
#' # plot haploNet
#' plotHapNet(hapNet,
#'            size = "freq",   # circle size
#'            scale = "log10", # scale circle with 'log10(size + 1)'
#'            cex = 1, # size of hap symbol
#'            col.link = 2, # link colors
#'            link.width = 2, # link widths
#'            show.mutation = 2, # mutation types one of c(0,1,2,3)
#'            legend = FALSE) # legend position
#' }
#' @return No return value
#' @export
plotHapNet <- function(hapNet,
                       size = "freq",
                       scale = 1,
                       cex = 0.8,
                       cex.legend = 0.6,
                       col.link = 1,
                       link.width = 1,
                       show.mutation = 1,
                       backGround = backGround,
                       hapGroup = hapGroup,
                       legend = FALSE,
                       main = main,
                       labels = TRUE,
                       ...) {
    if (!inherits(hapNet, "haploNet"))
        stop("'hapNet' must be of 'haploNet' class")

    if (missing(hapGroup))
        hapGroup <- attr(hapNet, "hapGroup")


    if (size == "freq")
        size <- attr(hapNet, "freq")
    else
        if (!is.numeric(size))
            stop("'size' should be 'freq' or a numeric vector")

    if (is.numeric(scale)){
        size.sc <- size/scale
    } else if(scale == "log10"){
        size.sc <- log10(size + 1)
    } else if(scale == "log2"){
        size.sc <- log2(size + 1)
    } else {
        warning("Scale should be one of 'log10' or 'log2' or a numeric, ",
                "scale will reset as 1")
    }


    if (!is.null(hapGroup)) {
        if (missing(backGround))
            backGround <- rainbow(ncol(hapGroup))

        plot(
            hapNet,
            col.link = col.link,
            threshold = 10,
            size = size.sc,
            cex = cex,
            show.mutation = show.mutation,
            lwd = link.width,
            bg = backGround,
            pie = hapGroup,
            labels = labels,
            ...
        )

    } else {
        if (missing(backGround))
            backGround <- "grey90"

        plot(
            hapNet,
            col.link = col.link,
            bg = backGround,
            size = size.sc,
            cex = cex,
            show.mutation = show.mutation,
            lwd = link.width,
            labels = labels,
            ...
        )
    }

    # add legend
    if(legend[1]){
        # get position
        if (is.logical(legend)) {
            cat("Click where you want to draw the legend (lefttop)")
            xy <- unlist(locator(1))
            cat("\nThe coordinates x = ", xy[1],
                ", y = ", xy[2], " are used\n", sep = "")
        } else {
            if (!is.numeric(legend) || length(legend) < 2)
                stop("wrong coordinates of legend")
            xy <- legend
        }

        # add size legend
        SZ <- unique(size)
        SZ.sc <- unique(size.sc)

        if (length(SZ) > 1) {
            # calculate size legend
            SZ.sc.50 <- (min(SZ.sc) + max(SZ.sc)) * 0.5
            SZ.sc.25 <- (min(SZ.sc) + SZ.sc.50) * 0.5
            SZ.sc.75 <- (SZ.sc.50 + max(SZ.sc)) * 0.5
            SZ.sc <- unique(c(min(SZ.sc),
                              SZ.sc.25,
                              SZ.sc.50,
                              SZ.sc.75,
                              max(SZ.sc)))
            if(is.numeric(scale)) SZ <- SZ.sc * scale else
                SZ <- switch (scale,
                              "log10" = 10^SZ.sc - 1,
                              "log2" = 2^SZ.sc - 1)
            SZ <- ceiling(SZ)

            # plot size legend
            SHIFT <- max(SZ.sc) * 0.5
            vspace <- strheight(" ")
            for (sz.sc in SZ.sc) {
                seqx <- seq(-sz.sc/2, 0, length.out = 100)
                seqy <- sqrt((sz.sc/2)^2 - seqx^2)
                seqx <- seqx + xy[1] + SHIFT
                seqy <- xy[2] + seqy - SHIFT
                lines(seqx, seqy)
                text(seqx[100], seqy[100], SZ[match(sz.sc, SZ.sc)], adj = c(-0.1, 0.5), cex = cex.legend)
            }
            xy[2] <- xy[2] - SHIFT - vspace
        }

        # add color legend
        if (!is.null(hapGroup)) {
            legend(
                x = xy[1],
                y = xy[2],
                legend = colnames(hapGroup),
                fill = backGround,
                cex = cex.legend,
                ...
            )
        }
    }

    # Add title
    if(!missing(main))
        title(main = main)
}


#' @name ashaplotype
#' @title as.haplotype
#' @usage
#' as.haplotype(hap)
#' @description convert `hapSummary` or `hapResult` class into `haplotype` class (pegas)
#' @note It's not advised for `hapSummary` or `hapResult` with indels, due to indels will
#' convert to SNPs with equal length of each indel.
#' @importFrom ape as.DNAbin
#' @examples
#' data("geneHapR_test")
#' hap <- as.haplotype(hapResult)
#' hapSummary <- hap_summary(hapResult)
#' hap <- as.haplotype(hapSummary)
#' @param hap object of `hapSummary` or `hapResult` class
#' @return haplotype class
#' @export
as.haplotype <- function(hap) {
    if (inherits(hap, "hapResult"))
        hap <- hap_summary(hap)
    if (!inherits(hap, "hapSummary"))
        stop("'hap' must be of 'hapResult' or 'hapSummary' class")
    # get freq
    freq <- hap$freq
    names(freq) <- hap$Hap
    freq <- freq[!is.na(freq)]

    # get hap mattrix
    hapDNAset <- hap2string(hap = hap, type = "DNA")
    hapBin <- ape::as.DNAbin(hapDNAset)
    hapmatt <- unlist(as.character(as.matrix(hapBin)))

    # set as haplotype
    class(hapmatt) <- c("haplotype", "character")
    rownames(hapmatt) <- names(freq)
    N <- nrow(hapmatt)
    attr(hapmatt, "index") <-
        lapply(1:N, function(i)
            seq_len(freq[i]))
    return(hapmatt)
}


#' @importFrom stringr str_detect str_pad str_length
#' @importFrom Biostrings DNAStringSet
hap2string <- function(hap, type = "DNA") {
    colNms <- colnames(hap)
    if ("Accession" %in% colNms)
        hap <- hap[, colnames(hap) != 'Accession']
    if ("freq" %in% colNms)
        hap <- hap[, colnames(hap) != 'freq']
    if (nrow(hap) <= 5)
        stop("There is only one Hap ?")
    meta <- hap[1:4, ]
    hap <- hap[5:nrow(hap), ]
    ALLELE <- meta[meta[, 1] == "ALLELE", ]

    # padding multiallelic indel sites
    if (type == "DNA") {
        multi_probe <- is.indel.allele(ALLELE)
        for (c in seq_len(ncol(hap))) {
            if (multi_probe[c]) {
                if (stringr::str_sub(ALLELE[c], 2, 2) == "/")
                    side = "right"
                else
                    side = "left"
                maxLen <- max(stringr::str_length(hap[, c]))
                hap[, c] <- stringr::str_pad(hap[, c],
                                             width = maxLen,
                                             side = side,
                                             pad = "-")
            }
        }

        # conneting strings
        DNASeqs <- sapply(seq_len(nrow(hap)),
                          function(i)
                              paste0(hap[i, -1], collapse = ""))
        names(DNASeqs) <- hap$Hap
        hapString <-
            Biostrings::DNAStringSet(DNASeqs, use.names = T, start = 1)
    } else {
        for (c in 2:ncol(hap)) {
            ALc <- ALLELE[c]
            ALs <- unlist(stringr::str_split(ALc, "[,/]"))
            ALn <- LETTERS[seq_len(length(ALs))]
            names(ALn) <- ALs
            hap[, c] <- ALn[hap[, c]]
        }

        hapString <- sapply(seq_len(nrow(hap)),
                            function(i)
                                paste0(hap[i, -1], collapse = ""))
        names(hapString) <- hap$Hap
    }



    return(hapString)
}



#' @importFrom stringdist stringdist
getStringDist <- function(hapSummary) {
    hapStrings <- hap2string(hapSummary, type = "LETTER")
    n <- names(hapStrings)
    l <- length(hapStrings)
    d <- matrix(nrow = l,
                ncol = l,
                dimnames = list(n, n))
    for (i in seq_len(length(hapStrings))) {
        for (j in seq_len(length(hapStrings))) {
            d[i, j] <- stringdist::stringdist(hapStrings[i],
                                              hapStrings[j],
                                              method = "lv")
        }
    }
    d <- d[lower.tri(d)]
    attr(d, "Size") <- l
    attr(d, "Labels") <- n
    attr(d, "method") <- "lv"
    attr(d, "Diag") <- attr(d, "Upper") <- FALSE
    class(d) <- "dist"
    return(d)
}


getHapGroup <- function(hap,
                        AccINFO = AccINFO,
                        groupName = groupName,
                        na.label = na.label) {
    # get indvidual group number of each hap
    accs.hap <- attr(hap, "hap2acc")
    accs.info <- row.names(AccINFO)
    probe <- accs.hap %in% accs.info
    haps <- names(accs.hap)
    infos <- AccINFO[accs.hap, groupName]
    infos[is.na(infos)] <- na.label
    acc.infos <- data.frame(Hap = haps, Type = infos)
    if(FALSE %in% probe){
        l <- table(probe)["FALSE"]
        l <- ifelse(l >= 3, 3, l)
        warning(table(probe)["FALSE"],
                 " accession(s) not in 'AccINFO', eg.: ",
                 paste(accs.hap[!probe][seq_len(l)], collapse = ", "))
        message("Type of those accessions will be assigned ", na.label)
    }

    with(acc.infos,
         table(hap = acc.infos$Hap, group = acc.infos$Type))
}
