% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_rates_from_prices.R
\name{get_rates_from_prices}
\alias{get_rates_from_prices}
\title{Calculate rates of return for given prices.}
\usage{
get_rates_from_prices(prices, quote = c("Open", "Close"), multi_day = TRUE,
  compounding = c("discrete", "continuous"))
}
\arguments{
\item{prices}{an object containing prices of securities. Three classes are
allowed: \code{list}, \code{data.frame}, and \code{zoo}.}

\item{quote}{a character vector specifying the type of the quote:
\code{"Open"} (default) or \code{"Close"}.}

\item{multi_day}{logical, is a rate of return between more than 1 day is
allowed?}

\item{compounding}{a character vector defining the type of compounding:
\code{"continuous"} (default) or \code{"discrete"}.}
}
\value{
Rates of returns of the same class as prices.
}
\description{
\code{get_rates_from_prices} is used for computing rates of return from
prices for different classes.
}
\details{
This is a generic function, dispatched for such classes as \code{list},
\code{data.frame}, and \code{zoo} that represent prices.

The calculation is made in C++ (\code{Rcpp}) in favor of speed.

If \code{prices} is a data frame, than the first column should be of the
class \code{Date} and contain ordered dates of prices.

The correspondence between dates and values of the rates depends on the
quote, which can be either Open or Close. If the quote is Open, than the
value of rate belongs to the former date. Otherwise, to the latter one. This
is also applied for the algorithm, if multiday is allowed: the value of the
rate of return is assigned to the latter day in case of Close price, and to
the former day in in case of Open quote.

The \code{multi_day} parameter specifies how to handle missing values and
weekends. If the value is TRUE, the function ignores missing values
and the rates are calculated between non-missing prices. If it is FALSE, then
only one-day period rates of return are computed (between two consecutive
calendar dates).

The function uses either continuous (by default) or discrete (periodic)
compounding.
}
\examples{
## Download historical prices of nine European insurance companies'
## stocks and estimate rates of returns form prices:
\dontrun{
library("magrittr")
tickers <- c("ALV.DE", "CS.PA", "G.MI", "HNR1.HA", "HSX.L", "MUV2.DE",
             "RSA.L", "TOP.CO")
rates <- tickers \%>\%
    get_prices_from_tickers(start = as.Date("2000-01-01"),
                            end = as.Date("2002-01-01"),
                            quote = "Close",
                            retclass = "zoo") \%>\%
    get_rates_from_prices(quote = "Close",
                          multi_day = TRUE,
                          compounding = "continuous")
}
## The result of the above code is stored in:
data(rates)

## Download historical prices of ESTX50 EUR P index and estimate rates of
## returns from prices:
\dontrun{
library("magrittr")
rates_indx <- get_prices_from_tickers("^STOXX50E",
                                      start = as.Date("2000-01-01"),
                                      end = as.Date("2002-01-01"),
                                      quote = "Close",
                                      retclass = "zoo") \%>\%
    get_rates_from_prices(quote = "Close",
                          multi_day = TRUE,
                          compounding = "continuous")
}
## The result of the above code is stored in:
data(rates_indx)

}
