#' Summarizing Adjusted Limited Dependent Variable Mixture Model Fits
#'
#' The method \code{summary.aldvmm} for the generic function
#' \ifelse{html}{\code{\link[base]{summary}}}{\code{base::summary()}} creates 
#' an object of class "summary.aldvmm" including key results from an object of 
#' class "aldvmm".
#' 
#' @param object an object inheriting from class 'aldvmm'.
#' @param digits an integer value of the number of digits in the output table.
#' @param level a numeric value of the confidence interval between 0 and 1.
#' @param ... further arguments passed to or from other methods.
#'
#' @return \code{summary.aldvmm} returns an object of class "summary.aldvmm" 
#' including the following elements. 
#'   \item{\code{call}}{a character value including the model call captured by
#'   \ifelse{html}{\code{\link[base]{match.call}}}{\code{base::match.call}}.}
#'   \item{\code{summary}}{a data frame generated by 
#'   \ifelse{html}{\code{\link[aldvmm]{aldvmm.sum}}}{\code{aldvmm::aldvmm.sum}} 
#'   including a formatted summary of model results for printing.}
#'   \item{\code{terms}}{a list of objects of class
#'   \ifelse{html}{\code{\link[stats]{terms}}}{\code{stats::terms}}.}
#'   \item{\code{contrasts}}{a nested list of character values showing 
#'   contrasts of factors used in models of component means ("beta") and 
#'   probabilities of component membership ("delta").}
#'   \item{\code{coef}}{a
#'   numeric vector of parameter estimates.}
#'   \item{\code{cov}}{a numeric matrix object with covariances of parameters.}
#'   \item{\code{n}}{a scalar representing the number of complete observations
#'   with no missing values that were used in the estimation.}
#'   \item{\code{df.residual}}{an integer value of the residual 
#'   degrees of freedom.}
#'   \item{\code{df.null}}{an integer value of the residual 
#'   degrees of freedom of a null model including intercepts and standard errors.}
#'   \item{\code{iter}}{an integer value of the number of iterations used in 
#'   optimization.}
#'   \item{\code{ll}}{a numeric value of the negative log-likelihood
#'   \eqn{-ll}.}
#'   \item{\code{aic}}{a numeric value of the Akaike information
#'   criterion \eqn{AIC = 2n_{par} - 2ll}{AIC = 2*npar - 2*ll}.}
#'   \item{\code{bic}}{a numeric value of the Bayesian information criterion
#'   \eqn{BIC = n_{par}*log(n_{obs}) - 2ll}{BIC = npar*log(nobs) - 2*ll}.}
#'   \item{\code{k}}{a numeric value of the number of components.}
#'   \item{\code{lcoef}}{a character vector of labels for objects including
#'   results on distributions (default \code{"beta"}) and the probabilities of
#'   component membership (default \code{"delta"}).} 
#'   \item{\code{lcpar}}{a
#'   character vector of labels for objects including constant distribution
#'   parameters (default \code{"sigma"} for \code{dist = "normal"}).}
#'   \item{\code{lcmp}}{a character value of the label for objects including
#'   results on different components (default "Comp")} 
#'   \item{\code{lvar}}{a
#'   list including 2 character vectors of covariate names for model parameters
#'   of distributions (\code{"beta"}) and the multinomial logit
#'   (\code{"delta"}).}
#'   
#' @method summary aldvmm
#' @rdname summary
#'
#' @export summary.aldvmm
#' 
#' @export

summary.aldvmm <- function(object, 
                           digits = max(3L, getOption("digits") - 3L),
                           level = 0.95,
                           ...) {
  
  # Create summary table
  #---------------------
  
  regtab <- aldvmm.sum(object = object,
                       digits = digits,
                       level = level)
  
  # Create output list
  #-------------------
  
  ans <- list()
  
  ans[["call"]] <- object[["call"]]
  ans[["summary"]] <- format(as.data.frame(regtab), trim = TRUE)
  ans[["terms"]] <- object[["terms"]]
  ans[["contrasts"]] <- object[["contrasts"]]
  ans[["coef"]] <- object[["coef"]]
  ans[["cov"]] <- object[["cov"]]
  ans[["n"]] <- object[["n"]]
  ans[["df.residual"]] <- object[["df.residual"]]
  ans[["df.null"]] <- object[["df.null"]]
  ans[["iter"]] <- object[["iter"]]
  ans[["ll"]] <- object[["gof"]][["ll"]]
  ans[["aic"]] <- object[["gof"]][["aic"]]
  ans[["bic"]] <- object[["gof"]][["bic"]]
  ans[["k"]] <- object[["k"]]
  ans[["lcoef"]] <- object[["label"]][["lcoef"]]
  ans[["lcpar"]] <- object[["label"]][["lcpar"]]
  ans[["lcmp"]] <- object[["label"]][["lcmp"]]
  ans[["lvar"]] <- object[["label"]][["lvar"]]
  
  class(ans) <- "summary.aldvmm"
  return(ans)
}
