#' Convergence of Gibbs Samplers
#'
#' @description This function provides the Separated Partial Means Test for MCMC in \insertCite{geweke2005contemporary;textual}{BayesianFactorZoo}.
#'
#' @param post_draws A matrix of posterior draws with dimension \eqn{ndraws \times k}, where \eqn{k} is the number of model parameters;
#'        and \eqn{ndraws} is the number of posterior draws (\code{post_draws} \eqn{\geq 2 \times M \times} \code{ndraws_test});
#' @param ndraws_test The number of posterior steps that we want to test (we aim to test whether the Gibbs sampler converges within \code{ndraws_test} steps);
#' @param M The number of separated partial means (Default = 5; see \bold{Details}).
#'
#' @details
#'
#' Suppose that the posterior draws \eqn{\{(\theta^{(n)}) \}_{n=1}^N} come from our Gibbs sampling algorithm (e.g., \code{continuous_ss_sdf}).
#' \eqn{N} is the number of posterior draws (\eqn{N = ndraws}), and \eqn{k} is the dimension of \eqn{\theta}.
#' We are interested in estimating the posterior moment \eqn{E [h(\theta)]}, where the function \eqn{h} must be a scalar function.
#' For example, if we want to estimate the posterior mean of \eqn{\gamma_j}, function \eqn{h(\theta) = \gamma_j}.
#' Assume that \eqn{\{ h(\theta^{(m)}) \}_{n=1}^N} is a stationary process with autocovariance function
#' \deqn{c_j = cov (h(\theta^{(n)}), h(\theta^{(n-j)})), \ \ (j = 0, \pm 1, \pm 2, \cdots), }
#' and its empirical counterpart is
#' \deqn{\hat{c}^{(N)}_j = \frac{1}{N} \sum_{n=j+1}^{N} [h(\theta^{(n)}) - \bar{h}^{(N)}]  [h(\theta^{(n-j)}) - \bar{h}^{(N)}].}
#' Under certain mild conditions,
#' \deqn{\hat{\tau}^{2, (N)} = \hat{c}^{(N)}_0 + 2 \sum_{s=1}^{L-1} \frac{L-s}{L} \hat{c}^{(N)}_s  \to  \tau^2,}
#' where \eqn{\tau^2} is the asymptotic variance of \eqn{\bar{h}^{(N)}}:
#' \eqn{\sqrt{N} (\bar{h}^{(N)}-\bar{h}) \to N (0, \tau^2)}.
#'
#'
#' \bold{Separated Partial Means Test for MCMC (Theorem 4.7.4 in \insertCite{geweke2005contemporary;textual}{BayesianFactorZoo})}
#'
#' Suppose that \eqn{|c_j| < c_0 \rho^j \ \ (j=1,2,\cdots)} for some \eqn{\rho \in [0,1)}.
#' Let \eqn{M} be a fixed positive integer. For each \eqn{N} such that \eqn{N_p = \frac{N}{2M}} is an integer,
#' define the \eqn{M} separated partial means:
#' \deqn{\bar{h}_{j,p}^{(N)} = \frac{1}{N_p} \sum_{n=1}^{N_p} h(\theta^{(n+\frac{N(2j-1)}{2M})}) \ \ (j = 1, \cdots, M).}
#' Let \eqn{\hat{\tau}_{j,p}^{2,(N)}} be the estimate of \eqn{\tau^2} computed for \eqn{\bar{h}_{j,p}^{(N)}} (\eqn{j=1,\cdots,M}).
#' Define the \eqn{(M-1) \times 1} vector \eqn{\bar{h}_M^{(N)}} with the \eqn{j}-th element \eqn{\bar{h}_{j+1,M}^{(N)}-\bar{h}_{j,M}^{(N)}},
#' and the \eqn{(M-1) \times (M-1)} tridiagonal matrix \eqn{\hat{V}_M^{(N)}} in which
#' \eqn{\hat{v}_{jj}^{(N)} = \frac{1}{N_p} (\hat{\tau}_{j,M}^{2,(N)}+\hat{\tau}_{j+1,M}^{2,(N)})}
#' and \eqn{\hat{v}_{j,j-1}^{(N)} =\hat{v}_{j-1,j}^{(N)} = -\frac{1}{N_p} \hat{\tau}_{j,M}^{2,(N)}}. Then
#' \deqn{\bar{h}_M^{(N),\top} [\hat{V}_M^{(N)}]^{-1} \bar{h}_M^{(N)} \to \chi^2 (M-1). }
#'
#'
#' @references
#' \insertRef{geweke2005contemporary}{BayesianFactorZoo}
#'
#'
#' @return
#' The return of \code{convergence_test} is :
#' \itemize{
#'   \item \code{chisq_test}: p-values of the Separated Partial Means Test (using upper tails of \eqn{\chi^2} distributions).
#' }
#'
#' @import nse
#'
#' @export
#'
#' @examples
#'
#' ## Load the example data
#' data("BFactor_zoo_example")
#' HML <- BFactor_zoo_example$HML
#' lambda_ols <- BFactor_zoo_example$lambda_ols
#' R2.ols.true <- BFactor_zoo_example$R2.ols.true
#' sim_f <- BFactor_zoo_example$sim_f
#' sim_R <- BFactor_zoo_example$sim_R
#' uf <- BFactor_zoo_example$uf
#'
#' ## sim_f: simulated strong factor
#' ## uf: simulated useless factor
#' ## Run the continuous spike-and-slab:
#' psi_hat <- psi_to_priorSR(sim_R, cbind(sim_f,uf), priorSR=0.1)
#' shrinkage <- continuous_ss_sdf(cbind(sim_f,uf), sim_R, 5000, psi0=psi_hat)
#'
#' ## Test whether the posterior draws in the above output converge in 500 steps:
#' cong_test_gamma <- convergence_test(shrinkage$gamma_path, 500, 5)
#' cong_test_lambda <- convergence_test(shrinkage$lambda_path[,2:3], 500, 5)
#'
#' cat("Null hypothesis: gammas converge within 500 steps", "\n")
#' cat("p-values of the Separated Partial Means Test for MCMC:",
#'     cong_test_gamma, "\n")
#'
#' cat("Null hypothesis: lambdas converge within 500 steps", "\n")
#' cat("p-values of the Separated Partial Means Test for MCMC:",
#'     cong_test_lambda, "\n")
#'

convergence_test <- function(post_draws, ndraws_test, M = 5) {

  k <- dim(post_draws)[2]
  ndraws <- dim(post_draws)[1]
  if (ndraws < 2*M*ndraws_test) {
    return("Please ensure that the number of posterior draws is greater than 2*M*ndraws_test!")
  }

  chisq_test <- matrix(0, nrow=1, ncol=k)
  start_draws <- 1+seq(1,M*2,2)*ndraws_test
  end_draws <- seq(2,M*2,2)*ndraws_test

  for (j in 1:k) {
    mp_seq <- rep(NA, M)
    se_seq <- rep(NA, M)
    for (m in 1:M) {
      mp_seq[m] <- mean(post_draws[start_draws[m]:end_draws[m], j])
      se_seq[m] <- nse.nw(post_draws[start_draws[m]:end_draws[m], j], lag.prewhite = NULL)
    }

    m_diff <- matrix(rep(NA,M-1), ncol=1)
    for (m in 1:(M-1)) {
      m_diff[m] <- mp_seq[m+1] - mp_seq[m]
    }
    V_diff <- matrix(0, ncol=M-1, nrow=M-1)
    for (m in 1:(M-1)) {
      V_diff[m,m] <- se_seq[m]^2 + se_seq[m+1]^2
      if (m < (M-1)) {
        V_diff[m,m+1] <- -se_seq[m+1]^2
      }
      if (m > 1) {
        V_diff[m,m-1] <- -se_seq[m]^2
      }
    }
    #print(t(m_diff) %*% solve(V_diff) %*% m_diff)
    #print(pchisq(t(m_diff) %*% solve(V_diff) %*% m_diff, df = M-1))
    chisq_test[1,j] <- pchisq(t(m_diff) %*% solve(V_diff) %*% m_diff, df = M-1, lower.tail = F)

  }

  return(chisq_test)
}
