% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sampleOccurrences.R
\name{sampleOccurrences}
\alias{sampleOccurrences}
\title{Sample occurrences in a virtual species distribution}
\usage{
sampleOccurrences(x, n, type = "presence only", sampling.area = NULL,
  detection.probability = 1, correct.by.suitability = FALSE,
  error.probability = 0, bias = "no.bias", bias.strength = 50,
  bias.area = NULL, weights = NULL, sample.prevalence = NULL,
  plot = TRUE)
}
\arguments{
\item{x}{a \code{rasterLayer} object or the output list from 
\code{generateSpFromFun}, \code{generateSpFromPCA}, \code{generateRandomSp}, \code{convertToPA}
or  \code{limitDistribution}
The raster must contain values of 0 or 1 (or NA).}

\item{n}{an integer. The number of occurrence points to sample.}

\item{type}{\code{"presence only"} or \code{"presence-absence"}. The type of 
occurrence points to sample.}

\item{sampling.area}{a character string, a \code{polygon} or an \code{extent} object.
The area in which the sampling will take place. See details.}

\item{detection.probability}{a numeric value between 0 and 1, corresponding to the
probability of detection of the species. See details.}

\item{correct.by.suitability}{\code{TRUE} or \code{FALSE}. If \code{TRUE}, then
the probability of detection will be weighted by the suitability, such that 
cells with lower suitabilities will further decrease the chance that the species
is detected when sampled.}

\item{error.probability}{\code{TRUE} or \code{FALSE}. Only useful if 
\code{type = "presence-absence"}. Probability to attribute an erroneous presence
in cells where the species is absent.}

\item{bias}{\code{"no.bias"},  \code{"country"},  \code{"region"},
\code{"extent"},  \code{"polygon"} or \code{"manual"}. The method used to 
generate a sampling bias: see details.}

\item{bias.strength}{a positive numeric value. The strength of the bias to be applied
in \code{area} (as a multiplier). Above 1, \code{area} will be oversampled. Below 1, \code{area}
will be undersampled.}

\item{bias.area}{\code{NULL}, a character string, a \code{polygon} or an \code{extent} object.
The area in which the sampling will be biased: see details. If \code{NULL}
and \code{bias = "extent"}, then you will be asked to draw an
extent on the map.}

\item{weights}{\code{NULL} or a raster layer. Only used if \code{bias = "manual"}.
The raster of bias weights to be applied to the sampling of occurrences.
Higher weights mean a higher probability of sampling.}

\item{sample.prevalence}{\code{NULL} or a numeric value between 0 and 1. Only useful if 
\code{type = "presence-absence"}. Defines the sample prevalence, i.e. the proportion of presences
sampled. Note that the probabilities of detection and error are applied AFTER this parameter,
so the final sample prevalence may not different if you apply probabilities of detection and/or error}

\item{plot}{\code{TRUE} or \code{FALSE}. If \code{TRUE}, the sampled occurrence
points will be plotted.}
}
\value{
a \code{list} with 3 (unbiased sampling) to 4 (biased sampling) elements:
\itemize{
\item{\code{sample.points}: the data.frame containing the coordinates of 
samples, the real presence-absences (or presence-only) and the sampled presence-
absences}
\item{\code{detection.probability}: the chosen probability of detection of
the virtual species}
\item{\code{error.probability}: the chosen probability to assign presence
in cells where the species is absent}
\item{\code{bias}: if a bias was chosen, then the type of bias and the
associated \code{area} will be included.}
}
}
\description{
This function samples presences within a species distribution, either
randomly or with a sampling bias. The sampling bias can be defined manually
or with a set of pre-defined biases.
}
\details{
\bold{How the function works:}

The function randomly selects \code{n} cells in which samples occur. If a \code{bias}
is chosen, then the selection of these cells will be biased according to the type and
strength of bias chosen. If the sampling is of \code{type "presence only"}, then only
cells where the species is present will be chosen. If the sampling is of 
\code{type "presence-absence"}, then all non-NA cells can be chosen.

The function then samples the species inside the chosen cells. In cells 
where the species is present the species will always be sampled unless 
the parameter \code{detection.probability} is lower than 1. In that case the
species will be sampled with the associated probability of detection.

In cells where the species is absent (in case of a \code{"presence-absence"}
sampling), the function will always assign absence unless \code{error.probability} 
is greater than 1. In that case, the species can be found present with the 
associated probability of error. Note that this step happens AFTER the detection
step. Hence, in cells where the species is present but not detected, it can
still be sampled due to a sampling error.

\bold{How to restrict the sampling area:}

Use the argument \code{sampling.area}:
\itemize{
\item{Provide the name (s) (or a combination of names) of country(ies), region(s) or continent(s).
Examples:
\itemize{
\item{\code{sampling.area = "Africa"}}
\item{\code{sampling.area = c("Africa", "North America", "France")}}
}}
\item{Provide a polygon (\code{SpatialPolygons} or \code{SpatialPolygonsDataFrame} 
of package \code{sp})}
\item{Provide an \code{extent} object}
}

\bold{How the sampling bias works:}

The argument \code{bias.strength} indicates the strength of the bias.
For example, a value of 50 will result in 50 times more samples within the
 \code{bias.area} than outside.
Conversely, a value of 0.5 will result in half less samples within the
\code{bias.area} than outside.

\bold{How to choose where the sampling is biased:}

You can choose to bias the sampling in:
\enumerate{
\item{a particular country, region or continent (assuming your raster has
the WGS84 projection): 

Set the argument
\code{bias} to \code{"country"}, \code{"region"} or
\code{"continent"}, and provide the name(s) of the associated countries,
regions or continents to \code{bias.area} (see examples). 

List of possible \code{bias.area} names:
\itemize{
\item{Countries: type \code{levels(getMap()@data$SOVEREIGNT)} in the console}
\item{Regions: "Africa", "Antarctica", "Asia", "Australia", "Europe", 
"North America", "South America"}
\item{Continents: "Africa", "Antarctica", "Australia", "Eurasia", 
"North America", "South America"}}
}
\item{a polygon:

Set \code{bias} to \code{"polygon"}, and provide your
polygon to \code{area}.
}
\item{an extent object:

Set \code{bias} to \code{"extent"}, and either provide your
extent object to \code{bias.area}, or leave it \code{NULL} to draw an extent on
the map.}
} 

Otherwise you can define manually your sampling bias, \emph{e.g.} to create
biases along roads. In that case you have to provide to \code{weights} a raster layer in which
each cell contains the probability to be sampled.
}
\examples{
# Create an example stack with six environmental variables
a <- matrix(rep(dnorm(1:100, 50, sd = 25)), 
            nrow = 100, ncol = 100, byrow = TRUE)
env <- stack(raster(a * dnorm(1:100, 50, sd = 25)),
             raster(a * 1:100),
             raster(a * logisticFun(1:100, alpha = 10, beta = 70)),
             raster(t(a)),
             raster(exp(a)),
             raster(log(a)))
names(env) <- paste("Var", 1:6, sep = "")   

# More than 6 variables: by default a PCA approach will be used
sp <- generateRandomSp(env, niche.breadth = "wide")

# Sampling of 25 presences
sampleOccurrences(sp, n = 25)

# Sampling of 30 presences and absebces
sampleOccurrences(sp, n = 30, type = "presence-absence")

# Reducing of the probability of detection
sampleOccurrences(sp, n = 30, type = "presence-absence", 
                  detection.probability = 0.5)
                  
# Further reducing in relation to environmental suitability
sampleOccurrences(sp, n = 30, type = "presence-absence", 
                  detection.probability = 0.5,
                  correct.by.suitability = TRUE)
                  
# Creating sampling errors (far too much)
sampleOccurrences(sp, n = 30, type = "presence-absence", 
                  error.probability = 0.5)
                  
# Introducing a sampling bias (oversampling)
biased.area <- extent(0.5, 0.7, 0.6, 0.8)
sampleOccurrences(sp, n = 50, type = "presence-absence", 
                  bias = "extent",
                  bias.area = biased.area)
# Showing the area in which the sampling is biased
plot(biased.area, add = TRUE)     

# Introducing a sampling bias (no sampling at all in the chosen area)
biased.area <- extent(0.5, 0.7, 0.6, 0.8)
sampleOccurrences(sp, n = 50, type = "presence-absence", 
                  bias = "extent",
                  bias.strength = 0,
                  bias.area = biased.area)
# Showing the area in which the sampling is biased
plot(biased.area, add = TRUE)             
}
\author{
Boris Leroy \email{leroy.boris@gmail.com}

with help from C. N. Meynard, C. Bellard & F. Courchamp
}
