% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tleaves.R
\name{.get_gtw}
\alias{.get_gtw}
\title{g_tw: total conductance to water vapour (m/s)}
\usage{
.get_gtw(T_leaf, pars, unitless)
}
\arguments{
\item{T_leaf}{Leaf temperature in Kelvin}

\item{pars}{Concatenated parameters (\code{leaf_par}, \code{enviro_par}, and \code{constants})}

\item{unitless}{Logical. Should function use parameters with \code{units}? The function is faster when FALSE, but input must be in correct units or else results will be incorrect without any warning.}
}
\value{
Value in m/s of class \code{units}
}
\description{
g_tw: total conductance to water vapour (m/s)
}
\details{
\bold{Total conductance to water vapor:} The total conductance to water vapor (\eqn{g_\mathrm{tw}}{g_tw}) is the sum of the parallel lower (abaxial) and upper (adaxial) conductances:

\deqn{g_\mathrm{tw} = g_\mathrm{w,lower} + g_\mathrm{w,upper}}{g_tw = gw_lower + gw_upper}

The conductance to water vapor on each surface is a function of parallel stomatal (\eqn{g_\mathrm{sw}}{g_sw}) and cuticular (\eqn{g_\mathrm{uw}}{g_uw}) conductances in series with the boundary layer conductance (\eqn{g_\mathrm{bw}}{g_bw}). The stomatal, cuticular, and boundary layer conductance on the lower surface are:

\deqn{g_\mathrm{sw,lower} = g_\mathrm{sw} (1 - sr) R (T_\mathrm{leaf} + T_\mathrm{air}) / 2}{gsw_lower = g_sw (1 - sr) R (T_leaf + T_air) / 2}
\deqn{g_\mathrm{uw,lower} = g_\mathrm{uw} / 2 R (T_\mathrm{leaf} + T_\mathrm{air}) / 2}{guw_lower = g_uw / 2 R (T_leaf + T_air) / 2}
\cr
See \code{\link{.get_gbw}} for details on calculating boundary layer conductance. The equations for the upper surface are:

\deqn{g_\mathrm{sw,upper} = g_\mathrm{sw} sr R (T_\mathrm{leaf} + T_\mathrm{air}) / 2}{gsw_upper = g_sw sr R (T_leaf + T_air) / 2}
\deqn{g_\mathrm{uw,upper} = g_\mathrm{uw} / 2 R (T_\mathrm{leaf} + T_\mathrm{air}) / 2}{guw_upper = g_uw / 2 R (T_leaf + T_air) / 2}
\cr
Note that the stomatal and cuticular conductances are given in units of (\eqn{\mu}mol H2O) / (m\eqn{^2} s Pa) (see \code{\link{make_leafpar}}) and converted to m/s using the ideal gas law. The total leaf stomtal (\eqn{g_\mathrm{sw}}{g_sw}) and cuticular (\eqn{g_\mathrm{uw}}{g_uw}) conductances are partitioned across lower and upper surfaces. The stomatal conductance on each surface depends on stomatal ratio (sr); the cuticular conductance is assumed identical on both surfaces. 

\tabular{lllll}{
\emph{Symbol} \tab \emph{R} \tab \emph{Description} \tab \emph{Units} \tab \emph{Default}\cr
\eqn{g_\mathrm{sw}}{g_sw} \tab \code{g_sw} \tab stomatal conductance to H2O \tab (\eqn{\mu}mol H2O) / (m\eqn{^2} s Pa) \tab 5\cr
\eqn{g_\mathrm{uw}}{g_uw} \tab \code{g_uw} \tab cuticular conductance to H2O \tab (\eqn{\mu}mol H2O) / (m\eqn{^2} s Pa) \tab 0.1\cr
\eqn{R} \tab \code{R} \tab ideal gas constant \tab J / (mol K) \tab 8.3144598\cr
\eqn{\mathrm{logit}(sr)}{logit(sr)} \tab \code{logit_sr} \tab stomatal ratio (logit transformed) \tab none \tab 0 = logit(0.5)\cr
\eqn{T_\mathrm{air}}{T_air} \tab \code{T_air} \tab air temperature \tab K \tab 298.15\cr
\eqn{T_\mathrm{leaf}}{T_leaf} \tab \code{T_leaf} \tab leaf temperature \tab K \tab input
}
}
\examples{

# Total conductance to water vapor

## Hypostomatous leaf; default parameters
leaf_par <- make_leafpar(replace = list(logit_sr = set_units(-Inf)))
enviro_par <- make_enviropar()
constants <- make_constants()
pars <- c(leaf_par, enviro_par, constants)
T_leaf <- set_units(300, K)

## Fixing boundary layer conductance rather than calculating
gbw_lower <- set_units(0.1, m / s)
gbw_upper <- set_units(0.1, m / s)

# Lower surface ----
## Note that pars$logit_sr is logit-transformed! Use stats::plogis() to convert to proportion.
gsw_lower <- set_units(pars$g_sw * (set_units(1) - stats::plogis(pars$logit_sr)) * pars$R * 
                         ((T_leaf + pars$T_air) / 2), "m / s")
guw_lower <- set_units(pars$g_uw * 0.5 * pars$R * ((T_leaf + pars$T_air) / 2), m / s)
gtw_lower <- 1 / (1 / (gsw_lower + guw_lower) + 1 / gbw_lower)

# Upper surface ----
gsw_upper <- set_units(pars$g_sw * stats::plogis(pars$logit_sr) * pars$R * 
                         ((T_leaf + pars$T_air) / 2), m / s)
guw_upper <- set_units(pars$g_uw * 0.5 * pars$R * ((T_leaf + pars$T_air) / 2), m / s)
gtw_upper <- 1 / (1 / (gsw_upper + guw_upper) + 1 / gbw_upper)

## Lower and upper surface are in parallel
g_tw <- gtw_lower + gtw_upper

}
