% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tbl-class.R
\name{id_tbl}
\alias{id_tbl}
\alias{is_id_tbl}
\alias{as_id_tbl}
\alias{ts_tbl}
\alias{is_ts_tbl}
\alias{as_ts_tbl}
\alias{as.data.table.id_tbl}
\alias{as.data.frame.id_tbl}
\alias{validate_tbl}
\title{Tabular ICU data classes}
\usage{
id_tbl(..., id_vars = 1L)

is_id_tbl(x)

as_id_tbl(x, id_vars = NULL, by_ref = FALSE)

ts_tbl(..., id_vars = 1L, index_var = NULL, interval = NULL)

is_ts_tbl(x)

as_ts_tbl(x, id_vars = NULL, index_var = NULL, interval = NULL, by_ref = FALSE)

\method{as.data.table}{id_tbl}(x, keep.rownames = FALSE, by_ref = FALSE, ...)

\method{as.data.frame}{id_tbl}(x, row.names = NULL, optional = FALSE, ...)

validate_tbl(x)
}
\arguments{
\item{...}{forwarded to \code{\link[data.table:data.table]{data.table::data.table()}} or generic consistency}

\item{id_vars}{Column name(s) to be used as \code{id} column(s)}

\item{x}{Object to query/operate on}

\item{by_ref}{Logical flag indicating whether to perform the operation by
reference}

\item{index_var}{Column name of the index column}

\item{interval}{Time series interval length specified as scalar-valued
\code{difftime} object}

\item{keep.rownames}{Default is \code{FALSE}. If \code{TRUE}, adds the input object's names as a separate column named \code{"rn"}. \code{keep.rownames = "id"} names the column \code{"id"} instead.}

\item{row.names}{\code{NULL} or a character vector giving the row
    names for the data frame.  Missing values are not allowed.}

\item{optional}{logical. If \code{TRUE}, setting row names and
    converting column names (to syntactic names: see
    \code{\link[base]{make.names}}) is optional.  Note that all of \R's
    \pkg{base} package \code{as.data.frame()} methods use
    \code{optional} only for column names treatment, basically with the
    meaning of \code{\link[base]{data.frame}(*, check.names = !optional)}.
    See also the \code{make.names} argument of the \code{matrix} method.}
}
\value{
Constructors \code{id_tbl()}/\code{ts_tbl()}, as well as coercion functions
\code{as_id_tbl()}/\code{as_ts_tbl()} return \code{id_tbl}/\code{ts_tbl} objects respectively,
while inheritance testers \code{is_id_tbl()}/\code{is_ts_tbl()} return logical flags
and \code{validate_tbl()} returns either \code{TRUE} or a string describing the
validation failure.
}
\description{
In order to simplify handling or tabular ICU data, \code{ricu} provides two
S3 classes, \code{id_tbl} and \code{ts_tbl}. The two classes essentially
consist of a \code{data.table} object, alongside some meta data and S3 dispatch
is used to enable more natural behavior for some data manipulation tasks.
For example, when merging two tables, a default for the \code{by} argument can
be chosen more sensibly if columns representing patient ID and timestamp
information can be identified.
}
\details{
The two classes are designed for two often encountered data scenarios:
\itemize{
\item \code{id_tbl} objects can be used to represent static (with respect to
relevant time scales) patient data such as patient age and such an object
is simply a \code{data.table} combined with a non-zero length character vector
valued attribute marking the columns tracking patient ID information
(\code{\link[=id_vars]{id_vars}}). All further columns are considered as
\link[=data_vars]{data_vars}.
\item \code{ts_tbl} objects are used for grouped time series data. A \code{data.table}
object again is augmented by attributes, including a non-zero length
character vector identifying patient ID columns (\link[=id_vars]{id_vars}),
a string, tracking the column holding time-stamps
(\link[=index_var]{index_var}) and a scalar \code{difftime} object determining
the time-series step size \link[=interval]{interval}. Again, all further
columns are treated as \link[=data_vars]{data_vars}.
}

Owing to the nested structure of required meta data, \code{ts_tbl} inherits from
\code{id_tbl}. Furthermore, both classes inherit from \code{data.table}. As such,
\code{data.table} \link[data.table:assign]{reference semantics} are available for
some operations, indicated by presence of a \code{by_ref} argument. At default,
value, \code{by_ref} is set to \code{FALSE} as this is in line with base R behavior
at the cost of potentially incurring unnecessary data copies. Some care has
to be taken when passing \code{by_ref = TRUE} and enabling by reference
operations as this can have side effects (see examples).

For instantiating \code{ts_tbl} objects, both \code{index_var} and \code{interval} can be
automatically determined if not specified. For the index column, the only
requirement is that a single \code{\link[base:difftime]{difftime}} column is
present, while for the time step, the minimal difference between two
consecutive observations is chosen (and all differences are therefore
required to be multiples of the minimum difference).

Upon instantiation, the data might be rearranged: columns are reordered
such that ID columns are moved to the front, followed by the index column
and a \code{\link[data.table:setkey]{data.table::key()}} is set on meta columns, causing rows to be sorted
accordingly. Moving meta columns to the front is done for reasons of
convenience for printing, while setting a key on meta columns is done to
improve efficiency of subsequent transformations such as merging or grouped
operations. Furthermore, \code{NA} values in either ID or index columns are not
allowed and therefore corresponding rows are silently removed.

Coercion between \code{id_tbl} and \code{ts_tbl} by default keeps intersecting
attributes fixed and new attributes are by default inferred as for class
instantiation. Each class comes with a class-specific implementation of the
S3 generic function \code{validate_tbl()} which returns \code{TRUE} if the object is
considered valid or a string outlining the type of validation failure that
was encountered. Validity requires
\enumerate{
\item inheriting from \code{data.table} and unique column names
\item for \code{id_tbl} that all columns specified by the non-zero length character
vector holding onto the \code{id_vars} specification are available
\item for \code{ts_tbl} that the string-valued \code{index_var} column is available and
does not intersect with \code{id_vars} and that the index column obeys the
specified interval.
}

Finally, inheritance can be checked by calling \code{is_id_tbl()} and
\code{is_ts_tbl()}. Note that due to \code{ts_tbl} inheriting from \code{id_tbl},
\code{is_id_tbl()} returns \code{TRUE} for both \code{id_tbl} and \code{ts_tbl} objects, while
\code{is_ts_tbl()} only returns \code{TRUE} for \code{ts_tbl} objects.
}
\section{Relationship to \code{data.table}}{

Both \code{id_tbl} and \code{ts_tbl} inherit from \code{data.table} and as such, functions
intended for use with \code{data.table} objects can be applied to \code{id_tbl} and
\code{ts_tbl} as well. But there are some caveats: Many functions introduced by
\code{data.table} are not S3 generic and therefore they would have to be masked
in order to retain control over how they operate on objects inheriting form
\code{data.table}. Take for example the function \code{\link[data.table:setattr]{data.table::setnames()}}, which
changes column names by reference. Using this function, the name of an
index column of an \code{id_tbl} object can me changed without updating the
attribute marking the column as such and thusly leaving the object in an
inconsistent state. Instead of masking the function \code{setnames()}, an
alternative is provided as \code{\link[=rename_cols]{rename_cols()}}. In places where it is possible
to seamlessly insert the appropriate function (such as
\code{\link[base:names]{base::names<-()}} or \code{\link[base:colnames]{base::colnames<-()}}) and the responsibility for not
using \code{\link[data.table:setattr]{data.table::setnames()}} in a way that breaks the \code{id_tbl} object is
left to the user.

Owing to \code{data.table} heritage, one of the functions that is often called
on \code{id_tbl} and \code{ts_tbl} objects is base S3 generic [base::\code{[}()]. As this
function is capable of modifying the object in a way that makes it
incompatible with attached meta data, an attempt is made at preserving as
much as possible and if all fails, a \code{data.table} object is returned
instead of an object inheriting form \code{id_tbl}. If for example the index
column is removed (or modified in a way that makes it incompatible with the
interval specification) from a \code{ts_tbl}, an \code{id_tbl} is returned. If
however the ID column is removed the only sensible thing to return is a
\code{data.table} (see examples).
}

\examples{
tbl <- id_tbl(a = 1:10, b = rnorm(10))
is_id_tbl(tbl)
is_ts_tbl(tbl)

dat <- data.frame(a = 1:10, b = hours(1:10), c = rnorm(10))
tbl <- as_ts_tbl(dat, "a")
is_id_tbl(tbl)
is_ts_tbl(tbl)

tmp <- as_id_tbl(tbl)
is_ts_tbl(tbl)
is_ts_tbl(tmp)

tmp <- as_id_tbl(tbl, by_ref = TRUE)
is_ts_tbl(tbl)
is_ts_tbl(tmp)

tbl <- id_tbl(a = 1:10, b = rnorm(10))
names(tbl) <- c("c", "b")
tbl

tbl <- id_tbl(a = 1:10, b = rnorm(10))
validate_tbl(data.table::setnames(tbl, c("c", "b")))

tbl <- id_tbl(a = 1:10, b = rnorm(10))
validate_tbl(rename_cols(tbl, c("c", "b")))

tbl <- ts_tbl(a = rep(1:2, each = 5), b = hours(rep(1:5, 2)), c = rnorm(10))
tbl[, c("a", "c"), with = FALSE]
tbl[, c("b", "c"), with = FALSE]
tbl[, list(a, b = as.double(b), c)]

}
