% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/accessibility.R
\name{accessibility}
\alias{accessibility}
\title{Calculate access to opportunities}
\usage{
accessibility(
  r5r_core,
  origins,
  destinations,
  opportunities_colname = "opportunities",
  mode = "WALK",
  mode_egress = "WALK",
  departure_datetime = Sys.time(),
  time_window = 1L,
  percentiles = 50L,
  decay_function = "step",
  cutoffs = 30L,
  decay_value = 1,
  max_walk_dist = Inf,
  max_bike_dist = Inf,
  max_trip_duration = 120L,
  walk_speed = 3.6,
  bike_speed = 12,
  max_rides = 3,
  max_lts = 2,
  n_threads = Inf,
  verbose = TRUE
)
}
\arguments{
\item{r5r_core}{a rJava object to connect with R5 routing engine}

\item{origins, destinations}{a spatial sf POINT object, or a data.frame
containing the columns 'id', 'lon', 'lat'}

\item{opportunities_colname}{string. The column name in the \code{destinations}
input that tells the number of opportunities in each location.
Defaults to "opportunities".}

\item{mode}{string. Transport modes allowed for the trips. Defaults to
"WALK". See details for other options.}

\item{mode_egress}{string. Transport mode used after egress from public
transport. It can be either 'WALK', 'BICYCLE', or 'CAR'.
Defaults to "WALK".}

\item{departure_datetime}{POSIXct object. If working with public transport
networks, please check \code{calendar.txt} within
the GTFS file for valid dates.}

\item{time_window}{numeric. Time window in minutes for which r5r will
calculate travel times departing each minute. When using
frequency-based GTFS files, 5 Monte Carlo simulations will
be run for each minute in the time window. See details for
further information.}

\item{percentiles}{numeric vector. Defaults to '50', returning the accessibility
value for the median travel time computed for a given
time_window. If a numeric vector is passed, for example
c(25, 50, 75), the function will return accessibility
estimates for each percentile, by travel time cutoff. Only
the first 5 cut points of the percentiles are considered.
For more details, see R5 documentation at
'https://docs.conveyal.com/analysis/methodology#accounting-for-variability'}

\item{decay_function}{string. Choice of one of the following decay functions:
'step', 'exponential', 'fixed_exponential', 'linear',
and 'logistic'. Defaults to 'step', which yields
cumulative opportunities accessibility metrics.
More info in \code{details}.}

\item{cutoffs}{numeric. Cutoff times in minutes for calculating cumulative
opportunities accessibility when using the 'step decay function'.
This parameter has different effects for each of the other decay
functions: it indicates the 'median' (or inflection point) of
the decay curves in the 'logistic' and 'linear' functions, and
the 'half-life' in the 'exponential' function. It has no effect
when using the 'fixed exponential' function.}

\item{decay_value}{numeric. Extra parameter to be passed to the selected
\code{decay_function}.}

\item{max_walk_dist}{numeric. Maximum walking distance (in meters) for the
whole trip. Defaults to no restrictions on walking, as
long as \code{max_trip_duration} is respected.}

\item{max_bike_dist}{numeric. Maximum cycling distance (in meters) for the
whole trip. Defaults to no restrictions on cycling, as
long as \code{max_trip_duration} is respected.}

\item{max_trip_duration}{numeric. Maximum trip duration in minutes. Defaults
to 120 minutes (2 hours).}

\item{walk_speed}{numeric. Average walk speed in km/h. Defaults to 3.6 km/h.}

\item{bike_speed}{numeric. Average cycling speed in km/h. Defaults to 12 km/h.}

\item{max_rides}{numeric. The max number of public transport rides allowed in
the same trip. Defaults to 3.}

\item{max_lts}{numeric (between 1 and 4). The maximum level of traffic stress
that cyclists will tolerate. A value of 1 means cyclists will
only travel through the quietest streets, while a value of 4
indicates cyclists can travel through any road. Defaults to 2.
See details for more information.}

\item{n_threads}{numeric. The number of threads to use in parallel computing.
Defaults to use all available threads (Inf).}

\item{verbose}{logical. \code{TRUE} to show detailed output messages (the default).
If verbose is set to \code{FALSE}, r5r prints a progress counter and
eventual \code{ERROR} messages. Setting \code{verbose} to  \code{FALSE} imposes
a small penalty for computation efficiency.}
}
\value{
A data.table with accessibility estimates for all origin points, by
a given transport mode, and per travel time cutoff and percentile.
}
\description{
Fast computation of access to opportunities given a selected
decay function. See \code{details} for the available decay functions.
}
\section{Decay functions:}{
R5 allows for multiple decay functions. More info at \url{https://docs.conveyal.com/learn-more/decay-functions}
The options include:
\subsection{Step \code{step} (cumulative opportunities)}{

A binary decay function used to calculate cumulative opportunities metrics.
}

\subsection{Logistic CDF \code{logistic}}{

This is the logistic function, i.e. the cumulative distribution function of
the logistic distribution, expressed such that its parameters are the median
(inflection point) and standard deviation. This function applies a sigmoid
rolloff that has a convenient relationship to discrete choice theory. Its
parameters can be set to reflect a whole population's tolerance for making
trips with different travel times. The function's value represents the
probability that a randomly chosen member of the population would accept
making a trip, given its duration. Opportunities are then weighted by how
likely it is a person would consider them "reachable".
\subsection{calibration}{

The median parameter is controlled by the \code{cutoff} parameter, leaving only
the standard deviation to configure through the \code{decay_value} parameter.
}

}

\subsection{Fixed Exponential \code{fixed_exponential}}{

This function is of the form e-Lt where L is a single fixed decay constant
in the range (0, 1). It is constrained to be positive to ensure weights
decrease (rather than grow) with increasing travel time.
\subsection{calibration}{

This function is controlled exclusively by the L constant, given by the
\code{decay_value} parameter. Values provided in \code{cutoffs} are ignored.
}

}

\subsection{Half-life Exponential Decay \code{exponential}}{

This is similar to the fixed-exponential option above, but in this case the
decay parameter is inferred from the \code{cutoffs} parameter values, which is
treated as the half-life of the decay.
}

\subsection{Linear \code{linear}}{

This is a simple, vaguely sigmoid option, which may be useful when you have
a sense of a maximum travel time that would be tolerated by any traveler,
and a minimum time below which all travel is perceived to be equally easy.
\subsection{calibration}{

The transition region is transposable and symmetric around the \code{cutoffs}
parameter values, taking \code{decay_value} minutes to taper down from one to zero.
}

}
}

\section{Transport modes:}{
R5 allows for multiple combinations of transport modes. The options include:
\subsection{Transit modes}{

TRAM, SUBWAY, RAIL, BUS, FERRY, CABLE_CAR, GONDOLA, FUNICULAR. The option
'TRANSIT' automatically considers all public transport modes available.
}

\subsection{Non transit modes}{

WALK, BICYCLE, CAR, BICYCLE_RENT, CAR_PARK
}
}

\section{max_lts, Maximum Level of Traffic Stress:}{
When cycling is enabled in R5, setting \code{max_lts} will allow cycling only on
streets with a given level of danger/stress. Setting \code{max_lts} to 1, for example,
will allow cycling only on separated bicycle infrastructure or low-traffic
streets; routing will revert to walking when traversing any links with LTS
exceeding 1. Setting \code{max_lts} to 3 will allow cycling on links with LTS 1, 2,
or 3.

The default methodology for assigning LTS values to network edges is based on
commonly tagged attributes of OSM ways. See more info about LTS at
\url{https://docs.conveyal.com/learn-more/traffic-stress}. In summary:
\itemize{
\item \strong{LTS 1}: Tolerable for children. This includes low-speed, low-volume streets,
as well as those with separated bicycle facilities (such as parking-protected
lanes or cycle tracks).
\item \strong{LTS 2}: Tolerable for the mainstream adult population. This includes streets
where cyclists have dedicated lanes and only have to interact with traffic at
formal crossing.
\item \strong{LTS 3}: Tolerable for “enthused and confident” cyclists. This includes streets
which may involve close proximity to moderate- or high-speed vehicular traffic.
\item \strong{LTS 4}: Tolerable for only “strong and fearless” cyclists. This includes streets
where cyclists are required to mix with moderate- to high-speed vehicular traffic.
}
}

\section{Routing algorithm:}{
The \code{accessibility()} function uses an R5-specific extension to the RAPTOR
routing algorithm (see Conway et al., 2017). This RAPTOR extension uses a
systematic sample of one departure per minute over the time window set by the
user in the 'time_window' parameter. A detailed description of base RAPTOR
can be found in Delling et al (2015).
\itemize{
\item Conway, M. W., Byrd, A., & van der Linden, M. (2017). Evidence-based transit
and land use sketch planning using interactive accessibility methods on
combined schedule and headway-based networks. Transportation Research Record,
2653(1), 45-53.
\item Delling, D., Pajor, T., & Werneck, R. F. (2015). Round-based public transit
routing. Transportation Science, 49(3), 591-604.
}
}

\examples{
if (interactive()) {
library(r5r)

# build transport network
data_path <- system.file("extdata/poa", package = "r5r")
r5r_core <- setup_r5(data_path = data_path)

# load origin/destination points
points <- read.csv(file.path(data_path, "poa_hexgrid.csv"))

  access <- accessibility(r5r_core,
                          origins = points,
                          destinations = points,
                          opportunities_colname = "schools",
                          mode = "WALK",
                          cutoffs = c(25, 30),
                          max_trip_duration = 30,
                          verbose = FALSE)

stop_r5(r5r_core)

}
}
\seealso{
Other routing: 
\code{\link{detailed_itineraries}()},
\code{\link{travel_time_matrix}()}
}
\concept{routing}
