#' Read a CSV file exported from Qualtrics
#'
#' Reads comma separated CSV files generated by Qualtrics
#' software. The second line containing the variable labels is imported.
#' Repetitive introductions to matrix questions are automatically removed.
#' Variable labels are stored as attributes.
#' @param ... All arguments for \code{\link[qualtRics]{read_survey}}
#'
#' @export
readSurvey <- function(...) {
  read_survey(...)
}


#' Read a CSV file exported from Qualtrics
#'
#' Reads comma separated CSV files generated by Qualtrics
#' software. The second line containing the variable labels is imported.
#' Repetitive introductions to matrix questions are automatically removed.
#' Variable labels are stored as attributes.
#'
#' @param file_name String. A CSV data file.
#' @param import_id Logical. If \code{TRUE}, use Qualtrics import IDs instead of
#' question IDs as column names. Defaults to \code{FALSE}.
#' @param strip_html Logical. If \code{TRUE}, then remove HTML tags. Defaults
#' to \code{TRUE}.
#' @param legacy Logical. If \code{TRUE}, then import "legacy" format CSV files
#' (as of 2017). Defaults to \code{FALSE}.
#'
#' @importFrom sjlabelled set_label
#' @importFrom jsonlite fromJSON
#' @importFrom stringr str_match
#' @importFrom readr read_csv
#' @return A data frame. Variable labels are stored as attributes. They are not
#' printed on the console but are visibile in the RStudio viewer.
#' @export
#' @examples
#' \dontrun{
#' # Generic use of read_survey()
#' df <- read_survey("<YOUR-PATH-TO-CSV-FILE>")
#' }
#' # Example using current data format
#' file <- system.file("extdata", "sample.csv", package = "qualtRics")
#' df <- read_survey(file)
#'
#' # Example using legacy data format
#' file <- system.file("extdata", "sample_legacy.csv", package = "qualtRics")
#' df <- read_survey(file, legacy = TRUE)
#'
#'
read_survey <- function(file_name,
                        strip_html = TRUE,
                        import_id = FALSE,
                        legacy = FALSE) {

  # START UP: CHECK ARGUMENTS PASSED BY USER ----

  if (import_id & legacy) {
    stop("Import IDs as column names are not supported for legacy CSVs.\nSet import_id = FALSE.",
         call. = FALSE)
  }

  # check if file exists
  assert_surveyFile_exists(file_name)
  # skip 2 rows if legacy format, else 3 when loading the data
  skipNr <- ifelse(legacy, 2, 3)

  # READ DATA ----

  # import data including variable names (row 1) and variable labels (row 2)
  rawdata <- suppressMessages(readr::read_csv(
    file = file_name,
    col_names = FALSE,
    col_types = readr::cols(.default = readr::col_character()),
    skip = skipNr,
    na = c("")
  ))
  # Need contingency when 0 rows
  assertthat::assert_that(nrow(rawdata) > 0,
                          msg = "The survey you are trying to import has no responses."
  ) # nolint
  # Load headers
  header <- suppressWarnings(suppressMessages(readr::read_csv(
    file = file_name,
    col_names = TRUE,
    col_types = readr::cols(.default = readr::col_character()),
    n_max = 1
  )))

  # MANIPULATE DATA ----

  # make them data.frame's, else the factor conversion
  # in `infer_data_types` crashes
  # rawdata <- as.data.frame(rawdata)
  # header <- as.data.frame(header)
  # Add names
  names(rawdata) <- names(header)

  if (import_id) {
    new_ids <- suppressMessages(readr::read_csv(
      file = file_name,
      col_names = FALSE,
      col_types = readr::cols(.default = readr::col_character()),
      skip = skipNr - 1,
      n_max = 1
    ))

    names(rawdata) <- gsub("^\\{'ImportId': '(.*)'\\}$", "\\1",
                           unlist(new_ids))
  }

  # If Qualtrics adds an empty column at the end, remove it
  if (grepl(",$", readLines(file_name, n = 1))) {
    header <- header[, 1:(ncol(header) - 1)]
    rawdata <- rawdata[, 1:(ncol(rawdata) - 1)]
  }
  # extract second row, remove it from df
  secondrow <- unlist(header)
  row.names(rawdata) <- NULL

  # Clean variable labels
  if (strip_html) {
    # weird regex to strip HTML tags, leaving only content
    # https://www.r-bloggers.com/htmltotext-extracting-text-from-html-via-xpath/ # nolint
    pattern <- "</?\\w+((\\s+\\w+(\\s*=\\s*(?:\".*?\"|'.*?'|[^'\">\\s]+))?)+\\s*|\\s*)/?>" # nolint
    secondrow <- gsub(pattern, "\\4", secondrow)
  }

  # Scale Question with subquestion:
  # If it matches one of ".?!" followed by "-", take subsequent part
  subquestions <- stringr::str_match(secondrow, ".*[:punct:]\\s*-(.*)")[, 2]

  # Else if subquestion returns NA, use whole string
  subquestions[is.na(subquestions)] <- unlist(secondrow[is.na(subquestions)])

  # Remaining NAs default to 'empty string'
  subquestions[is.na(subquestions)] <- ""

  rawdata <- readr::type_convert(rawdata)

  # Add labels to data
  rawdata <- sjlabelled::set_label(rawdata, unlist(subquestions))

  # RETURN ----

  return(rawdata)
}
